// -*- c++ -*-
#ifndef HUGO_XY_H
#define HUGO_XY_H

#include <iostream>

namespace hugo {

///\file
///\brief A simple two dimensional vector and a bounding box implementation 
///
/// The class \ref hugo::xy "xy" implements
///a two dimensional vector with the usual
/// operations.
///
/// The class \ref hugo::BoundingBox "BoundingBox" can be used to determine
/// the rectangular bounding box a set of \ref hugo::xy "xy"'s.


/** \brief
2 dimensional vector (plainvector) implementation

*/
  template<typename T>
    class xy {

    public:

      T x,y;     
      
      ///Default constructor: both coordinates become 0
      xy() : x(0), y(0) {}

      ///Constructing the instance from coordinates
      xy(T a, T b) : x(a), y(a) { }


      ///Gives back the square of the norm of the vector
      T normSquare(){
	return x*x+y*y;
      };
  
      ///Increments the left hand side by u
      xy<T>& operator +=(const xy<T>& u){
	x += u.x;
	y += u.y;
	return *this;
      };
  
      ///Decrements the left hand side by u
      xy<T>& operator -=(const xy<T>& u){
	x -= u.x;
	y -= u.y;
	return *this;
      };

      ///Multiplying the left hand side with a scalar
      xy<T>& operator *=(const T &u){
	x *= u;
	y *= u;
	return *this;
      };

      ///Dividing the left hand side by a scalar
      xy<T>& operator /=(const T &u){
	x /= u;
	y /= u;
	return *this;
      };
  
      ///Returns the scalar product of two vectors
      T operator *(const xy<T>& u){
	return x*u.x+y*u.y;
      };
  
      ///Returns the sum of two vectors
      xy<T> operator+(const xy<T> &u) const {
	xy<T> b=*this;
	return b+=u;
      };

      ///Returns the difference of two vectors
      xy<T> operator-(const xy<T> &u) const {
	xy<T> b=*this;
	return b-=u;
      };

      ///Returns a vector multiplied by a scalar
      xy<T> operator*(const T &u) const {
	xy<T> b=*this;
	return b*=u;
      };

      ///Returns a vector divided by a scalar
      xy<T> operator/(const T &u) const {
	xy<T> b=*this;
	return b/=u;
      };

      ///Testing equality
      bool operator==(const xy<T> &u){
	return (x==u.x) && (y==u.y);
      };

      ///Testing inequality
      bool operator!=(xy u){
	return  (x!=u.x) || (y!=u.y);
      };

    };

  ///Reading a plainvector from a stream
  template<typename T>
  inline
  std::istream& operator>>(std::istream &is, xy<T> &z)
  {

    is >> z.x >> z.y;
    return is;
  }

  ///Outputting a plainvector to a stream
  template<typename T>
  inline
  std::ostream& operator<<(std::ostream &os, xy<T> z)
  {
    os << "(" << z.x << ", " << z.y << ")";
    return os;
  }


  /** \brief
     Implementation of a bounding box of plainvectors.
     
  */
  template<typename T>
    class BoundingBox {
      xy<T> bottom_left, top_right;
      bool _empty;
    public:
      
      ///Default constructor: an empty bounding box
      BoundingBox() { _empty = true; }

      ///Constructing the instance from one point
      BoundingBox(xy<T> a) { bottom_left=top_right=a; _empty = false; }

      ///Is there any point added
      bool empty() const {
	return _empty;
      }

      ///Gives back the bottom left corner (if the bounding box is empty, then the return value is not defined) 
      xy<T> bottomLeft() const {
	return bottom_left;
      };

      ///Gives back the top right corner (if the bounding box is empty, then the return value is not defined) 
      xy<T> topRight() const {
	return top_right;
      };

      ///Checks whether a point is inside a bounding box
      bool inside(const xy<T>& u){
	if (_empty)
	  return false;
	else{
	  return ((u.x-bottom_left.x)*(top_right.x-u.x) >= 0 &&
		  (u.y-bottom_left.y)*(top_right.y-u.y) >= 0 );
	}
      }
  
      ///Increments a bounding box with a point
      BoundingBox& operator +=(const xy<T>& u){
	if (_empty){
	  bottom_left=top_right=u;
	  _empty = false;
	}
	else{
	  if (bottom_left.x > u.x) bottom_left.x = u.x;
	  if (bottom_left.y > u.y) bottom_left.y = u.y;
	  if (top_right.x < u.x) top_right.x = u.x;
	  if (top_right.y < u.y) top_right.y = u.y;
	}
	return *this;
      };
  
      ///Sums a bounding box and a point
      BoundingBox operator +(const xy<T>& u){
	BoundingBox b = *this;
	return b += u;
      };

      ///Increments a bounding box with an other bounding box
      BoundingBox& operator +=(const BoundingBox &u){
	if ( !u.empty() ){
	  *this += u.bottomLeft();
	  *this += u.topRight();
	}
	return *this;
      };
  
      ///Sums two bounding boxes
      BoundingBox operator +(const BoundingBox& u){
	BoundingBox b = *this;
	return b += u;
      };

    };//class Boundingbox




} //namespace hugo

#endif //HUGO_XY_H
