/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef ALGOWIN_H
#define ALGOWIN_H

#include <all_include.h>
#include <libgnomecanvasmm.h>
#include <libgnomecanvasmm/polygon.h>

class AlgoBox;
class MapStorage;

///Algorithm identifiers.
enum {GENERAL, KRUSKAL, ALGO_NUM}; // algorithm IDs;

///Window displaying digraphical interface for different algorithms.

///This class displays a digraphical interface to set up
///and run different algorithms. Different algorithms need
///different inputs, running methods, etc. Therefore
///class \ref AlgoWin is only a holder of a base class, the so
///called AlgoBox. \ref AlgoBox is the ancestor of other
///classes. These child classes realize interfaces of different
///algorithms, but as their common ancestor is \ref AlgoBox
///the interface of them is the same. \ref AlgoWin communicates
///with these classes through this common interface. But it the
///real object to be placed in \ref AlgoWin depends on the algorithm
///which the \ref AlgoWin actually has to display. It gets the
///id of algorithm to display at initialization, and therefore it is
///able to place in itself the requested child of \ref AlgoBox
/// visualizing the appropriate algorithm.
class AlgoWin : public Gtk::Window
{
private:
  ///Algorithm specific part of \ref AlgoWin
  AlgoBox * ab;

  ///Run button.

  ///If pressed, algorithm should run.
  ///That is why common ancestor of different
  ///algorithm realizer classes have to be. In case of
  ///pressing run button a common method can be called.
  Gtk::Button * runbutton;

  ///Close button. If pressed, \ref AlgoWin should close.
  Gtk::Button * closebutton;

protected:
  ///Signal emitted upon close of window

  ///It is necessary, because \ref MainWin have to
  ///score the opened \ref AlgoWin s, to be able to communicate
  ///with them: let them know about changement in tabs, maps, etc.
  ///If \ref AlgoWin is closed, \ref MainWin has to deregistrate it.
  ///Therefore signal contains address of emitter \ref AlgoWin.
  sigc::signal<void, AlgoWin *> signal_closed;

  ///Signal indicating that informatino on certain maplist is required.

  ///It is just a forwarded signal from \ref AlgoBox, benefit of common ancestor
  ///algorithm class. User can select the digraph (the holder \ref NoteBookTab) on
  ///which the algorithm should run. But different digraphs (\ref NoteBookTab) have
  ///different maps. If selected tab changes this signal is emitted by \ref AlgoBox,
  ///caught and reemitted by \ref AlgoWin.
  ///
  ///Signal contains the address of \ref AlgoWin to let \ref MainWin know
  ///where should the information needed forwarded, and the name of
  ///\ref NoteBookTab, of which maps are inquired.
  sigc::signal<void, AlgoWin *, std::string> signal_maplist_need;

  ///Signal that indicates that a \ref NewMapWin should be popped up.

  ///This is a forwarded signal. If \ref AlgoBox emits a signal
  ///to let a \ref NewMapWin pop up, |ref AlgoWin catch and reemit it.
  ///
  ///Signal contains the name of \ref NoteBookTab, in which the new map
  ///should be created and a boolean that indicates whether an arc or a
  ///nodemap should be created.
  sigc::signal<void, std::string, bool> signal_newmapwin_need;

public:
  ///Close window if escape key is pressed.
  bool closeIfEscapeIsPressed(GdkEventKey* e);

  ///Returns \ref signal_closed to be bindable somewhere.
  sigc::signal<void, AlgoWin *> signal_closing();

  ///Returns \ref signal_maplist_need to be bindable somewhere.
  sigc::signal<void, AlgoWin *, std::string> signal_maplist_needed();

  ///Returns \ref signal_newmapwin_need to be bindable somewhere.
  sigc::signal<void, std::string, bool> signal_newmapwin_needed(){return signal_newmapwin_need;};

  ///Forwards signal emitted by \ref AlgoBox, in which it indicates changement in selection of tabs.
  void emit_tab_change(std::string);

  ///Forwards signal emitted by \ref AlgoBox, in which it indicates need for \ref NewMapWin.
  void emit_new_map_signal(std::string tabname, bool itisarc){signal_newmapwin_need.emit(tabname, itisarc);};

  ///Constructor

  ///It builds the window according to the information provided
  ///by the creator. It needs the identifier of the algorithm
  ///to visualize, and a list of name of \ref NoteBookTab s that can
  ///be found in \ref MainWin.
  ///\param algoid identifier of algorithm to show
  ///\param tablist list of tabs in \ref MainWin
  AlgoWin(int algoid, std::vector<std::string> tablist);

  ///Forwards list of \ref NoteBookTabs toward \ref AlgoBox

  ///In case of changement in tabs in \ref MainWin
  ///\ref MainWin automatically updates tablist in
  ///\ref AlgoWin s.
  void update_tablist(std::vector<std::string> tabnames);

  ///Forwards list of requested maps toward \ref AlgoBox

  ///Upon catching the signal in which \ref AlgoBox requests
  ///list of maps \ref MainWin responds
  ///through this function.
  void update_maplist(MapStorage *);

  ///Called when window is closing.

  ///\ref AlgoWin has to be deregistrated in \ref MainWin
  ///thereforeit emits signal \ref signal_closed.
  void on_hide();
};
#endif //ALGOWIN_H
