/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef MAPSTORAGE_H
#define MAPSTORAGE_H

class Mapstorage;

#include <vector>
#include <map>
#include <string>
#include "all_include.h"
#include "xymap.h"
#include <libgnomecanvasmm.h>
#include "map_value.h"
#include "map_value_map.h"

///class MapStorage handles NodeMaps and ArcMaps.

///Class MapStorage is responsible for storing
///NodeMaps and ArcMaps that can be shown later
///on GUI. Therefore maps can be added to it,
///and datas over the added maps can be queried.
///The maps will be stored in an std::map,
///referenced with their names. Unfortunately at
///the moment it works only with double type maps
///
///\todo too many things are public!!
class MapStorage
{
private:
  std::string background_file_name;
  bool background_set;
  double background_scaling;
public:
  class Error : public std::exception
  {
    private:
      std::string message;
    public:
      Error(const std::string& msg) : message(msg) {}
      virtual const char* what() const throw()
      {
        return message.c_str();
      }
      ~Error() throw() {}
  };

  void setBackground(const std::string& file_name);
  const std::string& getBackgroundFilename();
  bool isBackgroundSet();
  double getBackgroundScaling();
  void setBackgroundScaling(double scaling);

  enum MapSaveDest { GUI_SECT, NESET_SECT, DONT_SAVE };
  enum GuiSectSaveDest { LGF_FILE, CONF_FILE };
  struct SpecMapSaveOpts
  {
    enum Dest { GUI_SECT, NESET_SECT };
    enum MapNum { ONE_MAP, TWO_MAPS };
  };

  typedef Digraph::NodeMap<double> NumericNodeMap;
  typedef Digraph::NodeMap<std::string> StringNodeMap;
  typedef Digraph::ArcMap<double> NumericArcMap;
  typedef Digraph::ArcMap<std::string> StringArcMap;
  typedef Digraph::NodeMap<int> NodeLabelMap;
  typedef Digraph::ArcMap<int> ArcLabelMap;
  typedef XYMap<Digraph::NodeMap<double> > NodeCoordMap;
  typedef XYMap<Digraph::ArcMap<double> > ArrowCoordMap;

  struct ArcMapData
  {
    /// where to save the map
    MapSaveDest save_dest;
    /// read-only or read-write
    bool writeable;
    /// default value
    MapValue default_value;
    virtual MapValue::Type type() = 0;
    virtual MapValue get(Arc e) = 0;
    virtual void set(Arc e, MapValue v) = 0;
    ArcMapData(MapValue def_val) :
      save_dest(GUI_SECT),
      writeable(true),
      default_value(def_val)
    {}
  };

  struct NumericArcMapData : public ArcMapData
  {
    NumericArcMap map;
    MapValue::Type type() { return MapValue::NUMERIC; }
    MapValue get(Arc e) { return MapValue(map[e]); }
    void set(Arc e, MapValue v) { map.set(e, static_cast<double>(v)); }
    NumericArcMapData(Digraph& g, double def_val) :
      ArcMapData(MapValue(def_val)),
      map(g, def_val)
    {}
  };

  struct StringArcMapData : public ArcMapData
  {
    StringArcMap map;
    MapValue::Type type() { return MapValue::STRING; }
    MapValue get(Arc e) { return MapValue(map[e]); }
    void set(Arc e, MapValue v) { map.set(e, static_cast<std::string>(v)); }
    StringArcMapData(Digraph& g, std::string def_val) :
      ArcMapData(MapValue(def_val)),
      map(g, def_val)
    {}
  };

  struct NodeMapData
  {
    /// where to save the map
    MapSaveDest save_dest;
    /// read-only or read-write
    bool writeable;
    /// default value
    MapValue default_value;
    virtual MapValue::Type type() = 0;
    virtual MapValue get(Node e) = 0;
    virtual void set(Node e, MapValue v) = 0;
    NodeMapData(MapValue def_val) :
      save_dest(GUI_SECT),
      writeable(true),
      default_value(def_val)
    {}
  };

  struct NumericNodeMapData : public NodeMapData
  {
    NumericNodeMap map;
    MapValue::Type type() { return MapValue::NUMERIC; }
    MapValue get(Node e) { return MapValue(map[e]); }
    void set(Node e, MapValue v) { map.set(e, static_cast<double>(v)); }
    NumericNodeMapData(Digraph& g, double def_val) :
      NodeMapData(MapValue(def_val)),
      map(g, def_val)
    {}
  };

  struct StringNodeMapData : public NodeMapData
  {
    StringNodeMap map;
    MapValue::Type type() { return MapValue::STRING; }
    MapValue get(Node e) { return MapValue(map[e]); }
    void set(Node e, MapValue v) { map.set(e, static_cast<std::string>(v)); }
    StringNodeMapData(Digraph& g, std::string def_val) :
      NodeMapData(MapValue(def_val)),
      map(g, def_val)
    {}
  };

  typedef std::map<std::string, NodeMapData*> NodeMapStore;
  typedef std::map<std::string, ArcMapData*> ArcMapStore;

  struct GUISectData
  {
    std::vector<std::string> main_node_map_names;
    std::vector<std::string> main_arc_map_names;

    std::vector<std::string> gui_node_map_names;
    std::vector<std::string> gui_arc_map_names;

    std::map<std::string, MapValue::Type> node_map_types;
    std::map<std::string, MapValue::Type> arc_map_types;

    std::map<std::string, std::map<int, double>* > numeric_node_maps;
    std::map<std::string, std::map<int, std::string>* > string_node_maps;

    std::map<std::string, std::map<int, double>* > numeric_arc_maps;
    std::map<std::string, std::map<int, std::string>* > string_arc_maps;

    std::map<int, XY> node_coord_map;
    std::map<int, XY> arrow_coord_map;

    SpecMapSaveOpts::Dest node_coords_save_dest;
    SpecMapSaveOpts::MapNum node_coords_save_map_num;
    std::string node_coords_one_map_name;
    std::string node_coords_two_maps_1_name;
    std::string node_coords_two_maps_2_name;

    SpecMapSaveOpts::Dest arrow_coords_save_dest;
    SpecMapSaveOpts::MapNum arrow_coords_save_map_num;
    std::string arrow_coords_one_map_name;
    std::string arrow_coords_two_maps_1_name;
    std::string arrow_coords_two_maps_2_name;

    ~GUISectData()
    {
      using std::map;
      using std::vector;
      using std::pair;
      using std::string;

      for (map<string, map<int, double>* >::iterator it =
          numeric_node_maps.begin(); it != numeric_node_maps.end(); ++it)
      {
        delete it->second;
      }
      for (map<string, map<int, string>* >::iterator it =
          string_node_maps.begin(); it != string_node_maps.end(); ++it)
      {
        delete it->second;
      }
      for (map<string, map<int, double>* >::iterator it =
          numeric_arc_maps.begin(); it != numeric_arc_maps.end(); ++it)
      {
        delete it->second;
      }
      for (map<string, map<int, string>* >::iterator it =
          string_arc_maps.begin(); it != string_arc_maps.end(); ++it)
      {
        delete it->second;
      }
    }
  };
public:
  ///The digraph for which the datas are stored.
  Digraph digraph;
  const Digraph& getDigraph();

private:
  GuiSectSaveDest gui_sect_save_dest;

  SpecMapSaveOpts::Dest node_coords_save_dest;
  SpecMapSaveOpts::MapNum node_coords_save_map_num;
  SpecMapSaveOpts::Dest arrow_coords_save_dest;
  SpecMapSaveOpts::MapNum arrow_coords_save_map_num;

  NodeMapStore nodemaps;
  ArcMapStore arcmaps;

  NodeLabelMap node_label;
  ArcLabelMap arc_label;

  /// the coordinates of the nodes
  NodeCoordMap node_coords;
  Digraph::NodeMap<double> node_coords_x;
  Digraph::NodeMap<double> node_coords_y;

  /// the coordinates of the arrows on the arcs
  ArrowCoordMap arrow_coords;
  Digraph::ArcMap<double> arrow_coords_x;
  Digraph::ArcMap<double> arrow_coords_y;

  ///The content of the object has changed, update is needed.
  bool modified;

  ///Name of file loaded in object.
  std::string file_name;

  // the largest node label
  int max_node_label;

  // the largest arc label
  int max_arc_label;

  std::string node_coords_one_map_name;
  std::string node_coords_two_maps_1_name;
  std::string node_coords_two_maps_2_name;

  std::string arrow_coords_one_map_name;
  std::string arrow_coords_two_maps_1_name;
  std::string arrow_coords_two_maps_2_name;

public:
  ///Stores the default values for the different visualization node attributes
  std::vector<Digraph::NodeMap<double> > default_nodemaps;

  ///Stores the default values for the different visualization arc attributes
  std::vector<Digraph::ArcMap<double> > default_arcmaps;

  ///Stores the active maps for the different visualization node attributes
  std::vector< std::string > active_nodemaps;

  /// Stores the active maps for the different visualization arc attributes
  std::vector< std::string > active_arcmaps;

protected:

  /// Signal emitted on any change made on map values

  /// Signal emitted if the visualization of the maps might have to be updated.
  /// bool shows us whether the changed map is arc or nodemap.
  /// int tells us the refreshed property
  sigc::signal<void, bool, int> signal_prop;

  /// Signal emitted in the case of nodemap addition

  /// std::string is the
  ///name of the new map
  sigc::signal<void, std::string, MapValue::Type> signal_node_map;

  /// Signal emitted in the case of arcmap addition

  /// std::string is the
  ///name of the new map
  sigc::signal<void, std::string, MapValue::Type> signal_arc_map;

  /// Signal emitted, when entry in \ref MapWin should be changed.
  sigc::signal<void, bool, int, std::string> signal_map_win;

  /// Signal emitted, when entry in \ref DesignWin should be changed.
  sigc::signal<void, double, double, int> signal_design_win;

  ///Signal emitted when background should be set by \ref NoteBookTab
  sigc::signal<void> signal_background;

  ///Iteration number during digraph design
  int iterations;

  ///Attraction factor during digraph design
  double attraction;

  ///Propulsation factor during digraph design
  double propulsation;

public:
  ///Constructor of MapStorage.

  ///Its all activity is initializing default values
  ///for different visualization attributes.
  MapStorage();

  ///Destructor of MapStorage

  ///Maps stored here are created with new. Destructor
  ///deletes them to free up the reserved memory.
  ~MapStorage();

  /// Registrates if the shown map by any attribute has changed to another.

  ///It handles the \ref active_arcmaps and
  ///\ref active_nodemaps vectors. It also emits \ref signal_prop signal to let
  ///know the interested objects that the visible map of a certain
  ///attribute has changed.
  ///\param itisarc arcmap or nodemap has changed
  ///\param prop the property of which the map is changed
  ///\param mapname the visible map
  void changeActiveMap(bool itisarc , int prop , std::string mapname);

  ///Emits signals that let change the active maps in \ref MapWin.
  void broadcastActiveMaps();

  /// Returns the active arcmap shown by a visualization property.

  /// \param prop is the property
  ///that shows the requested map.
  std::string getActiveArcMap(int prop);

  /// Returns the active nodemap shown by a visualization property.

  /// \param prop is the property
  ///that shows the requested map.
  std::string getActiveNodeMap(int prop);

  /// Returns the names of the arcmaps stored here.
  std::vector<std::string> getArcMapList(MapType type = ALL);

  /// Returns the names of the nodemaps stored here.
  std::vector<std::string> getNodeMapList(MapType type = ALL);

  ///returns \ref signal_prop to be able to connect functions to it
  sigc::signal<void, bool, int> signal_prop_ch();

  ///returns \ref signal_node_map to be able to connect functions to it
  sigc::signal<void, std::string, MapValue::Type> signal_node_map_ch(){return signal_node_map;};

  ///returns \ref signal_arc_map to be able to connect functions to it
  sigc::signal<void, std::string, MapValue::Type> signal_arc_map_ch(){return signal_arc_map;};

  ///returns \ref signal_map_win to be able to connect functions to it
  sigc::signal<void, bool, int, std::string> signal_map_win_ch(){return signal_map_win;};

  ///returns \ref signal_design_win to be able to connect functions to it
  sigc::signal<void, double, double, int> signal_design_win_ch(){return signal_design_win;};

  void createNodeMap(const std::string& name, MapValue::Type type,
    MapValue def_val);
  void createArcMap(const std::string& name, MapValue::Type type,
    MapValue def_val);

  ///returns \ref signal_background to be able to connect functions to it
  sigc::signal<void> signal_background_ch(){return signal_background;};


  ///Adds given map to storage.

  ///Emits \ref signal_prop if mapvalues have changed, and MapStorage gets to know it.

  ///If values in a map have changed, this function checks, whether it is displayed.
  ///This check means searching the given mapname between active maps
  ///(\ref active_nodemaps, \ref active_arcmaps). If it is there at a certain property,
  ///it emits a signal with the property, where the gotten mapname was found. One signal
  ///is emitted for each property displaying the given map.
  ///\param itisarc whether the map an arcmap or nodemap
  ///\param mapname name of map to visualize
  void mapChanged(bool itisarc, std::string mapname);

  ///Read datas from the given filename.
  int readFromFile(const std::string &);

  ///Save datas to the given filename.
  void writeToFile(const std::string &);

  ///Deletes all datastructures stored here.
  void clear();

  void get_design_data(double &, double &, int &);
  void set_attraction(double);
  void set_propulsation(double);
  void set_iteration(int);

  void redesign_data_changed();

  XY getNodeCoords(Node n) const;
  void setNodeCoords(Node n, XY c);
  XY getArrowCoords(Arc e) const;
  void setArrowCoords(Arc e, XY c);

  MapValue get(const std::string& name, Node node) const;
  void set(const std::string& name, Node node, MapValue val);
  MapValue get(const std::string& name, Arc arc) const;
  void set(const std::string& name, Arc arc, MapValue val);

  const std::string& getFileName() const;
  void setFileName(const std::string& fn);

  bool getModified() const;
  void setModified(bool m = true);

  Node addNode(XY);
  Arc addArc(Node, Node);

  NumericNodeMap& getNumericNodeMap(const std::string& name);
  StringNodeMap& getStringNodeMap(const std::string& name);
  NumericArcMap& getNumericArcMap(const std::string& name);
  StringArcMap& getStringArcMap(const std::string& name);

  MapValueArcMap getArcMap(const std::string& name);
  MapValueNodeMap getNodeMap(const std::string& name);

  int getLabel(Node) const;
  int getLabel(Arc) const;

  GuiSectSaveDest getGUIDataSaveLocation();
  void setGUIDataSaveLocation(GuiSectSaveDest dest);

  MapSaveDest getNodeMapSaveDest(std::string name) const;
  MapSaveDest getArcMapSaveDest(std::string name) const;
  void setNodeMapSaveDest(std::string name, MapSaveDest dest);
  void setArcMapSaveDest(std::string name, MapSaveDest dest);

  SpecMapSaveOpts::Dest getNodeCoordsSaveDest();
  SpecMapSaveOpts::Dest getArrowCoordsSaveDest();
  void setNodeCoordsSaveDest(SpecMapSaveOpts::Dest dest);
  void setArrowCoordsSaveDest(SpecMapSaveOpts::Dest dest);

  SpecMapSaveOpts::MapNum getNodeCoordsSaveMapNum();
  SpecMapSaveOpts::MapNum getArrowCoordsSaveMapNum();
  void setNodeCoordsSaveMapNum(SpecMapSaveOpts::MapNum num);
  void setArrowCoordsSaveMapNum(SpecMapSaveOpts::MapNum num);

  MapValue::Type getNodeMapElementType(std::string name) const;
  MapValue::Type getArcMapElementType(std::string name) const;

  const NodeLabelMap& getNodeLabelMap();
  const ArcLabelMap& getArcLabelMap();

  bool nodeMapExists(std::string name);
  bool arcMapExists(std::string name);

  std::vector<std::string> getArcMaps(MapType type = ALL);
  std::vector<std::string> getNodeMaps(MapType type = ALL);

  NodeCoordMap& getNodeCoordMap();
  ArrowCoordMap& getArrowCoordMap();

  const std::string& getNodeCoordsOneMapName();
  const std::string& getNodeCoordsTwoMaps1Name();
  const std::string& getNodeCoordsTwoMaps2Name();
  void setNodeCoordsOneMapName(const std::string& name);
  void setNodeCoordsTwoMaps1Name(const std::string& name);
  void setNodeCoordsTwoMaps2Name(const std::string& name);

  const std::string& getArrowCoordsOneMapName();
  const std::string& getArrowCoordsTwoMaps1Name();
  const std::string& getArrowCoordsTwoMaps2Name();
  void setArrowCoordsOneMapName(const std::string& name);
  void setArrowCoordsTwoMaps1Name(const std::string& name);
  void setArrowCoordsTwoMaps2Name(const std::string& name);

private:
  ArcMapData* getArcMapData(std::string name) const;
  NodeMapData* getNodeMapData(std::string name) const;
  void readLGF(
      const std::string& filename,
      bool read_arc_label,
      const std::vector<std::string>& node_map_names,
      const std::vector<std::string>& arc_map_names,
      const std::map<std::string, MapValue::Type>& node_map_types,
      const std::map<std::string, MapValue::Type>& arc_map_types,
      const std::string& node_coord_xmap_name,
      const std::string& node_coord_ymap_name,
      const std::string& arrow_coord_xmap_name,
      const std::string& arrow_coord_ymap_name);

public:
  void exportDigraphToEPS(std::vector<bool>, std::string, std::string);
};

#endif //MAPSTORAGE_H
