/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include <new_map_win.h>
#include <nbtab.h>
#include <mapstorage.h>

bool NewMapWin::closeIfEscapeIsPressed(GdkEventKey* e)
{
  if(e->keyval==GDK_Escape)
  {
    hide();
  }
  return true;
}

NewMapWin::NewMapWin(const std::string& title, NoteBookTab & mw, bool itisarc, bool arcnode, MapType type):Gtk::Dialog(title, true, true),mytab(mw),node("Create NodeMap"),arc("Create ArcMap"),map_type(type)
{
  set_default_size(200, 50);

  signal_key_press_event().connect(sigc::mem_fun(*this, &NewMapWin::closeIfEscapeIsPressed));

  Gtk::VBox * vbox=get_vbox();

  //entries
  table=new Gtk::Table(5, 2, false);

  label=new Gtk::Label;
  label->set_text("Name of new map:");
  name.set_text("");

  (*table).attach(*label,0,1,0,1,Gtk::SHRINK,Gtk::SHRINK,10,3);
  (*table).attach(name,1,2,0,1,Gtk::SHRINK,Gtk::SHRINK,10,3);

  lblType.set_label("Element type:");
  if (map_type & NUM)
    cbType.append_text("Numeric");
  if (map_type & STR)
    cbType.append_text("String");
  cbType.set_active(0);

  (*table).attach(lblType,0,1,1,2,Gtk::SHRINK,Gtk::SHRINK,10,3);
  (*table).attach(cbType, 1,2,1,2,Gtk::SHRINK,Gtk::SHRINK,10,3);

  label=new Gtk::Label;
  label->set_text("Default value in the map:");
  default_value.set_text("0");

  (*table).attach(*label,0,1,2,3,Gtk::SHRINK,Gtk::SHRINK,10,3);
  (*table).attach(default_value,1,2,2,3,Gtk::SHRINK,Gtk::SHRINK,10,3);

  //node vs. arc map selector
  Gtk::RadioButton::Group group = node.get_group();
  arc.set_group(group);

  if(arcnode)
  {
    (*table).attach(node,0,1,3,4,Gtk::SHRINK,Gtk::SHRINK,10,3);
    (*table).attach(arc,1,2,3,4,Gtk::SHRINK,Gtk::SHRINK,10,3);
  }
  else
  {
    if(itisarc)
    {
      arc.set_active();
    }
    else
    {
      node.set_active();
    }
  }

  (*table).attach(lblErrorMsg,0,2,4,5,Gtk::SHRINK,Gtk::SHRINK,10,3);

  vbox->pack_start(*table);

  //OK button
  add_button(Gtk::Stock::OK, Gtk::RESPONSE_OK);

  show_all_children();

}

void NewMapWin::setErrorMsg(const Glib::ustring& msg)
{
  lblErrorMsg.set_markup("<i><small>" + msg + "</small></i>");
}

std::vector<double>* NewMapWin::evaluate_expr(const std::string polishform, bool itisarc)
{
  MapStorage& ms = *mytab.mapstorage;

  std::vector<double>* ret = new std::vector<double>;
  std::stack<double> polishstack;

  if (itisarc)
  {
    for(ArcIt k(ms.digraph); k!=INVALID; ++k)
    {
      for(int i=0;i<(int)polishform.size();i++)
      {
        double op1=0, op2=0;
        bool operation=true;
        switch(polishform[i])
        {
          case '+':
          case '-':
          case '/':
          case '*':
            op1=polishstack.top();
            polishstack.pop();
            op2=polishstack.top();
            polishstack.pop();
            break;
          default:
            //substitute variable
            std::vector<std::string> maps = ms.getArcMapList(NUM);
            bool itisvar=(std::find(maps.begin(), maps.end(), ch2var[ polishform[i] ]) != maps.end());
            if(itisvar)
            {
              polishstack.push(ms.get(ch2var[ polishform[i] ], k));
            }
            else
            {
              polishstack.push(atof(ch2var[ polishform[i] ].c_str()));
            }
            operation=false;
            break;
        }
        if(operation)
        {
          double res;
          switch(polishform[i])
          {
            case '+':
              res=op1+op2;
              break;
            case '-':
              res=op2-op1;
              break;
            case '/':
              res=op2/op1;
              break;
            case '*':
              res=op1*op2;
              break;
            default:
              std::cout << "How could we get here?" << std::endl;
              break;
          }
          polishstack.push(res);
        }
      }//foreach letter in polishform
      ret->push_back(polishstack.top());
    }//foreach arc
  }
  else
  {
    for(NodeIt k(ms.digraph); k!=INVALID; ++k)
    {
      for(int i=0;i<(int)polishform.size();i++)
      {
        double op1=0, op2=0;
        bool operation=true;
        switch(polishform[i])
        {
          case '+':
          case '-':
          case '/':
          case '*':
            op1=polishstack.top();
            polishstack.pop();
            op2=polishstack.top();
            polishstack.pop();
            break;
          default:
            //substitute variable
            std::vector<std::string> maps = ms.getNodeMapList(NUM);
            bool itisvar=(std::find(maps.begin(), maps.end(), ch2var[ polishform[i] ]) != maps.end());
            if(itisvar)
            {
              polishstack.push(ms.get(ch2var[ polishform[i] ], k));
            }
            else
            {
              polishstack.push(atof(ch2var[ polishform[i] ].c_str()));
            }
            operation=false;
            break;
        }
        if(operation)
        {
          double res;
          switch(polishform[i])
          {
            case '+':
              res=op1+op2;
              break;
            case '-':
              res=op2-op1;
              break;
            case '/':
              res=op2/op1;
              break;
            case '*':
              res=op1*op2;
              break;
            default:
              std::cout << "How could we get here?" << std::endl;
              break;
          }
          polishstack.push(res);
        }
      }//foreach letter in polishform
      ret->push_back(polishstack.top());
    }//foreach arc
  }
  return ret;
}

void NewMapWin::on_response(int response_id)
{
  MapStorage& ms = *mytab.mapstorage;

  if(response_id==Gtk::RESPONSE_OK)
  {
    std::string map_name = name.get_text();
    std::string def_val = default_value.get_text();

    if (map_name.empty())
    {
      setErrorMsg("No map name given.");
      return;
    }

    // check whether the map already exists
    if (arc.get_active())
    {
      if (ms.arcMapExists(map_name))
      {
        setErrorMsg("Map '" + map_name + "' already exists.");
        return;
      }
    }
    else
    {
      if (ms.nodeMapExists(map_name))
      {
        setErrorMsg("Map '" + map_name + "' already exists.");
        return;
      }
    }

    Glib::ustring text = cbType.get_active_text();
    if (text == "Numeric")
    {
      double d;
      char *endptr;
      d = strtod(def_val.c_str(), &endptr);
      if (def_val.c_str() + def_val.length() == endptr)
      {
        // the full string was a number
        if (arc.get_active())
          ms.createArcMap(map_name, MapValue::NUMERIC,
              MapValue(d));
        else
          ms.createNodeMap(map_name, MapValue::NUMERIC,
              MapValue(d));
      }
      else
      {
        // let't try to evaluate the string as an arithmetic expression
        std::string polishform =
          string2Polishform(def_val, arc.get_active());
        if (polishform.empty())
          return;
        std::vector<double>* values =
          evaluate_expr(polishform, arc.get_active());
        if (arc.get_active())
        {
          ms.createArcMap(map_name, MapValue::NUMERIC,
              MapValue(0.0));
          std::vector<double>::const_iterator vit = values->begin();
          for (ArcIt it(ms.digraph); it != INVALID; ++it)
          {
            ms.set(map_name, it, MapValue(*vit));
            ++vit;
          }
        }
        else
        {
          ms.createNodeMap(map_name, MapValue::NUMERIC,
              MapValue(0.0));
          std::vector<double>::const_iterator vit = values->begin();
          for (NodeIt it(ms.digraph); it != INVALID; ++it)
          {
            ms.set(map_name, it, MapValue(*vit));
            ++vit;
          }
        }
        delete values;
      }
    }
    else if (text == "String")
    {
      if (arc.get_active())
        ms.createArcMap(map_name, MapValue::STRING,
            MapValue(def_val));
      else
        ms.createNodeMap(map_name, MapValue::STRING,
            MapValue(def_val));
    }

    name.set_text("");
    default_value.set_text("0");
    arc.show();
    node.show();
    hide();
  }
}


std::string NewMapWin::string2Polishform(std::string rawcommand, bool itisarc)
{
  bool valid_entry=true;

  std::map<std::string, int> str2i;

  std::string command;

  std::string variable;

  char index='a';

  for(int i=0;(valid_entry&&(i<(int)rawcommand.size()));i++)
  {
    switch(rawcommand[i])
    {
      case '+':
      case '-':
      case '*':
      case '/':
      case ')':
      case '(':
        if(!variable.empty())
        {
          valid_entry=validVariable(variable, itisarc);
          ch2var[index]=variable;
          command+=index;
          index++;
          variable.erase(0,variable.size());	  
        }
        command+=rawcommand[i];
        break;
      default:
        variable+=rawcommand[i];
        break;
    }
  }

  if(!variable.empty()&&valid_entry)
  {
    valid_entry=validVariable(variable, itisarc);
    ch2var[index]=variable;
    command+=index;
    index++;
    variable.erase(0,variable.size());	  
  }

  if(valid_entry)
  {
    unsigned int pr=10000;
    bool prevmult=false;
    unsigned int prev_change=pr;
    unsigned int prev_br=pr;
    int counter=0;
    std::string comm_nobr="";
    std::vector<unsigned int> p;
    p.resize(counter+1);

    //limits
    //6 brackets embedded
    //100 operation in a row from the same priority

    for(int i=0;i<(int)command.size();i++)
    {
      bool put_in_string=true;
      switch(command[i])
      {
        case '(':
          pr=prev_br+10000;
          prev_br=pr;
          prevmult=false;
          put_in_string=false;
          break;
        case ')':
          pr=prev_br-10000;
          prev_br=pr;
          prevmult=false;
          put_in_string=false;
          break;
        case '+':
        case '-':
          if(prevmult)
          {
            pr=prev_change;
          }
          p[counter]=pr;
          pr-=100;

          prevmult=false;
          break;
        case '/':
        case '*':
          if(!prevmult)
          {
            prev_change=pr;
            pr+=200;
            pr-=1;
          }
          p[counter]=pr;
          pr-=1;
          prevmult=true;
          break;
        default:
          p[counter]=65000;
          break;
      }
      if(put_in_string)
      {
        counter++;
        p.resize(counter+1);
        comm_nobr=comm_nobr+command[i];
      }
    }

    tree_node * root=weightedString2Tree(comm_nobr, p, 0);

    std::string polishform=postOrder(root);

    deleteTree(root);

    return polishform;
  }
  return "";
}

void NewMapWin::deleteTree(NewMapWin::tree_node * node)
{
  if(node->left_child!=NULL)
  {
    deleteTree(node->left_child);
  }
  if(node->right_child!=NULL)
  {
    deleteTree(node->right_child);
  }
  delete node;
}

NewMapWin::tree_node * NewMapWin::weightedString2Tree(std::string to_tree, std::vector<unsigned int> & p, int offset)
{
  unsigned int min=p[offset];
  int minplace=0;
  for(int i=0;i<(int)to_tree.size();i++)
  {
    if(min>p[offset+i])
    {
      min=p[offset+i];
      minplace=i;
    }
  }
  tree_node * act_node=new tree_node;
  act_node->ch=to_tree[minplace];
  if(to_tree.size()>=3)
  {
    act_node->left_child=weightedString2Tree(to_tree.substr(0,minplace), p, offset);
    act_node->right_child=weightedString2Tree(to_tree.substr(minplace+1,to_tree.size()-minplace-1), p, offset+minplace+1);
  }
  else
  {
    act_node->left_child=NULL;
    act_node->right_child=NULL;
  }
  return act_node;
}

std::string NewMapWin::postOrder(tree_node * subtree)
{
  std::string subtree_to_string;
  if(subtree->left_child)
  {
    subtree_to_string=postOrder(subtree->left_child);
  }
  if(subtree->right_child)
  {
    subtree_to_string=subtree_to_string+postOrder(subtree->right_child);
  }
  subtree_to_string=subtree_to_string+subtree->ch;
  return subtree_to_string;
}

bool NewMapWin::validVariable(std::string variable, bool itisarc)
{
  MapStorage& ms = *mytab.mapstorage;

  bool cancel;
  //is it mapname?
  if(itisarc)
  {
    std::vector<std::string> arc_maps =
      ms.getArcMapList(NUM);
    cancel=(std::find(arc_maps.begin(), arc_maps.end(), variable)==arc_maps.end());
  }
  else
  {
    std::vector<std::string> node_maps =
      ms.getNodeMapList(NUM);
    cancel=(std::find(node_maps.begin(), node_maps.end(), variable)==node_maps.end());
  }
  //maybe it is number
  int point_num=0;
  if(cancel)
  {
    cancel=false;
    for(int j=0;(!cancel)&&(j<(int)variable.size());j++)
    {
      if(((variable[j]<'0')||(variable[j]>'9'))&&(variable[j]!='.'))
      {
        cancel=true;
      }
      else
      {
        if(variable[j]=='.')
        {
          point_num++;
          if(point_num>1)
          {
            cancel=true;
          }
        }
      }
    }
  }
  if(cancel)
  {
    return false;
  }
  return true;
}
