/* -*- C++ -*-
 * lemon/graph_utils.h - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2005 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef LEMON_GRAPH_UTILS_H
#define LEMON_GRAPH_UTILS_H

#include <iterator>
#include <vector>
#include <map>

#include <lemon/invalid.h>
#include <lemon/utility.h>
#include <lemon/maps.h>
#include <lemon/bits/alteration_notifier.h>

///\ingroup gutils
///\file
///\brief Graph utilities.
///
///


namespace lemon {

  /// \addtogroup gutils
  /// @{

  /// \brief Function to count the items in the graph.
  ///
  /// This function counts the items (nodes, edges etc) in the graph.
  /// The complexity of the function is O(n) because
  /// it iterates on all of the items.

  template <typename Graph, typename ItemIt>
  inline int countItems(const Graph& g) {
    int num = 0;
    for (ItemIt it(g); it != INVALID; ++it) {
      ++num;
    }
    return num;
  }

  // Node counting:

  template <typename Graph>
  inline
  typename enable_if<typename Graph::NodeNumTag, int>::type
  _countNodes(const Graph &g) {
    return g.nodeNum();
  }

  template <typename Graph>
  inline int _countNodes(Wrap<Graph> w) {
    return countItems<Graph, typename Graph::NodeIt>(w.value);
  }

  /// \brief Function to count the nodes in the graph.
  ///
  /// This function counts the nodes in the graph.
  /// The complexity of the function is O(n) but for some
  /// graph structures it is specialized to run in O(1).
  ///
  /// \todo refer how to specialize it

  template <typename Graph>
  inline int countNodes(const Graph& g) {
    return _countNodes<Graph>(g);
  }

  // Edge counting:

  template <typename Graph>
  inline
  typename enable_if<typename Graph::EdgeNumTag, int>::type
  _countEdges(const Graph &g) {
    return g.edgeNum();
  }

  template <typename Graph>
  inline int _countEdges(Wrap<Graph> w) {
    return countItems<Graph, typename Graph::EdgeIt>(w.value);
  }

  /// \brief Function to count the edges in the graph.
  ///
  /// This function counts the edges in the graph.
  /// The complexity of the function is O(e) but for some
  /// graph structures it is specialized to run in O(1).

  template <typename Graph>
  inline int countEdges(const Graph& g) {
    return _countEdges<Graph>(g);
  }

  // Undirected edge counting:

  template <typename Graph>
  inline
  typename enable_if<typename Graph::EdgeNumTag, int>::type
  _countUndirEdges(const Graph &g) {
    return g.undirEdgeNum();
  }

  template <typename Graph>
  inline int _countUndirEdges(Wrap<Graph> w) {
    return countItems<Graph, typename Graph::UndirEdgeIt>(w.value);
  }

  /// \brief Function to count the undirected edges in the graph.
  ///
  /// This function counts the undirected edges in the graph.
  /// The complexity of the function is O(e) but for some
  /// graph structures it is specialized to run in O(1).

  template <typename Graph>
  inline int countUndirEdges(const Graph& g) {
    return _countUndirEdges<Graph>(g);
  }



  template <typename Graph, typename DegIt>
  inline int countNodeDegree(const Graph& _g, const typename Graph::Node& _n) {
    int num = 0;
    for (DegIt it(_g, _n); it != INVALID; ++it) {
      ++num;
    }
    return num;
  }

  /// \brief Function to count the number of the out-edges from node \c n.
  ///
  /// This function counts the number of the out-edges from node \c n
  /// in the graph.  
  template <typename Graph>
  inline int countOutEdges(const Graph& _g,  const typename Graph::Node& _n) {
    return countNodeDegree<Graph, typename Graph::OutEdgeIt>(_g, _n);
  }

  /// \brief Function to count the number of the in-edges to node \c n.
  ///
  /// This function counts the number of the in-edges to node \c n
  /// in the graph.  
  template <typename Graph>
  inline int countInEdges(const Graph& _g,  const typename Graph::Node& _n) {
    return countNodeDegree<Graph, typename Graph::InEdgeIt>(_g, _n);
  }


  template <typename Graph>
  inline
  typename enable_if<typename Graph::FindEdgeTag, typename Graph::Edge>::type 
  _findEdge(const Graph &g,
	    typename Graph::Node u, typename Graph::Node v,
	    typename Graph::Edge prev = INVALID) {
    return g.findEdge(u, v, prev);
  }

  template <typename Graph>
  inline typename Graph::Edge 
  _findEdge(Wrap<Graph> w,
	    typename Graph::Node u, 
	    typename Graph::Node v,
	    typename Graph::Edge prev = INVALID) {
    const Graph& g = w.value;
    if (prev == INVALID) {
      typename Graph::OutEdgeIt e(g, u);
      while (e != INVALID && g.target(e) != v) ++e;
      return e;
    } else {
      typename Graph::OutEdgeIt e(g, prev); ++e;
      while (e != INVALID && g.target(e) != v) ++e;
      return e;
    }
  }

  /// \brief Finds an edge between two nodes of a graph.
  ///
  /// Finds an edge from node \c u to node \c v in graph \c g.
  ///
  /// If \c prev is \ref INVALID (this is the default value), then
  /// it finds the first edge from \c u to \c v. Otherwise it looks for
  /// the next edge from \c u to \c v after \c prev.
  /// \return The found edge or \ref INVALID if there is no such an edge.
  ///
  /// Thus you can iterate through each edge from \c u to \c v as it follows.
  /// \code
  /// for(Edge e=findEdge(g,u,v);e!=INVALID;e=findEdge(g,u,v,e)) {
  ///   ...
  /// }
  /// \endcode
  // /// \todo We may want to use the "GraphBase" 
  // /// interface here...
  // /// It would not work with the undirected graphs.
  template <typename Graph>
  inline typename Graph::Edge findEdge(const Graph &g,
				       typename Graph::Node u, 
				       typename Graph::Node v,
				       typename Graph::Edge prev = INVALID) {
    return _findEdge<Graph>(g, u, v, prev);
  }

  /// \brief Iterator for iterating on edges connected the same nodes.
  ///
  /// Iterator for iterating on edges connected the same nodes. It is 
  /// higher level interface for the findEdge() function. You can
  /// use it the following way:
  /// \code
  /// for (ConEdgeIt<Graph> it(g, src, trg); it != INVALID; ++it) {
  ///   ...
  /// }
  /// \endcode
  ///
  /// \author Balazs Dezso 
  template <typename _Graph>
  class ConEdgeIt : public _Graph::Edge {
  public:

    typedef _Graph Graph;
    typedef typename Graph::Edge Parent;

    typedef typename Graph::Edge Edge;
    typedef typename Graph::Node Node;

    /// \brief Constructor.
    ///
    /// Construct a new ConEdgeIt iterating on the edges which
    /// connects the \c u and \c v node.
    ConEdgeIt(const Graph& g, Node u, Node v) : graph(g) {
      Parent::operator=(findEdge(graph, u, v));
    }

    /// \brief Constructor.
    ///
    /// Construct a new ConEdgeIt which continues the iterating from 
    /// the \c e edge.
    ConEdgeIt(const Graph& g, Edge e) : Parent(e), graph(g) {}
    
    /// \brief Increment operator.
    ///
    /// It increments the iterator and gives back the next edge.
    ConEdgeIt& operator++() {
      Parent::operator=(findEdge(graph, graph.source(*this), 
				 graph.target(*this), *this));
      return *this;
    }
  private:
    const Graph& graph;
  };

  /// \brief Copy a map.
  ///
  /// This function copies the \c source map to the \c target map. It uses the
  /// given iterator to iterate on the data structure and it uses the \c ref
  /// mapping to convert the source's keys to the target's keys.
  template <typename Target, typename Source, 
	    typename ItemIt, typename Ref>	    
  void copyMap(Target& target, const Source& source, 
	       ItemIt it, const Ref& ref) {
    for (; it != INVALID; ++it) {
      target[ref[it]] = source[it];
    }
  }

  /// \brief Copy the source map to the target map.
  ///
  /// Copy the \c source map to the \c target map. It uses the given iterator
  /// to iterate on the data structure.
  template <typename Target, typename Source, 
	    typename ItemIt>	    
  void copyMap(Target& target, const Source& source, ItemIt it) {
    for (; it != INVALID; ++it) {
      target[it] = source[it];
    }
  }


  /// \brief Class to copy a graph.
  ///
  /// Class to copy a graph to an other graph (duplicate a graph). The
  /// simplest way of using it is through the \c copyGraph() function.
  template <typename Target, typename Source>
  class GraphCopy {
  public: 
    typedef typename Source::Node Node;
    typedef typename Source::NodeIt NodeIt;
    typedef typename Source::Edge Edge;
    typedef typename Source::EdgeIt EdgeIt;

    typedef typename Source::template NodeMap<typename Target::Node>NodeRefMap;
    typedef typename Source::template EdgeMap<typename Target::Edge>EdgeRefMap;

    /// \brief Constructor for the GraphCopy.
    ///
    /// It copies the content of the \c _source graph into the
    /// \c _target graph. It creates also two references, one beetween
    /// the two nodeset and one beetween the two edgesets.
    GraphCopy(Target& _target, const Source& _source) 
      : source(_source), target(_target), 
	nodeRefMap(_source), edgeRefMap(_source) {
      for (NodeIt it(source); it != INVALID; ++it) {
	nodeRefMap[it] = target.addNode();
      }
      for (EdgeIt it(source); it != INVALID; ++it) {
	edgeRefMap[it] = target.addEdge(nodeRefMap[source.source(it)], 
					nodeRefMap[source.target(it)]);
      }
    }

    /// \brief Copies the node references into the given map.
    ///
    /// Copies the node references into the given map.
    template <typename NodeRef>
    const GraphCopy& nodeRef(NodeRef& map) const {
      for (NodeIt it(source); it != INVALID; ++it) {
	map.set(it, nodeRefMap[it]);
      }
      return *this;
    }

    /// \brief Reverse and copies the node references into the given map.
    ///
    /// Reverse and copies the node references into the given map.
    template <typename NodeRef>
    const GraphCopy& nodeCrossRef(NodeRef& map) const {
      for (NodeIt it(source); it != INVALID; ++it) {
	map.set(nodeRefMap[it], it);
      }
      return *this;
    }

    /// \brief Copies the edge references into the given map.
    ///
    /// Copies the edge references into the given map.
    template <typename EdgeRef>
    const GraphCopy& edgeRef(EdgeRef& map) const {
      for (EdgeIt it(source); it != INVALID; ++it) {
	map.set(it, edgeRefMap[it]);
      }
      return *this;
    }

    /// \brief Reverse and copies the edge references into the given map.
    ///
    /// Reverse and copies the edge references into the given map.
    template <typename EdgeRef>
    const GraphCopy& edgeCrossRef(EdgeRef& map) const {
      for (EdgeIt it(source); it != INVALID; ++it) {
	map.set(edgeRefMap[it], it);
      }
      return *this;
    }

    /// \brief Make copy of the given map.
    ///
    /// Makes copy of the given map for the newly created graph. 
    /// The new map's key type is the target graph's node type,
    /// and the copied map's key type is the source graph's node
    /// type.  
    template <typename TargetMap, typename SourceMap>
    const GraphCopy& nodeMap(TargetMap& tMap, const SourceMap& sMap) const {
      copyMap(tMap, sMap, NodeIt(source), nodeRefMap);
      return *this;
    }

    /// \brief Make copy of the given map.
    ///
    /// Makes copy of the given map for the newly created graph. 
    /// The new map's key type is the target graph's edge type,
    /// and the copied map's key type is the source graph's edge
    /// type.  
    template <typename TargetMap, typename SourceMap>
    const GraphCopy& edgeMap(TargetMap& tMap, const SourceMap& sMap) const {
      copyMap(tMap, sMap, EdgeIt(source), edgeRefMap);
      return *this;
    }

    /// \brief Gives back the stored node references.
    ///
    /// Gives back the stored node references.
    const NodeRefMap& nodeRef() const {
      return nodeRefMap;
    }

    /// \brief Gives back the stored edge references.
    ///
    /// Gives back the stored edge references.
    const EdgeRefMap& edgeRef() const {
      return edgeRefMap;
    }

  private:
    
    const Source& source;
    Target& target;

    NodeRefMap nodeRefMap;
    EdgeRefMap edgeRefMap;
  };

  /// \brief Copy a graph to an other graph.
  ///
  /// Copy a graph to an other graph.
  /// The usage of the function:
  /// 
  /// \code
  /// copyGraph(trg, src).nodeRef(nr).edgeCrossRef(ecr);
  /// \endcode
  /// 
  /// After the copy the \c nr map will contain the mapping from the
  /// source graph's nodes to the target graph's nodes and the \c ecr will
  /// contain the mapping from the target graph's edges to the source's
  /// edges.
  template <typename Target, typename Source>
  GraphCopy<Target, Source> copyGraph(Target& target, const Source& source) {
    return GraphCopy<Target, Source>(target, source);
  }

  template <typename _Graph, typename _Item>
  class ItemSetTraits {};
  
  template <typename _Graph>
  class ItemSetTraits<_Graph, typename _Graph::Node> {
  public:
    
    typedef _Graph Graph;

    typedef typename Graph::Node Item;
    typedef typename Graph::NodeIt ItemIt;

    template <typename _Value>
    class Map : public Graph::template NodeMap<_Value> {
    public:
      typedef typename Graph::template NodeMap<_Value> Parent; 
      typedef typename Parent::Value Value;

      Map(const Graph& _graph) : Parent(_graph) {}
      Map(const Graph& _graph, const Value& _value) 
	: Parent(_graph, _value) {}
    };

  };

  template <typename _Graph>
  class ItemSetTraits<_Graph, typename _Graph::Edge> {
  public:
    
    typedef _Graph Graph;

    typedef typename Graph::Edge Item;
    typedef typename Graph::EdgeIt ItemIt;

    template <typename _Value>
    class Map : public Graph::template EdgeMap<_Value> {
    public:
      typedef typename Graph::template EdgeMap<_Value> Parent; 
      typedef typename Parent::Value Value;

      Map(const Graph& _graph) : Parent(_graph) {}
      Map(const Graph& _graph, const Value& _value) 
	: Parent(_graph, _value) {}
    };

  };

  template <typename _Graph>
  class ItemSetTraits<_Graph, typename _Graph::UndirEdge> {
  public:
    
    typedef _Graph Graph;

    typedef typename Graph::UndirEdge Item;
    typedef typename Graph::UndirEdgeIt ItemIt;

    template <typename _Value>
    class Map : public Graph::template UndirEdgeMap<_Value> {
    public:
      typedef typename Graph::template UndirEdgeMap<_Value> Parent; 
      typedef typename Parent::Value Value;

      Map(const Graph& _graph) : Parent(_graph) {}
      Map(const Graph& _graph, const Value& _value) 
	: Parent(_graph, _value) {}
    };

  };

  /// @}

  /// \addtogroup graph_maps
  /// @{

  template <typename Map, typename Enable = void>
  struct ReferenceMapTraits {
    typedef typename Map::Value Value;
    typedef typename Map::Value& Reference;
    typedef const typename Map::Value& ConstReference;
    typedef typename Map::Value* Pointer;
    typedef const typename Map::Value* ConstPointer;
  };

  template <typename Map>
  struct ReferenceMapTraits<
    Map, 
    typename enable_if<typename Map::FullTypeTag, void>::type
  > {
    typedef typename Map::Value Value;
    typedef typename Map::Reference Reference;
    typedef typename Map::ConstReference ConstReference;
    typedef typename Map::Pointer Pointer;
    typedef typename Map::ConstPointer ConstPointer;
  };

  /// Provides an immutable and unique id for each item in the graph.

  /// The IdMap class provides a unique and immutable id for each item of the
  /// same type (e.g. node) in the graph. This id is <ul><li>\b unique:
  /// different items (nodes) get different ids <li>\b immutable: the id of an
  /// item (node) does not change (even if you delete other nodes).  </ul>
  /// Through this map you get access (i.e. can read) the inner id values of
  /// the items stored in the graph. This map can be inverted with its member
  /// class \c InverseMap.
  ///
  template <typename _Graph, typename _Item>
  class IdMap {
  public:
    typedef _Graph Graph;
    typedef int Value;
    typedef _Item Item;
    typedef _Item Key;

    typedef True NeedCopy;

    /// \brief Constructor.
    ///
    /// Constructor for creating id map.
    IdMap(const Graph& _graph) : graph(&_graph) {}

    /// \brief Gives back the \e id of the item.
    ///
    /// Gives back the immutable and unique \e id of the map.
    int operator[](const Item& item) const { return graph->id(item);}


  private:
    const Graph* graph;

  public:

    /// \brief The class represents the inverse of its owner (IdMap).
    ///
    /// The class represents the inverse of its owner (IdMap).
    /// \see inverse()
    class InverseMap {
    public:

      typedef True NeedCopy;

      /// \brief Constructor.
      ///
      /// Constructor for creating an id-to-item map.
      InverseMap(const Graph& _graph) : graph(&_graph) {}

      /// \brief Constructor.
      ///
      /// Constructor for creating an id-to-item map.
      InverseMap(const IdMap& idMap) : graph(idMap.graph) {}

      /// \brief Gives back the given item from its id.
      ///
      /// Gives back the given item from its id.
      /// 
      Item operator[](int id) const { return graph->fromId(id, Item());}
    private:
      const Graph* graph;
    };

    /// \brief Gives back the inverse of the map.
    ///
    /// Gives back the inverse of the IdMap.
    InverseMap inverse() const { return InverseMap(*graph);} 

  };

  
  /// \brief General invertable graph-map type.

  /// This type provides simple invertable graph-maps. 
  /// The InvertableMap wraps an arbitrary ReadWriteMap 
  /// and if a key is set to a new value then store it
  /// in the inverse map.
  /// \param _Graph The graph type.
  /// \param _Map The map to extend with invertable functionality. 
  template <
    typename _Graph,
    typename _Item, 
    typename _Value,
    typename _Map 
    = typename ItemSetTraits<_Graph, _Item>::template Map<_Value>::Parent 
  >
  class InvertableMap : protected _Map {

  public:
 
    typedef _Map Map;
    typedef _Graph Graph;

    /// The key type of InvertableMap (Node, Edge, UndirEdge).
    typedef typename _Map::Key Key;
    /// The value type of the InvertableMap.
    typedef typename _Map::Value Value;

    /// \brief Constructor.
    ///
    /// Construct a new InvertableMap for the graph.
    ///
    InvertableMap(const Graph& graph) : Map(graph) {} 
    
    /// \brief The setter function of the map.
    ///
    /// Sets the mapped value.
    void set(const Key& key, const Value& val) {
      Value oldval = Map::operator[](key);
      typename Container::iterator it = invMap.find(oldval);
      if (it != invMap.end() && it->second == key) {
	invMap.erase(it);
      }      
      invMap.insert(make_pair(val, key));
      Map::set(key, val);
    }

    /// \brief The getter function of the map.
    ///
    /// It gives back the value associated with the key.
    const Value operator[](const Key& key) const {
      return Map::operator[](key);
    }

  protected:

    /// \brief Add a new key to the map.
    ///
    /// Add a new key to the map. It is called by the
    /// \c AlterationNotifier.
    virtual void add(const Key& key) {
      Map::add(key);
    }

    /// \brief Erase the key from the map.
    ///
    /// Erase the key to the map. It is called by the
    /// \c AlterationNotifier.
    virtual void erase(const Key& key) {
      Value val = Map::operator[](key);
      typename Container::iterator it = invMap.find(val);
      if (it != invMap.end() && it->second == key) {
	invMap.erase(it);
      }
      Map::erase(key);
    }

    /// \brief Clear the keys from the map and inverse map.
    ///
    /// Clear the keys from the map and inverse map. It is called by the
    /// \c AlterationNotifier.
    virtual void clear() {
      invMap.clear();
      Map::clear();
    }

  private:
    
    typedef std::map<Value, Key> Container;
    Container invMap;    

  public:

    /// \brief The inverse map type.
    ///
    /// The inverse of this map. The subscript operator of the map
    /// gives back always the item what was last assigned to the value. 
    class InverseMap {
    public:
      /// \brief Constructor of the InverseMap.
      ///
      /// Constructor of the InverseMap.
      InverseMap(const InvertableMap& _inverted) : inverted(_inverted) {}

      /// The value type of the InverseMap.
      typedef typename InvertableMap::Key Value;
      /// The key type of the InverseMap.
      typedef typename InvertableMap::Value Key; 

      /// \brief Subscript operator. 
      ///
      /// Subscript operator. It gives back always the item 
      /// what was last assigned to the value.
      Value operator[](const Key& key) const {
	typename Container::const_iterator it = inverted.invMap.find(key);
	return it->second;
      }
      
    private:
      const InvertableMap& inverted;
    };

    /// \brief It gives back the just readeable inverse map.
    ///
    /// It gives back the just readeable inverse map.
    InverseMap inverse() const {
      return InverseMap(*this);
    } 


    
  };

  /// \brief Provides a mutable, continuous and unique descriptor for each 
  /// item in the graph.
  ///
  /// The DescriptorMap class provides a unique and continuous (but mutable)
  /// descriptor (id) for each item of the same type (e.g. node) in the
  /// graph. This id is <ul><li>\b unique: different items (nodes) get
  /// different ids <li>\b continuous: the range of the ids is the set of
  /// integers between 0 and \c n-1, where \c n is the number of the items of
  /// this type (e.g. nodes) (so the id of a node can change if you delete an
  /// other node, i.e. this id is mutable).  </ul> This map can be inverted
  /// with its member class \c InverseMap.
  ///
  /// \param _Graph The graph class the \c DescriptorMap belongs to.
  /// \param _Item The Item is the Key of the Map. It may be Node, Edge or 
  /// UndirEdge.
  /// \param _Map A ReadWriteMap mapping from the item type to integer.
  template <
    typename _Graph,   
    typename _Item,
    typename _Map 
    = typename ItemSetTraits<_Graph, _Item>::template Map<int>::Parent
  >
  class DescriptorMap : protected _Map {

    typedef _Item Item;
    typedef _Map Map;

  public:
    /// The graph class of DescriptorMap.
    typedef _Graph Graph;

    /// The key type of DescriptorMap (Node, Edge, UndirEdge).
    typedef typename _Map::Key Key;
    /// The value type of DescriptorMap.
    typedef typename _Map::Value Value;

    /// \brief Constructor.
    ///
    /// Constructor for descriptor map.
    DescriptorMap(const Graph& _graph) : Map(_graph) {
      build();
    }

  protected:

    /// \brief Add a new key to the map.
    ///
    /// Add a new key to the map. It is called by the
    /// \c AlterationNotifier.
    virtual void add(const Item& item) {
      Map::add(item);
      Map::set(item, invMap.size());
      invMap.push_back(item);
    }

    /// \brief Erase the key from the map.
    ///
    /// Erase the key to the map. It is called by the
    /// \c AlterationNotifier.
    virtual void erase(const Item& item) {
      Map::set(invMap.back(), Map::operator[](item));
      invMap[Map::operator[](item)] = invMap.back();
      invMap.pop_back();
      Map::erase(item);
    }

    /// \brief Build the unique map.
    ///
    /// Build the unique map. It is called by the
    /// \c AlterationNotifier.
    virtual void build() {
      Map::build();
      Item it;
      const typename Map::Graph* graph = Map::getGraph(); 
      for (graph->first(it); it != INVALID; graph->next(it)) {
	Map::set(it, invMap.size());
	invMap.push_back(it);	
      }      
    }
    
    /// \brief Clear the keys from the map.
    ///
    /// Clear the keys from the map. It is called by the
    /// \c AlterationNotifier.
    virtual void clear() {
      invMap.clear();
      Map::clear();
    }

  public:

    /// \brief Swaps the position of the two items in the map.
    ///
    /// Swaps the position of the two items in the map.
    void swap(const Item& p, const Item& q) {
      int pi = Map::operator[](p);
      int qi = Map::operator[](q);
      Map::set(p, qi);
      invMap[qi] = p;
      Map::set(q, pi);
      invMap[pi] = q;
    }

    /// \brief Gives back the \e descriptor of the item.
    ///
    /// Gives back the mutable and unique \e descriptor of the map.
    int operator[](const Item& item) const {
      return Map::operator[](item);
    }
    
  private:

    typedef std::vector<Item> Container;
    Container invMap;

  public:
    /// \brief The inverse map type of DescriptorMap.
    ///
    /// The inverse map type of DescriptorMap.
    class InverseMap {
    public:
      /// \brief Constructor of the InverseMap.
      ///
      /// Constructor of the InverseMap.
      InverseMap(const DescriptorMap& _inverted) 
	: inverted(_inverted) {}


      /// The value type of the InverseMap.
      typedef typename DescriptorMap::Key Value;
      /// The key type of the InverseMap.
      typedef typename DescriptorMap::Value Key; 

      /// \brief Subscript operator. 
      ///
      /// Subscript operator. It gives back the item 
      /// that the descriptor belongs to currently.
      Value operator[](const Key& key) const {
	return inverted.invMap[key];
      }

      /// \brief Size of the map.
      ///
      /// Returns the size of the map.
      int size() const {
	return inverted.invMap.size();
      }
      
    private:
      const DescriptorMap& inverted;
    };

    /// \brief Gives back the inverse of the map.
    ///
    /// Gives back the inverse of the map.
    const InverseMap inverse() const {
      return InverseMap(*this);
    }
  };

  /// \brief Returns the source of the given edge.
  ///
  /// The SourceMap gives back the source Node of the given edge. 
  /// \author Balazs Dezso
  template <typename Graph>
  class SourceMap {
  public:

    typedef True NeedCopy;

    typedef typename Graph::Node Value;
    typedef typename Graph::Edge Key;

    /// \brief Constructor
    ///
    /// Constructor
    /// \param _graph The graph that the map belongs to.
    SourceMap(const Graph& _graph) : graph(_graph) {}

    /// \brief The subscript operator.
    ///
    /// The subscript operator.
    /// \param edge The edge 
    /// \return The source of the edge 
    Value operator[](const Key& edge) {
      return graph.source(edge);
    }

  private:
    const Graph& graph;
  };

  /// \brief Returns a \ref SourceMap class
  ///
  /// This function just returns an \ref SourceMap class.
  /// \relates SourceMap
  template <typename Graph>
  inline SourceMap<Graph> sourceMap(const Graph& graph) {
    return SourceMap<Graph>(graph);
  } 

  /// \brief Returns the target of the given edge.
  ///
  /// The TargetMap gives back the target Node of the given edge. 
  /// \author Balazs Dezso
  template <typename Graph>
  class TargetMap {
  public:

    typedef True NeedCopy;

    typedef typename Graph::Node Value;
    typedef typename Graph::Edge Key;

    /// \brief Constructor
    ///
    /// Constructor
    /// \param _graph The graph that the map belongs to.
    TargetMap(const Graph& _graph) : graph(_graph) {}

    /// \brief The subscript operator.
    ///
    /// The subscript operator.
    /// \param e The edge 
    /// \return The target of the edge 
    Value operator[](const Key& e) {
      return graph.target(e);
    }

  private:
    const Graph& graph;
  };

  /// \brief Returns a \ref TargetMap class
  ///
  /// This function just returns a \ref TargetMap class.
  /// \relates TargetMap
  template <typename Graph>
  inline TargetMap<Graph> targetMap(const Graph& graph) {
    return TargetMap<Graph>(graph);
  }

  /// \brief Returns the "forward" directed edge view of an undirected edge.
  ///
  /// Returns the "forward" directed edge view of an undirected edge.
  /// \author Balazs Dezso
  template <typename Graph>
  class ForwardMap {
  public:

    typedef True NeedCopy;

    typedef typename Graph::Edge Value;
    typedef typename Graph::UndirEdge Key;

    /// \brief Constructor
    ///
    /// Constructor
    /// \param _graph The graph that the map belongs to.
    ForwardMap(const Graph& _graph) : graph(_graph) {}

    /// \brief The subscript operator.
    ///
    /// The subscript operator.
    /// \param key An undirected edge 
    /// \return The "forward" directed edge view of undirected edge 
    Value operator[](const Key& key) const {
      return graph.edgeWithSource(key, graph.source(key));
    }

  private:
    const Graph& graph;
  };

  /// \brief Returns a \ref ForwardMap class
  ///
  /// This function just returns an \ref ForwardMap class.
  /// \relates ForwardMap
  template <typename Graph>
  inline ForwardMap<Graph> forwardMap(const Graph& graph) {
    return ForwardMap<Graph>(graph);
  }

  /// \brief Returns the "backward" directed edge view of an undirected edge.
  ///
  /// Returns the "backward" directed edge view of an undirected edge.
  /// \author Balazs Dezso
  template <typename Graph>
  class BackwardMap {
  public:
    typedef True NeedCopy;

    typedef typename Graph::Edge Value;
    typedef typename Graph::UndirEdge Key;

    /// \brief Constructor
    ///
    /// Constructor
    /// \param _graph The graph that the map belongs to.
    BackwardMap(const Graph& _graph) : graph(_graph) {}

    /// \brief The subscript operator.
    ///
    /// The subscript operator.
    /// \param key An undirected edge 
    /// \return The "backward" directed edge view of undirected edge 
    Value operator[](const Key& key) const {
      return graph.edgeWithSource(key, graph.target(key));
    }

  private:
    const Graph& graph;
  };

  /// \brief Returns a \ref BackwardMap class

  /// This function just returns a \ref BackwardMap class.
  /// \relates BackwardMap
  template <typename Graph>
  inline BackwardMap<Graph> backwardMap(const Graph& graph) {
    return BackwardMap<Graph>(graph);
  }

  template <typename _Graph>
  class DegMapBase {
  public:
    
    typedef _Graph Graph;

  protected:

    typedef typename Graph::template NodeMap<int> IntNodeMap;
    
  };

  /// \brief Map of the node in-degrees.
  ///
  /// This map returns the in-degree of a node. Once it is constructed,
  /// the degrees are stored in a standard NodeMap, so each query is done
  /// in constant time. On the other hand, the values are updated automatically
  /// whenever the graph changes.
  ///
  /// \sa OutDegMap

  template <typename _Graph>
  class InDegMap  
    : protected AlterationNotifier<typename _Graph::Edge>::ObserverBase {

  public:
    
    typedef _Graph Graph;
    typedef int Value;
    typedef typename Graph::Node Key;

  private:

    class AutoNodeMap : public Graph::template NodeMap<int> {
    public:

      typedef typename Graph::template NodeMap<int> Parent;

      typedef typename Parent::Key Key;
      typedef typename Parent::Value Value;
      
      AutoNodeMap(const Graph& graph) : Parent(graph, 0) {}
      
      void add(const Key& key) {
	Parent::add(key);
	Parent::set(key, 0);
      }
    };

  public:

    /// \brief Constructor.
    ///
    /// Constructor for creating in-degree map.
    InDegMap(const Graph& _graph) : graph(_graph), deg(_graph) {
      AlterationNotifier<typename _Graph::Edge>
	::ObserverBase::attach(graph.getNotifier(typename _Graph::Edge()));
      
      for(typename _Graph::NodeIt it(graph); it != INVALID; ++it) {
	deg[it] = countInEdges(graph, it);
      }
    }

    virtual ~InDegMap() {
      AlterationNotifier<typename _Graph::Edge>::
	ObserverBase::detach();
    }
    
    /// Gives back the in-degree of a Node.
    int operator[](const Key& key) const {
      return deg[key];
    }

  protected:
    
    typedef typename Graph::Edge Edge;

    virtual void add(const Edge& edge) {
      ++deg[graph.target(edge)];
    }

    virtual void erase(const Edge& edge) {
      --deg[graph.target(edge)];
    }


    virtual void build() {
      for(typename _Graph::NodeIt it(graph); it != INVALID; ++it) {
	deg[it] = countInEdges(graph, it);
      }      
    }

    virtual void clear() {
      for(typename _Graph::NodeIt it(graph); it != INVALID; ++it) {
	deg[it] = 0;
      }
    }
  private:
    
    const _Graph& graph;
    AutoNodeMap deg;
  };


  /// \brief Map of the node out-degrees.
  ///
  /// This map returns the out-degree of a node. Once it is constructed,
  /// the degrees are stored in a standard NodeMap, so each query is done
  /// in constant time. On the other hand, the values are updated automatically
  /// whenever the graph changes.
  ///
  /// \sa InDegMap

  template <typename _Graph>
  class OutDegMap  
    : protected AlterationNotifier<typename _Graph::Edge>::ObserverBase {

  public:
    
    typedef _Graph Graph;
    typedef int Value;
    typedef typename Graph::Node Key;

  private:

    class AutoNodeMap : public Graph::template NodeMap<int> {
    public:

      typedef typename Graph::template NodeMap<int> Parent;

      typedef typename Parent::Key Key;
      typedef typename Parent::Value Value;
      
      AutoNodeMap(const Graph& graph) : Parent(graph, 0) {}
      
      void add(const Key& key) {
	Parent::add(key);
	Parent::set(key, 0);
      }
    };

  public:

    /// \brief Constructor.
    ///
    /// Constructor for creating out-degree map.
    OutDegMap(const Graph& _graph) : graph(_graph), deg(_graph) {
      AlterationNotifier<typename _Graph::Edge>
	::ObserverBase::attach(graph.getNotifier(typename _Graph::Edge()));
      
      for(typename _Graph::NodeIt it(graph); it != INVALID; ++it) {
	deg[it] = countOutEdges(graph, it);
      }
    }

    virtual ~OutDegMap() {
      AlterationNotifier<typename _Graph::Edge>::
	ObserverBase::detach();
    }
    
    /// Gives back the in-degree of a Node.
    int operator[](const Key& key) const {
      return deg[key];
    }

  protected:
    
    typedef typename Graph::Edge Edge;

    virtual void add(const Edge& edge) {
      ++deg[graph.source(edge)];
    }

    virtual void erase(const Edge& edge) {
      --deg[graph.source(edge)];
    }


    virtual void build() {
      for(typename _Graph::NodeIt it(graph); it != INVALID; ++it) {
	deg[it] = countOutEdges(graph, it);
      }      
    }

    virtual void clear() {
      for(typename _Graph::NodeIt it(graph); it != INVALID; ++it) {
	deg[it] = 0;
      }
    }
  private:
    
    const _Graph& graph;
    AutoNodeMap deg;
  };

  /// @}

} //END OF NAMESPACE LEMON

#endif
