/* -*- C++ -*-
 * src/hugo/skeletons/graph.h - Part of HUGOlib, a generic C++ optimization library
 *
 * Copyright (C) 2004 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef HUGO_SKELETON_GRAPH_H
#define HUGO_SKELETON_GRAPH_H

///\ingroup skeletons
///\file
///\brief Declaration of Graph.

#include <hugo/invalid.h>
#include <hugo/skeletons/maps.h>

namespace hugo {
  namespace skeleton {
    
    /// \addtogroup skeletons
    /// @{

    /// An empty static graph class.
  
    /// This class provides all the common features of a graph structure,
    /// however completely without implementations and real data structures
    /// behind the interface.
    /// All graph algorithms should compile with this class, but it will not
    /// run properly, of course.
    ///
    /// It can be used for checking the interface compatibility,
    /// or it can serve as a skeleton of a new graph structure.
    /// 
    /// Also, you will find here the full documentation of a certain graph
    /// feature, the documentation of a real graph imlementation
    /// like @ref ListGraph or
    /// @ref SmartGraph will just refer to this structure.
    class StaticGraph
    {
    public:
      /// Defalult constructor.

      /// Defalult constructor.
      ///
      StaticGraph() { }
      ///Copy consructor.

//       ///\todo It is not clear, what we expect from a copy constructor.
//       ///E.g. How to assign the nodes/edges to each other? What about maps?
//       StaticGraph(const StaticGraph& g) { }

      /// The base type of node iterators, 
      /// or in other words, the trivial node iterator.

      /// This is the base type of each node iterator,
      /// thus each kind of node iterator converts to this.
      /// More precisely each kind of node iterator should be inherited 
      /// from the trivial node iterator.
      class Node {
      public:
	/// Default constructor

	/// @warning The default constructor sets the iterator
	/// to an undefined value.
	Node() { }
	/// Copy constructor.

	/// Copy constructor.
	///
	Node(const Node&) { }

	/// Invalid constructor \& conversion.

	/// This constructor initializes the iterator to be invalid.
	/// \sa Invalid for more details.
	Node(Invalid) { }
	/// Equality operator

	/// Two iterators are equal if and only if they point to the
	/// same object or both are invalid.
	bool operator==(Node) const { return true; }

	/// Inequality operator
	
	/// \sa \ref operator==(Node n)
	///
	bool operator!=(Node) const { return true; }

 	///Comparison operator.

	///This is a strict ordering between the nodes.
	///
	///This ordering can be different from the order in which NodeIt
	///goes through the nodes.
	///\todo Possibly we don't need it.
	bool operator<(Node) const { return true; }
      };
    
      /// This iterator goes through each node.

      /// This iterator goes through each node.
      /// Its usage is quite simple, for example you can count the number
      /// of nodes in graph \c g of type \c Graph like this:
      /// \code
      /// int count=0;
      /// for (Graph::NodeIt n(g); n!=INVALID; ++n) ++count;
      /// \endcode
      class NodeIt : public Node {
      public:
	/// Default constructor

	/// @warning The default constructor sets the iterator
	/// to an undefined value.
	NodeIt() { }
	/// Copy constructor.
	
	/// Copy constructor.
	///
	NodeIt(const NodeIt&) { }
	/// Invalid constructor \& conversion.

	/// Initialize the iterator to be invalid.
	/// \sa Invalid for more details.
	NodeIt(Invalid) { }
	/// Sets the iterator to the first node.

	/// Sets the iterator to the first node of \c g.
	///
	NodeIt(const StaticGraph& g) { }
	/// Node -> NodeIt conversion.

	/// Sets the iterator to the node of \c g pointed by the trivial 
	/// iterator n.
	/// This feature necessitates that each time we 
	/// iterate the edge-set, the iteration order is the same.
	NodeIt(const StaticGraph& g, const Node& n) { }
	/// Next node.

	/// Assign the iterator to the next node.
	///
	NodeIt& operator++() { return *this; }
      };
    
    
      /// The base type of the edge iterators.

      /// The base type of the edge iterators.
      ///
      class Edge {
      public:
	/// Default constructor

	/// @warning The default constructor sets the iterator
	/// to an undefined value.
	Edge() { }
	/// Copy constructor.

	/// Copy constructor.
	///
	Edge(const Edge&) { }
	/// Initialize the iterator to be invalid.

	/// Initialize the iterator to be invalid.
	///
	Edge(Invalid) { }
	/// Equality operator

	/// Two iterators are equal if and only if they point to the
	/// same object or both are invalid.
	bool operator==(Edge) const { return true; }
	/// Inequality operator

	/// \sa \ref operator==(Node n)
	///
	bool operator!=(Edge) const { return true; }
 	///Comparison operator.

	///This is a strict ordering between the nodes.
	///
	///This ordering can be different from the order in which NodeIt
	///goes through the nodes.
	///\todo Possibly we don't need it.
 	bool operator<(Edge) const { return true; }
      };
    
      /// This iterator goes trough the outgoing edges of a node.

      /// This iterator goes trough the \e outgoing edges of a certain node
      /// of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of outgoing edges of a node \c n
      /// in graph \c g of type \c Graph as follows.
      /// \code
      /// int count=0;
      /// for (Graph::OutEdgeIt e(g, n); e!=INVALID; ++e) ++count;
      /// \endcode
    
      class OutEdgeIt : public Edge {
      public:
	/// Default constructor

	/// @warning The default constructor sets the iterator
	/// to an undefined value.
	OutEdgeIt() { }
	/// Copy constructor.

	/// Copy constructor.
	///
	OutEdgeIt(const OutEdgeIt&) { }
	/// Initialize the iterator to be invalid.

	/// Initialize the iterator to be invalid.
	///
	OutEdgeIt(Invalid) { }
	/// This constructor sets the iterator to first outgoing edge.
    
	/// This constructor set the iterator to the first outgoing edge of
	/// node
	///@param n the node
	///@param g the graph
	OutEdgeIt(const StaticGraph& g, const Node& n) { }
	/// Edge -> OutEdgeIt conversion

	/// Sets the iterator to the value of the trivial iterator \c e.
	/// This feature necessitates that each time we 
	/// iterate the edge-set, the iteration order is the same.
	OutEdgeIt(const StaticGraph& g, const Edge& e) { }
	///Next outgoing edge
	
	/// Assign the iterator to the next 
	/// outgoing edge of the corresponding node.
	OutEdgeIt& operator++() { return *this; }
      };

      /// This iterator goes trough the incoming edges of a node.

      /// This iterator goes trough the \e incoming edges of a certain node
      /// of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of outgoing edges of a node \c n
      /// in graph \c g of type \c Graph as follows.
      /// \code
      /// int count=0;
      /// for(Graph::InEdgeIt e(g, n); e!=INVALID; ++e) ++count;
      /// \endcode

      class InEdgeIt : public Edge {
      public:
	/// Default constructor

	/// @warning The default constructor sets the iterator
	/// to an undefined value.
	InEdgeIt() { }
	/// Copy constructor.

	/// Copy constructor.
	///
	InEdgeIt(const InEdgeIt&) { }
	/// Initialize the iterator to be invalid.

	/// Initialize the iterator to be invalid.
	///
	InEdgeIt(Invalid) { }
	/// This constructor sets the iterator to first incoming edge.
    
	/// This constructor set the iterator to the first incoming edge of
	/// node
	///@param n the node
	///@param g the graph
	InEdgeIt(const StaticGraph& g, const Node& n) { }
	/// Edge -> InEdgeIt conversion

	/// Sets the iterator to the value of the trivial iterator \c e.
	/// This feature necessitates that each time we 
	/// iterate the edge-set, the iteration order is the same.
	InEdgeIt(const StaticGraph& g, const Edge& n) { }
	/// Next incoming edge

	/// Assign the iterator to the next inedge of the corresponding node.
	///
	InEdgeIt& operator++() { return *this; }
      };
      /// This iterator goes through each edge.

      /// This iterator goes through each edge of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of edges in a graph \c g of type \c Graph as follows:
      /// \code
      /// int count=0;
      /// for(Graph::EdgeIt e(g); e!=INVALID; ++e) ++count;
      /// \endcode
      class EdgeIt : public Edge {
      public:
	/// Default constructor

	/// @warning The default constructor sets the iterator
	/// to an undefined value.
	EdgeIt() { }
	/// Copy constructor.

	/// Copy constructor.
	///
	EdgeIt(const EdgeIt&) { }
	/// Initialize the iterator to be invalid.

	/// Initialize the iterator to be invalid.
	///
	EdgeIt(Invalid) { }
	/// This constructor sets the iterator to first edge.
    
	/// This constructor set the iterator to the first edge of
	/// node
	///@param g the graph
	EdgeIt(const StaticGraph& g) { }
	/// Edge -> EdgeIt conversion

	/// Sets the iterator to the value of the trivial iterator \c e.
	/// This feature necessitates that each time we 
	/// iterate the edge-set, the iteration order is the same.
	EdgeIt(const StaticGraph&, const Edge&) { } 
    	///Next edge
	
	/// Assign the iterator to the next 
	/// edge of the corresponding node.
	EdgeIt& operator++() { return *this; }
      };

      /// First node of the graph.

      /// \retval i the first node.
      /// \return the first node.
      ///
      NodeIt& first(NodeIt& i) const { return i; }

      /// The first incoming edge.

      /// The first incoming edge.
      ///
      InEdgeIt& first(InEdgeIt &i, Node) const { return i; }
      /// The first outgoing edge.

      /// The first outgoing edge.
      ///
      OutEdgeIt& first(OutEdgeIt& i, Node) const { return i; }
      /// The first edge of the Graph.

      /// The first edge of the Graph.
      ///
      EdgeIt& first(EdgeIt& i) const { return i; }

      ///Gives back the head node of an edge.

      ///Gives back the head node of an edge.
      ///
      Node head(Edge) const { return INVALID; }
      ///Gives back the tail node of an edge.

      ///Gives back the tail node of an edge.
      ///
      Node tail(Edge) const { return INVALID; }
  
      ///Gives back the \e id of a node.

      ///\warning Not all graph structures provide this feature.
      ///
      ///\todo Should each graph provide \c id?
      int id(const Node&) const { return 0; }
      ///Gives back the \e id of an edge.

      ///\warning Not all graph structures provide this feature.
      ///
      ///\todo Should each graph provide \c id?
      int id(const Edge&) const { return 0; }

      /// .
      
      ///\todo Should it be in the concept?
      ///
      int nodeNum() const { return 0; }
      /// .

      ///\todo Should it be in the concept?
      ///
      int edgeNum() const { return 0; }


      ///Reference map of the nodes to type \c T.

      /// \ingroup skeletons
      ///Reference map of the nodes to type \c T.
      /// \sa Reference
      /// \warning Making maps that can handle bool type (NodeMap<bool>)
      /// needs some extra attention!
      template<class T> class NodeMap : public ReferenceMap< Node, T >
      {
      public:

	/// .
	NodeMap(const StaticGraph&) { }
	/// .
	NodeMap(const StaticGraph&, T) { }

	///Copy constructor
	template<typename TT> NodeMap(const NodeMap<TT>&) { }
	///Assignment operator
	template<typename TT> NodeMap& operator=(const NodeMap<TT>&)
	{ return *this; }
      };

      ///Reference map of the edges to type \c T.

      /// \ingroup skeletons
      ///Reference map of the edges to type \c T.
      /// \sa Reference
      /// \warning Making maps that can handle bool type (EdgeMap<bool>)
      /// needs some extra attention!
      template<class T> class EdgeMap
	: public ReferenceMap<Edge,T>
      {
      public:

	/// .
	EdgeMap(const StaticGraph&) { }
	/// .
	EdgeMap(const StaticGraph&, T) { }
    
	///Copy constructor
	template<typename TT> EdgeMap(const EdgeMap<TT>&) { }
	///Assignment operator
	template<typename TT> EdgeMap &operator=(const EdgeMap<TT>&)
	{ return *this; }
      };
    };


  
    /// An empty non-static graph class.

    /// This class provides everything that \ref StaticGraph
    /// with additional functionality which enables to build a
    /// graph from scratch.
    class ExtendableGraph : public StaticGraph
    {
    public:
      /// Defalult constructor.

      /// Defalult constructor.
      ///
      ExtendableGraph() { }
      ///Add a new node to the graph.

      /// \return the new node.
      ///
      Node addNode() { return INVALID; }
      ///Add a new edge to the graph.

      ///Add a new edge to the graph with tail node \c t
      ///and head node \c h.
      ///\return the new edge.
      Edge addEdge(Node h, Node t) { return INVALID; }
    
      /// Resets the graph.

      /// This function deletes all edges and nodes of the graph.
      /// It also frees the memory allocated to store them.
      /// \todo It might belong to \ref ErasableGraph.
      void clear() { }
    };

    /// An empty erasable graph class.
  
    /// This class is an extension of \ref ExtendableGraph. It also makes it
    /// possible to erase edges or nodes.
    class ErasableGraph : public ExtendableGraph
    {
    public:
      /// Defalult constructor.

      /// Defalult constructor.
      ///
      ErasableGraph() { }
      /// Deletes a node.

      /// Deletes node \c n node.
      ///
      void erase(Node n) { }
      /// Deletes an edge.

      /// Deletes edge \c e edge.
      ///
      void erase(Edge e) { }
    };

    // @}
  } //namespace skeleton  
} //namespace hugo



#endif // HUGO_SKELETON_GRAPH_H
