/* -*- C++ -*-
 * src/hugo/default_map.h - Part of HUGOlib, a generic C++ optimization library
 *
 * Copyright (C) 2004 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef HUGO_DEFAULT_MAP_H
#define HUGO_DEFAULT_MAP_H


#include <hugo/array_map.h>
#include <hugo/vector_map.h>

///\ingroup graphmaps
///\file
///\brief Graph maps that construates and destruates
///their elements dynamically.

namespace hugo {

/// \addtogroup graphmaps
/// @{

  /** The ArrayMap template class is graph map structure what
   *  automatically updates the map when a key is added to or erased from
   *  the map. This map uses the VectorMap if the ValueType is a primitive
   *  type and the ArrayMap for the other cases.
   *
   *  The template parameter is the MapRegistry that the maps
   *  will belong to and the ValueType.
   */


  /** Macro to implement the DefaultMap.
   */
#define DEFAULT_MAP_BODY(DynMap, Value) \
{ \
\
public: \
\
typedef DynMap<MapRegistry, Value> Parent; \
\
typedef typename MapRegistry::Graph Graph; \
\
DefaultMap(const Graph& g, MapRegistry& r) : Parent(g, r) {} \
DefaultMap(const Graph& g, MapRegistry& r, const Value& v) \
  : Parent(g, r, v) {} \
DefaultMap(const DefaultMap& copy) \
  : Parent(static_cast<const Parent&>(copy)) {} \
template <typename TT> \
DefaultMap(const DefaultMap<MapRegistry, TT>& copy) \
  : { \
  Parent::MapBase::operator= \
    (static_cast<const typename Parent::MapBase&>(copy)); \
  if (Parent::getGraph()) { \
    for (typename Parent::KeyIt it(*Parent::getGraph()); it!=INVALID; ++it) {\
      Parent::add(it); \
      Parent::operator[](it) = copy[it]; \
    } \
  } \
} \
DefaultMap& operator=(const DefaultMap& copy) { \
  Parent::operator=(static_cast<const Parent&>(copy)); \
  return *this; \
} \
template <typename TT> \
DefaultMap& operator=(const DefaultMap<MapRegistry, TT>& copy) { \
  if (Parent::getGraph() != copy.getGraph()) { \
    Parent::clear(); \
    Parent::MapBase::operator=(copy); \
    Parent::construct(); \
  } \
  if (Parent::getGraph()) { \
    for (typename Parent::KeyIt it(*Parent::getGraph()); it!=INVALID; ++it) {\
      Parent::operator[](it) = copy[it]; \
    } \
  } \
  return *this; \
} \
};


  template <typename MapRegistry, typename Type>
  class DefaultMap : public ArrayMap<MapRegistry, Type> 
  DEFAULT_MAP_BODY(ArrayMap, Type);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, bool> 
    : public VectorMap<MapRegistry, bool> 
  DEFAULT_MAP_BODY(VectorMap, bool);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, char> 
    : public VectorMap<MapRegistry, char> 
  DEFAULT_MAP_BODY(VectorMap, char);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, int> 
    : public VectorMap<MapRegistry, int> 
  DEFAULT_MAP_BODY(VectorMap, int);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, short> 
    : public VectorMap<MapRegistry, short> 
  DEFAULT_MAP_BODY(VectorMap, short);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, long> 
    : public VectorMap<MapRegistry, long> 
  DEFAULT_MAP_BODY(VectorMap, long);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, float> 
    : public VectorMap<MapRegistry, float> 
  DEFAULT_MAP_BODY(VectorMap, float);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, double> 
    : public VectorMap<MapRegistry, double> 
  DEFAULT_MAP_BODY(VectorMap, double);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, long double> 
    : public VectorMap<MapRegistry, long double> 
  DEFAULT_MAP_BODY(VectorMap, long double);

  template <typename MapRegistry, typename Type>
  class DefaultMap<MapRegistry, Type*>
    : public VectorMap<MapRegistry, Type*> 
  DEFAULT_MAP_BODY(VectorMap, Type*);

}

#endif
