/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef LEMON_LIST_GRAPH_H
#define LEMON_LIST_GRAPH_H

///\ingroup graphs
///\file
///\brief ListGraph, ListUGraph classes.

#include <lemon/bits/graph_extender.h>

#include <lemon/error.h>

#include <vector>
#include <list>

namespace lemon {

  class ListGraphBase {

  protected:
    struct NodeT {
      int first_in, first_out;
      int prev, next;
    };
 
    struct EdgeT {
      int target, source;
      int prev_in, prev_out;
      int next_in, next_out;
    };

    std::vector<NodeT> nodes;

    int first_node;

    int first_free_node;

    std::vector<EdgeT> edges;

    int first_free_edge;
    
  public:
    
    typedef ListGraphBase Graph;
    
    class Node {
      friend class ListGraphBase;
    protected:

      int id;
      Node(int pid) { id = pid;}

    public:
      Node() {}
      Node (Invalid) { id = -1; }
      bool operator==(const Node& node) const {return id == node.id;}
      bool operator!=(const Node& node) const {return id != node.id;}
      bool operator<(const Node& node) const {return id < node.id;}
    };

    class Edge {
      friend class ListGraphBase;
    protected:

      int id;
      Edge(int pid) { id = pid;}

    public:
      Edge() {}
      Edge (Invalid) { id = -1; }
      bool operator==(const Edge& edge) const {return id == edge.id;}
      bool operator!=(const Edge& edge) const {return id != edge.id;}
      bool operator<(const Edge& edge) const {return id < edge.id;}
    };



    ListGraphBase()
      : nodes(), first_node(-1),
	first_free_node(-1), edges(), first_free_edge(-1) {}

    
    /// Maximum node ID.
    
    /// Maximum node ID.
    ///\sa id(Node)
    int maxNodeId() const { return nodes.size()-1; } 

    /// Maximum edge ID.
    
    /// Maximum edge ID.
    ///\sa id(Edge)
    int maxEdgeId() const { return edges.size()-1; }

    Node source(Edge e) const { return edges[e.id].source; }
    Node target(Edge e) const { return edges[e.id].target; }


    void first(Node& node) const { 
      node.id = first_node;
    }

    void next(Node& node) const {
      node.id = nodes[node.id].next;
    }


    void first(Edge& e) const { 
      int n;
      for(n = first_node; 
	  n!=-1 && nodes[n].first_in == -1; 
	  n = nodes[n].next);
      e.id = (n == -1) ? -1 : nodes[n].first_in;
    }

    void next(Edge& edge) const {
      if (edges[edge.id].next_in != -1) {
	edge.id = edges[edge.id].next_in;
      } else {
	int n;
	for(n = nodes[edges[edge.id].target].next;
	  n!=-1 && nodes[n].first_in == -1; 
	  n = nodes[n].next);
	edge.id = (n == -1) ? -1 : nodes[n].first_in;
      }      
    }

    void firstOut(Edge &e, const Node& v) const {
      e.id = nodes[v.id].first_out;
    }
    void nextOut(Edge &e) const {
      e.id=edges[e.id].next_out;
    }

    void firstIn(Edge &e, const Node& v) const {
      e.id = nodes[v.id].first_in;
    }
    void nextIn(Edge &e) const {
      e.id=edges[e.id].next_in;
    }

    
    static int id(Node v) { return v.id; }
    static int id(Edge e) { return e.id; }

    static Node nodeFromId(int id) { return Node(id);}
    static Edge edgeFromId(int id) { return Edge(id);}

    /// Adds a new node to the graph.

    /// \warning It adds the new node to the front of the list.
    /// (i.e. the lastly added node becomes the first.)
    Node addNode() {     
      int n;
      
      if(first_free_node==-1) {
	n = nodes.size();
	nodes.push_back(NodeT());
      } else {
	n = first_free_node;
	first_free_node = nodes[n].next;
      }
      
      nodes[n].next = first_node;
      if(first_node != -1) nodes[first_node].prev = n;
      first_node = n;
      nodes[n].prev = -1;
      
      nodes[n].first_in = nodes[n].first_out = -1;
      
      return Node(n);
    }
    
    Edge addEdge(Node u, Node v) {
      int n;      

      if (first_free_edge == -1) {
	n = edges.size();
	edges.push_back(EdgeT());
      } else {
	n = first_free_edge;
	first_free_edge = edges[n].next_in;
      }
      
      edges[n].source = u.id; 
      edges[n].target = v.id;

      edges[n].next_out = nodes[u.id].first_out;
      if(nodes[u.id].first_out != -1) {
	edges[nodes[u.id].first_out].prev_out = n;
      }
      
      edges[n].next_in = nodes[v.id].first_in;
      if(nodes[v.id].first_in != -1) {
	edges[nodes[v.id].first_in].prev_in = n;
      }
      
      edges[n].prev_in = edges[n].prev_out = -1;
	
      nodes[u.id].first_out = nodes[v.id].first_in = n;

      return Edge(n);
    }
    
    void erase(const Node& node) {
      int n = node.id;
      
      if(nodes[n].next != -1) {
	nodes[nodes[n].next].prev = nodes[n].prev;
      }
      
      if(nodes[n].prev != -1) {
	nodes[nodes[n].prev].next = nodes[n].next;
      } else {
	first_node = nodes[n].next;
      }
      
      nodes[n].next = first_free_node;
      first_free_node = n;

    }
    
    void erase(const Edge& edge) {
      int n = edge.id;
      
      if(edges[n].next_in!=-1) {
	edges[edges[n].next_in].prev_in = edges[n].prev_in;
      }

      if(edges[n].prev_in!=-1) {
	edges[edges[n].prev_in].next_in = edges[n].next_in;
      } else {
	nodes[edges[n].target].first_in = edges[n].next_in;
      }

      
      if(edges[n].next_out!=-1) {
	edges[edges[n].next_out].prev_out = edges[n].prev_out;
      } 

      if(edges[n].prev_out!=-1) {
	edges[edges[n].prev_out].next_out = edges[n].next_out;
      } else {
	nodes[edges[n].source].first_out = edges[n].next_out;
      }
      
      edges[n].next_in = first_free_edge;
      first_free_edge = n;      

    }

    void clear() {
      edges.clear();
      nodes.clear();
      first_node = first_free_node = first_free_edge = -1;
    }

  protected:
    void _changeTarget(Edge e, Node n) 
    {
      if(edges[e.id].next_in != -1)
	edges[edges[e.id].next_in].prev_in = edges[e.id].prev_in;
      if(edges[e.id].prev_in != -1)
	edges[edges[e.id].prev_in].next_in = edges[e.id].next_in;
      else nodes[edges[e.id].target].first_in = edges[e.id].next_in;
      if (nodes[n.id].first_in != -1) {
	edges[nodes[n.id].first_in].prev_in = e.id;
      }
      edges[e.id].target = n.id;
      edges[e.id].prev_in = -1;
      edges[e.id].next_in = nodes[n.id].first_in;
      nodes[n.id].first_in = e.id;
    }
    void _changeSource(Edge e, Node n) 
    {
      if(edges[e.id].next_out != -1)
	edges[edges[e.id].next_out].prev_out = edges[e.id].prev_out;
      if(edges[e.id].prev_out != -1)
	edges[edges[e.id].prev_out].next_out = edges[e.id].next_out;
      else nodes[edges[e.id].source].first_out = edges[e.id].next_out;
      if (nodes[n.id].first_out != -1) {
	edges[nodes[n.id].first_out].prev_out = e.id;
      }
      edges[e.id].source = n.id;
      edges[e.id].prev_out = -1;
      edges[e.id].next_out = nodes[n.id].first_out;
      nodes[n.id].first_out = e.id;
    }

  };

  typedef GraphExtender<ListGraphBase> ExtendedListGraphBase;

  /// \addtogroup graphs
  /// @{

  ///A list graph class.

  ///This is a simple and fast erasable graph implementation.
  ///
  ///It addition that it conforms to the
  ///\ref concept::ErasableGraph "ErasableGraph" concept,
  ///it also provides several additional useful extra functionalities.
  ///\sa concept::ErasableGraph.

  class ListGraph : public ExtendedListGraphBase 
  {
  public:
    /// Changes the target of \c e to \c n

    /// Changes the target of \c e to \c n
    ///
    ///\note The <tt>Edge</tt>'s and <tt>OutEdge</tt>'s
    ///referencing the changed edge remain
    ///valid. However <tt>InEdge</tt>'s are invalidated.
    void changeTarget(Edge e, Node n) { 
      _changeTarget(e,n); 
    }
    /// Changes the source of \c e to \c n

    /// Changes the source of \c e to \c n
    ///
    ///\note The <tt>Edge</tt>'s and <tt>InEdge</tt>'s
    ///referencing the changed edge remain
    ///valid. However <tt>OutEdge</tt>'s are invalidated.
    void changeSource(Edge e, Node n) { 
      _changeSource(e,n);
    }

    /// Invert the direction of an edge.

    ///\note The <tt>Edge</tt>'s
    ///referencing the changed edge remain
    ///valid. However <tt>OutEdge</tt>'s  and <tt>InEdge</tt>'s are invalidated.
    void reverseEdge(Edge e) {
      Node t=target(e);
      _changeTarget(e,source(e));
      _changeSource(e,t);
    }

    ///Using this it possible to avoid the superfluous memory allocation.

    ///Using this it possible to avoid the superfluous memory allocation.
    ///\todo more docs...
    void reserveEdge(int n) { edges.reserve(n); };

    ///Contract two nodes.

    ///This function contracts two nodes.
    ///
    ///Node \p b will be removed but instead of deleting
    ///its neighboring edges, they will be joined to \p a.
    ///The last parameter \p r controls whether to remove loops. \c true
    ///means that loops will be removed.
    ///
    ///\note The <tt>Edge</tt>s
    ///referencing a moved edge remain
    ///valid. However <tt>InEdge</tt>'s and <tt>OutEdge</tt>'s
    ///may be invalidated.
    void contract(Node a, Node b, bool r = true) 
    {
      for(OutEdgeIt e(*this,b);e!=INVALID;) {
	OutEdgeIt f=e;
	++f;
	if(r && target(e)==a) erase(e);
	else changeSource(e,a);
	e=f;
      }
      for(InEdgeIt e(*this,b);e!=INVALID;) {
	InEdgeIt f=e;
	++f;
	if(r && source(e)==a) erase(e);
	else changeTarget(e,a);
	e=f;
      }
      erase(b);
    }

    ///Split a node.

    ///This function splits a node. First a new node is added to the graph,
    ///then the source of each outgoing edge of \c n is moved to this new node.
    ///If \c connect is \c true (this is the default value), then a new edge
    ///from \c n to the newly created node is also added.
    ///\return The newly created node.
    ///
    ///\note The <tt>Edge</tt>s
    ///referencing a moved edge remain
    ///valid. However <tt>InEdge</tt>'s and <tt>OutEdge</tt>'s
    ///may be invalidated.
    ///\warning This functionality cannot be used together with the Snapshot
    ///feature.
    ///\todo It could be implemented in a bit faster way.
    Node split(Node n, bool connect = true) 
    {
      Node b = addNode();
      for(OutEdgeIt e(*this,n);e!=INVALID;) {
 	OutEdgeIt f=e;
	++f;
	changeSource(e,b);
	e=f;
      }
      if(connect) addEdge(n,b);
      return b;
    }
      
    ///Split an edge.

    ///This function splits an edge. First a new node \c b is added to the graph,
    ///then the original edge is re-targetes to \c b. Finally an edge
    ///from \c b to the original target is added.
    ///\return The newly created node.
    ///\warning This functionality cannot be used together with the Snapshot
    ///feature.
    Node split(Edge e) 
    {
      Node b = addNode();
      addEdge(b,target(e));
      changeTarget(e,b);
      return b;
    }
      
    ///Class to make a snapshot of the graph and to restrore to it later.

    ///Class to make a snapshot of the graph and to restrore to it later.
    ///
    ///The newly added nodes and edges can be removed using the
    ///restore() function.
    ///
    ///\warning Edge and node deletions cannot be restored.
    ///\warning Snapshots cannot be nested.
    class Snapshot : protected AlterationNotifier<Node>::ObserverBase,
		     protected AlterationNotifier<Edge>::ObserverBase
    {
    public:
      
      class UnsupportedOperation : public LogicError {
      public:
	virtual const char* exceptionName() const {
	  return "lemon::ListGraph::Snapshot::UnsupportedOperation";
	}
      };
            

      protected:
      
      ListGraph *g;
      std::list<Node> added_nodes;
      std::list<Edge> added_edges;
      
      bool active;
      virtual void add(const Node& n) {
	added_nodes.push_back(n);
      };
      virtual void erase(const Node&) 
      {
	throw UnsupportedOperation();
      }
      virtual void add(const Edge& n) {
	added_edges.push_back(n);
      };
      virtual void erase(const Edge&) 
      {
	throw UnsupportedOperation();
      }

      ///\bug What is this used for?
      ///
      virtual void build() {}
      ///\bug What is this used for?
      ///
      virtual void clear() {}

      void regist(ListGraph &_g) {
	g=&_g;
	AlterationNotifier<Node>::ObserverBase::
	  attach(g->getNotifier(Node()));
	AlterationNotifier<Edge>::ObserverBase::
	  attach(g->getNotifier(Edge()));
      }
            
      void deregist() {
	AlterationNotifier<Node>::ObserverBase::
	  detach();
	AlterationNotifier<Edge>::ObserverBase::
	  detach();
	g=0;
      }

    public:
      ///Default constructur.
      
      ///Default constructur.
      ///To actually make a snapshot you must call save().
      ///
      Snapshot() : g(0) {}
      ///Constructor that immediately makes a snapshot.
      
      ///This constructor immediately makes a snapshot of the graph.
      ///\param _g The graph we make a snapshot of.
      Snapshot(ListGraph &_g) {
	regist(_g);
      }
      ///\bug Is it necessary?
      ///
      ~Snapshot() 
      {
	if(g) deregist();
      }
      
      ///Make a snapshot.

      ///Make a snapshot of the graph.
      ///
      ///This function can be called more than once. In case of a repeated
      ///call, the previous snapshot gets lost.
      ///\param _g The graph we make the snapshot of.
      void save(ListGraph &_g) 
      {
	if(g!=&_g) {
	  if(g) deregist();
	  regist(_g);
	}
	added_nodes.clear();
	added_edges.clear();
      }
      
    ///Undo the changes until the last snapshot.

    ///Undo the changes until last snapshot created by save().
    ///
    ///\todo This function might be called undo().
      void restore() {
	ListGraph &old_g=*g;
	deregist();
	while(!added_edges.empty()) {
	  old_g.erase(added_edges.front());
	  added_edges.pop_front();
	}
 	while(!added_nodes.empty()) {
	  old_g.erase(added_nodes.front());
	  added_nodes.pop_front();
	}
      }
    };
    
  };

  ///@}

  /**************** Undirected List Graph ****************/

  typedef UGraphExtender<UGraphBaseExtender<
    ListGraphBase> > ExtendedListUGraphBase;

  /// \addtogroup graphs
  /// @{

  ///An undirected list graph class.

  ///This is a simple and fast erasable undirected graph implementation.
  ///
  ///It conforms to the
  ///\ref concept::UGraph "UGraph" concept.
  ///
  ///\sa concept::UGraph.
  ///
  ///\todo Snapshot, reverseEdge(), changeTarget(), changeSource(), contract()
  ///haven't been implemented yet.
  ///
  class ListUGraph : public ExtendedListUGraphBase {
  public:
    typedef ExtendedListUGraphBase Parent;
    /// \brief Changes the target of \c e to \c n
    ///
    /// Changes the target of \c e to \c n
    ///
    /// \note The <tt>Edge</tt>'s and <tt>OutEdge</tt>'s
    /// referencing the changed edge remain
    /// valid. However <tt>InEdge</tt>'s are invalidated.
    void changeTarget(UEdge e, Node n) { 
      _changeTarget(e,n); 
    }
    /// Changes the source of \c e to \c n
    ///
    /// Changes the source of \c e to \c n
    ///
    ///\note The <tt>Edge</tt>'s and <tt>InEdge</tt>'s
    ///referencing the changed edge remain
    ///valid. However <tt>OutEdge</tt>'s are invalidated.
    void changeSource(UEdge e, Node n) { 
      _changeSource(e,n); 
    }
    /// \brief Contract two nodes.
    ///
    /// This function contracts two nodes.
    ///
    /// Node \p b will be removed but instead of deleting
    /// its neighboring edges, they will be joined to \p a.
    /// The last parameter \p r controls whether to remove loops. \c true
    /// means that loops will be removed.
    ///
    /// \note The <tt>Edge</tt>s
    /// referencing a moved edge remain
    /// valid.
    void contract(Node a, Node b, bool r = true) {
      for(IncEdgeIt e(*this, b); e!=INVALID;) {
	IncEdgeIt f = e; ++f;
	if (r && runningNode(e) == a) {
	  erase(e);
	} else if (source(e) == b) {
	  changeSource(e, a);
	} else {
	  changeTarget(e, a);
	}
	e = f;
      }
      erase(b);
    }
  };


  class ListBpUGraphBase {
  public:

    class NodeSetError : public LogicError {
      virtual const char* exceptionName() const { 
	return "lemon::ListBpUGraph::NodeSetError";
      }
    };

  protected:

    struct NodeT {
      int first_edge, next_node;
    };

    struct EdgeT {
      int aNode, prev_out, next_out;
      int bNode, prev_in, next_in;
    };

    std::vector<NodeT> aNodes;
    std::vector<NodeT> bNodes;

    std::vector<EdgeT> edges;

    int first_anode;
    int first_free_anode;

    int first_bnode;
    int first_free_bnode;

    int first_free_edge;

  public:
  
    class Node {
      friend class ListBpUGraphBase;
    protected:
      int id;

      Node(int _id) : id(_id) {}
    public:
      Node() {}
      Node(Invalid) { id = -1; }
      bool operator==(const Node i) const {return id==i.id;}
      bool operator!=(const Node i) const {return id!=i.id;}
      bool operator<(const Node i) const {return id<i.id;}
    };

    class Edge {
      friend class ListBpUGraphBase;
    protected:
      int id;

      Edge(int _id) { id = _id;}
    public:
      Edge() {}
      Edge (Invalid) { id = -1; }
      bool operator==(const Edge i) const {return id==i.id;}
      bool operator!=(const Edge i) const {return id!=i.id;}
      bool operator<(const Edge i) const {return id<i.id;}
    };

    ListBpUGraphBase()
      : first_anode(-1), first_free_anode(-1),
        first_bnode(-1), first_free_bnode(-1),
        first_free_edge(-1) {}

    void firstANode(Node& node) const {
      node.id = first_anode != -1 ? (first_anode << 1) : -1;
    }
    void nextANode(Node& node) const {
      node.id = aNodes[node.id >> 1].next_node;
    }

    void firstBNode(Node& node) const {
      node.id =  first_bnode != -1 ? (first_bnode << 1) + 1 : -1;
    }
    void nextBNode(Node& node) const {
      node.id = bNodes[node.id >> 1].next_node;
    }

    void first(Node& node) const {
      if (first_anode != -1) {
        node.id = (first_anode << 1);
      } else if (first_bnode != -1) {
        node.id = (first_bnode << 1) + 1;
      } else {
        node.id = -1;
      }
    }
    void next(Node& node) const {
      if (aNode(node)) {
        node.id = aNodes[node.id >> 1].next_node;
        if (node.id == -1) {
          if (first_bnode != -1) {
            node.id = (first_bnode << 1) + 1;
          }
        }
      } else {
        node.id = bNodes[node.id >> 1].next_node;
      }
    }
  
    void first(Edge& edge) const {
      int aNodeId = first_anode;
      while (aNodeId != -1 && aNodes[aNodeId].first_edge == -1) {
        aNodeId = aNodes[aNodeId].next_node != -1 ? 
          aNodes[aNodeId].next_node >> 1 : -1;
      }
      if (aNodeId != -1) {
        edge.id = aNodes[aNodeId].first_edge;
      } else {
        edge.id = -1;
      }
    }
    void next(Edge& edge) const {
      int aNodeId = edges[edge.id].aNode >> 1;
      edge.id = edges[edge.id].next_out;
      if (edge.id == -1) {
        aNodeId = aNodes[aNodeId].next_node != -1 ? 
          aNodes[aNodeId].next_node >> 1 : -1;
        while (aNodeId != -1 && aNodes[aNodeId].first_edge == -1) {
          aNodeId = aNodes[aNodeId].next_node != -1 ? 
          aNodes[aNodeId].next_node >> 1 : -1;
        }
        if (aNodeId != -1) {
          edge.id = aNodes[aNodeId].first_edge;
        } else {
          edge.id = -1;
        }
      }
    }

    void firstOut(Edge& edge, const Node& node) const {
      LEMON_ASSERT((node.id & 1) == 0, NodeSetError());
      edge.id = aNodes[node.id >> 1].first_edge;
    }
    void nextOut(Edge& edge) const {
      edge.id = edges[edge.id].next_out;
    }

    void firstIn(Edge& edge, const Node& node) const {
      LEMON_ASSERT((node.id & 1) == 1, NodeSetError());
      edge.id = bNodes[node.id >> 1].first_edge;
    }
    void nextIn(Edge& edge) const {
      edge.id = edges[edge.id].next_in;
    }

    static int id(const Node& node) {
      return node.id;
    }
    static Node nodeFromId(int id) {
      return Node(id);
    }
    int maxNodeId() const {
      return aNodes.size() > bNodes.size() ?
	aNodes.size() * 2 - 2 : bNodes.size() * 2 - 1;
    }
  
    static int id(const Edge& edge) {
      return edge.id;
    }
    static Edge edgeFromId(int id) {
      return Edge(id);
    }
    int maxEdgeId() const {
      return edges.size();
    }
  
    static int aNodeId(const Node& node) {
      return node.id >> 1;
    }
    static Node fromANodeId(int id, Node) {
      return Node(id << 1);
    }
    int maxANodeId() const {
      return aNodes.size();
    }

    static int bNodeId(const Node& node) {
      return node.id >> 1;
    }
    static Node fromBNodeId(int id) {
      return Node((id << 1) + 1);
    }
    int maxBNodeId() const {
      return bNodes.size();
    }

    Node aNode(const Edge& edge) const {
      return Node(edges[edge.id].aNode);
    }
    Node bNode(const Edge& edge) const {
      return Node(edges[edge.id].bNode);
    }

    static bool aNode(const Node& node) {
      return (node.id & 1) == 0;
    }

    static bool bNode(const Node& node) {
      return (node.id & 1) == 1;
    }

    Node addANode() {
      int aNodeId;
      if (first_free_anode == -1) {
        aNodeId = aNodes.size();
        aNodes.push_back(NodeT());
      } else {
        aNodeId = first_free_anode;
        first_free_anode = aNodes[first_free_anode].next_node;
      }
      aNodes[aNodeId].next_node = 
        first_anode != -1 ? (first_anode << 1) : -1;
      first_anode = aNodeId;
      aNodes[aNodeId].first_edge = -1;
      return Node(aNodeId << 1);
    }

    Node addBNode() {
      int bNodeId;
      if (first_free_bnode == -1) {
        bNodeId = bNodes.size();
        bNodes.push_back(NodeT());
      } else {
        bNodeId = first_free_bnode;
        first_free_bnode = bNodes[first_free_bnode].next_node;
      }
      bNodes[bNodeId].next_node = 
        first_bnode != -1 ? (first_bnode << 1) + 1 : -1;
      first_bnode = bNodeId;
      bNodes[bNodeId].first_edge = -1;
      return Node((bNodeId << 1) + 1);
    }

    Edge addEdge(const Node& source, const Node& target) {
      LEMON_ASSERT(((source.id ^ target.id) & 1) == 1, NodeSetError());
      int edgeId;
      if (first_free_edge != -1) {
        edgeId = first_free_edge;
        first_free_edge = edges[edgeId].next_out;
      } else {
        edgeId = edges.size();
        edges.push_back(EdgeT());
      }
      if ((source.id & 1) == 0) {
	edges[edgeId].aNode = source.id;
	edges[edgeId].bNode = target.id;
      } else {
	edges[edgeId].aNode = target.id;
	edges[edgeId].bNode = source.id;
      }
      edges[edgeId].next_out = aNodes[edges[edgeId].aNode >> 1].first_edge;
      edges[edgeId].prev_out = -1;
      if (aNodes[edges[edgeId].aNode >> 1].first_edge != -1) {
        edges[aNodes[edges[edgeId].aNode >> 1].first_edge].prev_out = edgeId;
      }
      aNodes[edges[edgeId].aNode >> 1].first_edge = edgeId;
      edges[edgeId].next_in = bNodes[edges[edgeId].bNode >> 1].first_edge;
      edges[edgeId].prev_in = -1;
      if (bNodes[edges[edgeId].bNode >> 1].first_edge != -1) {
        edges[bNodes[edges[edgeId].bNode >> 1].first_edge].prev_in = edgeId;
      }
      bNodes[edges[edgeId].bNode >> 1].first_edge = edgeId;
      return Edge(edgeId);
    }

    void erase(const Node& node) {
      if (aNode(node)) {
        int aNodeId = node.id >> 1;
        aNodes[aNodeId].next_node = first_free_anode;
        first_free_anode = aNodeId;
      } else {
        int bNodeId = node.id >> 1;
        bNodes[bNodeId].next_node = first_free_bnode;
        first_free_bnode = bNodeId;
      }
    }

    void erase(const Edge& edge) {
      if (edges[edge.id].prev_out != -1) {
        edges[edges[edge.id].prev_out].next_out = edges[edge.id].next_out;
      } else {
        aNodes[edges[edge.id].aNode].first_edge = edges[edge.id].next_out;
      }
      if (edges[edge.id].next_out != -1) {
        edges[edges[edge.id].next_out].prev_out = edges[edge.id].prev_out;
      }
      if (edges[edge.id].prev_in != -1) {
        edges[edges[edge.id].prev_in].next_in = edges[edge.id].next_in;
      } else {
        bNodes[edges[edge.id].bNode].first_edge = edges[edge.id].next_in;
      }
      if (edges[edge.id].next_in != -1) {
        edges[edges[edge.id].next_in].prev_in = edges[edge.id].prev_in;
      }
      edges[edge.id].next_out = first_free_edge;
      first_free_edge = edge.id;
    }

    void clear() {
      aNodes.clear();
      bNodes.clear();
      edges.clear();
      first_anode = -1;
      first_free_anode = -1;
      first_bnode = -1;
      first_free_bnode = -1;
      first_free_edge = -1;
    }

  };


  typedef BpUGraphExtender< BpUGraphBaseExtender<
    ListBpUGraphBase> > ExtendedListBpUGraphBase;

  /// \ingroup graphs
  ///
  /// \brief A smart bipartite undirected graph class.
  ///
  /// This is a bipartite undirected graph implementation.
  /// Except from this it conforms to 
  /// the \ref concept::BpUGraph "BpUGraph" concept.
  /// \sa concept::BpUGraph.
  ///
  class ListBpUGraph : public ExtendedListBpUGraphBase {};

  
  /// @}  
} //namespace lemon
  

#endif
