// -*- c++ -*-
#ifndef HUGO_MINLENGTHPATHS_H
#define HUGO_MINLENGTHPATHS_H

///\ingroup flowalgs
///\file
///\brief An algorithm for finding k paths of minimal total length.


//#include <hugo/dijkstra.h>
//#include <hugo/graph_wrapper.h>
#include <hugo/maps.h>
#include <vector>
#include <hugo/mincostflows.h>

namespace hugo {

/// \addtogroup flowalgs
/// @{

  ///\brief Implementation of an algorithm for finding k paths between 2 nodes 
  /// of minimal total length 
  ///
  /// The class \ref hugo::MinLengthPaths "MinLengthPaths" implements
  /// an algorithm for finding k edge-disjoint paths
  /// from a given source node to a given target node in an
  /// edge-weighted directed graph having minimal total weigth (length).
  ///
  ///\warning It is assumed that the lengths are positive, since the
  /// general flow-decomposition is not implemented yet.
  ///
  ///\author Attila Bernath
  template <typename Graph, typename LengthMap>
  class MinLengthPaths{


    typedef typename LengthMap::ValueType Length;
    
    typedef typename Graph::Node Node;
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::Edge Edge;
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    typedef typename Graph::template EdgeMap<int> EdgeIntMap;

    typedef ConstMap<Edge,int> ConstMap;

    //Input
    const Graph& G;

    //Auxiliary variables
    //This is the capacity map for the mincostflow problem
    ConstMap const1map;
    //This MinCostFlows instance will actually solve the problem
    MinCostFlows<Graph, LengthMap, ConstMap> mincost_flow;

    //Container to store found paths
    std::vector< std::vector<Edge> > paths;

  public :


    MinLengthPaths(Graph& _G, LengthMap& _length) : G(_G),
      const1map(1), mincost_flow(_G, _length, const1map){}

    ///Runs the algorithm.

    ///Runs the algorithm.
    ///Returns k if there are at least k edge-disjoint paths from s to t.
    ///Otherwise it returns the number of found edge-disjoint paths from s to t.
    int run(Node s, Node t, int k) {

      int i = mincost_flow.run(s,t,k);
      


      //Let's find the paths
      //We put the paths into stl vectors (as an inner representation). 
      //In the meantime we lose the information stored in 'reversed'.
      //We suppose the lengths to be positive now.

      //We don't want to change the flow of mincost_flow, so we make a copy
      //The name here suggests that the flow has only 0/1 values.
      EdgeIntMap reversed(G); 

      for(typename Graph::EdgeIt e(G); e!=INVALID; ++e) 
	reversed[e] = mincost_flow.getFlow()[e];
      
      paths.clear();
      //total_length=0;
      paths.resize(k);
      for (int j=0; j<i; ++j){
	Node n=s;
	OutEdgeIt e;

	while (n!=t){


	  G.first(e,n);
	  
	  while (!reversed[e]){
	    ++e;
	  }
	  n = G.head(e);
	  paths[j].push_back(e);
	  //total_length += length[e];
	  reversed[e] = 1-reversed[e];
	}
	
      }
      return i;
    }

    
    ///Total length of the paths
    
    ///This function gives back the total length of the found paths.
    ///\pre \ref run() must
    ///be called before using this function.
    Length totalLength(){
      return mincost_flow.totalLength();
    }

    ///Return the found flow.

    ///This function returns a const reference to the EdgeMap \c flow.
    ///\pre \ref run() must
    ///be called before using this function.
    const EdgeIntMap &getFlow() const { return mincost_flow.flow;}

    /// Return the optimal dual solution
    
    ///This function returns a const reference to the NodeMap
    ///\c potential (the dual solution).
    /// \pre \ref run() must be called before using this function.
    const EdgeIntMap &getPotential() const { return mincost_flow.potential;}

    ///Checks whether the complementary slackness holds.

    ///This function checks, whether the given solution is optimal
    ///Running after a \c run() should return with true
    ///Currently this function only checks optimality,
    ///doesn't bother with feasibility
    ///
    ///\todo Is this OK here?
    bool checkComplementarySlackness(){
      return mincost_flow.checkComplementarySlackness();
    }

    ///Read the found paths.
    
    ///This function gives back the \c j-th path in argument p.
    ///Assumes that \c run() has been run and nothing changed since then.
    /// \warning It is assumed that \c p is constructed to
    ///be a path of graph \c G.
    ///If \c j is not less than the result of previous \c run,
    ///then the result here will be an empty path (\c j can be 0 as well).
    template<typename Path>
    void getPath(Path& p, size_t j){
      
      p.clear();
      if (j>paths.size()-1){
	return;
      }
      typename DirPath::Builder B(p);
      for(typename std::vector<Edge>::iterator i=paths[j].begin(); 
	  i!=paths[j].end(); ++i ){
	B.pushBack(*i);
      }

      B.commit();
    }

  }; //class MinLengthPaths

  ///@}

} //namespace hugo

#endif //HUGO_MINLENGTHPATHS_H
