// -*- C++ -*-
#ifndef HUGO_PREFLOW_H
#define HUGO_PREFLOW_H

#include <vector>
#include <queue>

#include <hugo/invalid.h>
#include <hugo/maps.h>

/// \file
/// \ingroup flowalgs

namespace hugo {

  /// \addtogroup flowalgs
  /// @{                                                   

  ///%Preflow algorithms class.

  ///This class provides an implementation of the \e preflow \e
  ///algorithm producing a flow of maximum value in a directed
  ///graph. The preflow algorithms are the fastest max flow algorithms
  ///up to now. The \e source node, the \e target node, the \e
  ///capacity of the edges and the \e starting \e flow value of the
  ///edges should be passed to the algorithm through the
  ///constructor. It is possible to change these quantities using the
  ///functions \ref setSource, \ref setTarget, \ref setCap and \ref
  ///setFlow.
  ///
  ///After running \ref phase1() or \ref preflow(), the actual flow
  ///value can be obtained by calling \ref flowValue(). The minimum
  ///value cut can be written into a <tt>bool</tt> node map by
  ///calling \ref minCut(). (\ref minMinCut() and \ref maxMinCut() writes
  ///the inclusionwise minimum and maximum of the minimum value cuts,
  ///resp.)
  ///
  ///\param Graph The directed graph type the algorithm runs on.
  ///\param Num The number type of the capacities and the flow values.
  ///\param CapMap The capacity map type.
  ///\param FlowMap The flow map type.
  ///
  ///\author Jacint Szabo 
  template <typename Graph, typename Num,
	    typename CapMap=typename Graph::template EdgeMap<Num>,
            typename FlowMap=typename Graph::template EdgeMap<Num> >
  class Preflow {
  protected:
    typedef typename Graph::Node Node;
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::EdgeIt EdgeIt;
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    typedef typename Graph::InEdgeIt InEdgeIt;

    typedef typename Graph::template NodeMap<Node> NNMap;
    typedef typename std::vector<Node> VecNode;

    const Graph* g;
    Node s;
    Node t;
    const CapMap* capacity;
    FlowMap* flow;
    int n;      //the number of nodes of G
    
    typename Graph::template NodeMap<int> level;  
    typename Graph::template NodeMap<Num> excess;

    // constants used for heuristics
    static const int H0=20;
    static const int H1=1;

    public:

    ///Indicates the property of the starting flow map.

    ///Indicates the property of the starting flow map. The meanings are as follows:
    ///- \c ZERO_FLOW: constant zero flow
    ///- \c GEN_FLOW: any flow, i.e. the sum of the in-flows equals to
    ///the sum of the out-flows in every node except the \e source and
    ///the \e target.
    ///- \c PRE_FLOW: any preflow, i.e. the sum of the in-flows is at 
    ///least the sum of the out-flows in every node except the \e source.
    ///- \c NO_FLOW: indicates an unspecified edge map. \ref flow will be 
    ///set to the constant zero flow in the beginning of the algorithm in this case.
    ///
    enum FlowEnum{
      NO_FLOW,
      ZERO_FLOW,
      GEN_FLOW,
      PRE_FLOW
    };

    ///Indicates the state of the preflow algorithm.

    ///Indicates the state of the preflow algorithm. The meanings are as follows:
    ///- \c AFTER_NOTHING: before running the algorithm or at an unspecified state.
    ///- \c AFTER_PREFLOW_PHASE_1: right after running \c phase1
    ///- \c AFTER_PREFLOW_PHASE_2: after running \ref phase2()
    ///
    enum StatusEnum {
      AFTER_NOTHING,
      AFTER_PREFLOW_PHASE_1,      
      AFTER_PREFLOW_PHASE_2
    };
    
    protected: 
      FlowEnum flow_prop;
    StatusEnum status; // Do not needle this flag only if necessary.
    
  public: 
    ///The constructor of the class.

    ///The constructor of the class. 
    ///\param _G The directed graph the algorithm runs on. 
    ///\param _s The source node.
    ///\param _t The target node.
    ///\param _capacity The capacity of the edges. 
    ///\param _flow The flow of the edges. 
    ///Except the graph, all of these parameters can be reset by
    ///calling \ref setSource, \ref setTarget, \ref setCap and \ref
    ///setFlow, resp.
      Preflow(const Graph& _G, Node _s, Node _t, 
	      const CapMap& _capacity, FlowMap& _flow) :
	g(&_G), s(_s), t(_t), capacity(&_capacity),
	flow(&_flow), n(_G.nodeNum()), level(_G), excess(_G,0), 
	flow_prop(NO_FLOW), status(AFTER_NOTHING) { }


                                                                              
    ///Runs the preflow algorithm.  

    ///Runs the preflow algorithm.
    ///
    void run() {
      phase1(flow_prop);
      phase2();
    }
    
    ///Runs the preflow algorithm.  
    
    ///Runs the preflow algorithm. 
    ///\pre The starting flow map must be
    /// - a constant zero flow if \c fp is \c ZERO_FLOW,
    /// - an arbitrary flow if \c fp is \c GEN_FLOW,
    /// - an arbitrary preflow if \c fp is \c PRE_FLOW,
    /// - any map if \c fp is NO_FLOW.
    ///If the starting flow map is a flow or a preflow then 
    ///the algorithm terminates faster.
    void run(FlowEnum fp) {
      flow_prop=fp;
      run();
    }
      
    ///Runs the first phase of the preflow algorithm.

    ///The preflow algorithm consists of two phases, this method runs the
    ///first phase. After the first phase the maximum flow value and a
    ///minimum value cut can already be computed, though a maximum flow
    ///is not yet obtained. So after calling this method \ref flowValue
    ///and \ref minCut gives proper results.
    ///\warning \ref minMinCut and \ref maxMinCut do not
    ///give minimum value cuts unless calling \ref phase2.
    ///\pre The starting flow must be
    /// - a constant zero flow if \c fp is \c ZERO_FLOW,
    /// - an arbitary flow if \c fp is \c GEN_FLOW,
    /// - an arbitary preflow if \c fp is \c PRE_FLOW,
    /// - any map if \c fp is NO_FLOW.
    void phase1(FlowEnum fp)
    {
      flow_prop=fp;
      phase1();
    }

    
    ///Runs the first phase of the preflow algorithm.

    ///The preflow algorithm consists of two phases, this method runs the
    ///first phase. After the first phase the maximum flow value and a
    ///minimum value cut can already be computed, though a maximum flow
    ///is not yet obtained. So after calling this method \ref flowValue
    ///and \ref actMinCut gives proper results.
    ///\warning \ref minCut, \ref minMinCut and \ref maxMinCut do not
    ///give minimum value cuts unless calling \ref phase2.
    void phase1()
    {
      int heur0=(int)(H0*n);  //time while running 'bound decrease'
      int heur1=(int)(H1*n);  //time while running 'highest label'
      int heur=heur1;         //starting time interval (#of relabels)
      int numrelabel=0;

      bool what_heur=1;
      //It is 0 in case 'bound decrease' and 1 in case 'highest label'

      bool end=false;
      //Needed for 'bound decrease', true means no active 
      //nodes are above bound b.

      int k=n-2;  //bound on the highest level under n containing a node
      int b=k;    //bound on the highest level under n of an active node

      VecNode first(n, INVALID);
      NNMap next(*g, INVALID);

      NNMap left(*g, INVALID);
      NNMap right(*g, INVALID);
      VecNode level_list(n,INVALID);
      //List of the nodes in level i<n, set to n.

      preflowPreproc(first, next, level_list, left, right);

      //Push/relabel on the highest level active nodes.
      while ( true ) {
	if ( b == 0 ) {
	  if ( !what_heur && !end && k > 0 ) {
	    b=k;
	    end=true;
	  } else break;
	}

	if ( first[b]==INVALID ) --b;
	else {
	  end=false;
	  Node w=first[b];
	  first[b]=next[w];
	  int newlevel=push(w, next, first);
	  if ( excess[w] > 0 ) relabel(w, newlevel, first, next, level_list, 
				       left, right, b, k, what_heur);

	  ++numrelabel;
	  if ( numrelabel >= heur ) {
	    numrelabel=0;
	    if ( what_heur ) {
	      what_heur=0;
	      heur=heur0;
	      end=false;
	    } else {
	      what_heur=1;
	      heur=heur1;
	      b=k;
	    }
	  }
	}
      }
      flow_prop=PRE_FLOW;
      status=AFTER_PREFLOW_PHASE_1;
    }
    // Heuristics:
    //   2 phase
    //   gap
    //   list 'level_list' on the nodes on level i implemented by hand
    //   stack 'active' on the active nodes on level i      
    //   runs heuristic 'highest label' for H1*n relabels
    //   runs heuristic 'bound decrease' for H0*n relabels, starts with 'highest label'
    //   Parameters H0 and H1 are initialized to 20 and 1.


    ///Runs the second phase of the preflow algorithm.

    ///The preflow algorithm consists of two phases, this method runs
    ///the second phase. After calling \ref phase1 and then
    ///\ref phase2 the methods \ref flowValue, \ref minCut,
    ///\ref minMinCut and \ref maxMinCut give proper results.
    ///\pre \ref phase1 must be called before.
    void phase2()
    {

      int k=n-2;  //bound on the highest level under n containing a node
      int b=k;    //bound on the highest level under n of an active node

    
      VecNode first(n, INVALID);
      NNMap next(*g, INVALID); 
      level.set(s,0);
      std::queue<Node> bfs_queue;
      bfs_queue.push(s);

      while ( !bfs_queue.empty() ) {

	Node v=bfs_queue.front();
	bfs_queue.pop();
	int l=level[v]+1;

	for(InEdgeIt e(*g,v); e!=INVALID; ++e) {
	  if ( (*capacity)[e] <= (*flow)[e] ) continue;
	  Node u=g->tail(e);
	  if ( level[u] >= n ) {
	    bfs_queue.push(u);
	    level.set(u, l);
	    if ( excess[u] > 0 ) {
	      next.set(u,first[l]);
	      first[l]=u;
	    }
	  }
	}

	for(OutEdgeIt e(*g,v); e!=INVALID; ++e) {
	  if ( 0 >= (*flow)[e] ) continue;
	  Node u=g->head(e);
	  if ( level[u] >= n ) {
	    bfs_queue.push(u);
	    level.set(u, l);
	    if ( excess[u] > 0 ) {
	      next.set(u,first[l]);
	      first[l]=u;
	    }
	  }
	}
      }
      b=n-2;

      while ( true ) {

	if ( b == 0 ) break;
	if ( first[b]==INVALID ) --b;
	else {
	  Node w=first[b];
	  first[b]=next[w];
	  int newlevel=push(w,next, first);
	  
	  //relabel
	  if ( excess[w] > 0 ) {
	    level.set(w,++newlevel);
	    next.set(w,first[newlevel]);
	    first[newlevel]=w;
	    b=newlevel;
	  }
	} 
      } // while(true)
      flow_prop=GEN_FLOW;
      status=AFTER_PREFLOW_PHASE_2;
    }

    /// Returns the value of the maximum flow.

    /// Returns the value of the maximum flow by returning the excess
    /// of the target node \ref t. This value equals to the value of
    /// the maximum flow already after running \ref phase1.
    Num flowValue() const {
      return excess[t];
    }


    ///Returns a minimum value cut.

    ///Sets \c M to the characteristic vector of a minimum value
    ///cut. This method can be called both after running \ref
    ///phase1 and \ref phase2. It is much faster after
    ///\ref phase1.  \pre M should be a bool-valued node-map. \pre
    ///If \ref mincut is called after \ref phase2 then M should
    ///be initialized to false.
    template<typename _CutMap>
    void minCut(_CutMap& M) const {
      switch ( status ) {
	case AFTER_PREFLOW_PHASE_1:
	for(NodeIt v(*g); v!=INVALID; ++v) {
	  if (level[v] < n) {
	    M.set(v, false);
	  } else {
	    M.set(v, true);
	  }
	}
	break;
	case AFTER_PREFLOW_PHASE_2:
	minMinCut(M);
	break;
	case AFTER_NOTHING:
	break;
      }
    }

    ///Returns the inclusionwise minimum of the minimum value cuts.

    ///Sets \c M to the characteristic vector of the minimum value cut
    ///which is inclusionwise minimum. It is computed by processing a
    ///bfs from the source node \c s in the residual graph.  \pre M
    ///should be a node map of bools initialized to false.  \pre \ref
    ///phase2 should already be run.
    template<typename _CutMap>
    void minMinCut(_CutMap& M) const {

      std::queue<Node> queue;
      M.set(s,true);
      queue.push(s);
      
      while (!queue.empty()) {
	Node w=queue.front();
	queue.pop();
	
	for(OutEdgeIt e(*g,w) ; e!=INVALID; ++e) {
	  Node v=g->head(e);
	  if (!M[v] && (*flow)[e] < (*capacity)[e] ) {
	    queue.push(v);
	    M.set(v, true);
	  }
	}
	
	for(InEdgeIt e(*g,w) ; e!=INVALID; ++e) {
	  Node v=g->tail(e);
	  if (!M[v] && (*flow)[e] > 0 ) {
	    queue.push(v);
	    M.set(v, true);
	  }
	}
      }
    }
    
    ///Returns the inclusionwise maximum of the minimum value cuts.

    ///Sets \c M to the characteristic vector of the minimum value cut
    ///which is inclusionwise maximum. It is computed by processing a
    ///backward bfs from the target node \c t in the residual graph.
    ///\pre \ref phase2() or preflow() should already be run.
    template<typename _CutMap>
    void maxMinCut(_CutMap& M) const {

      for(NodeIt v(*g) ; v!=INVALID; ++v) M.set(v, true);

      std::queue<Node> queue;

      M.set(t,false);
      queue.push(t);

      while (!queue.empty()) {
        Node w=queue.front();
	queue.pop();

	for(InEdgeIt e(*g,w) ; e!=INVALID; ++e) {
	  Node v=g->tail(e);
	  if (M[v] && (*flow)[e] < (*capacity)[e] ) {
	    queue.push(v);
	    M.set(v, false);
	  }
	}

	for(OutEdgeIt e(*g,w) ; e!=INVALID; ++e) {
	  Node v=g->head(e);
	  if (M[v] && (*flow)[e] > 0 ) {
	    queue.push(v);
	    M.set(v, false);
	  }
	}
      }
    }

    ///Sets the source node to \c _s.

    ///Sets the source node to \c _s.
    /// 
    void setSource(Node _s) { 
      s=_s; 
      if ( flow_prop != ZERO_FLOW ) flow_prop=NO_FLOW;
      status=AFTER_NOTHING; 
    }

    ///Sets the target node to \c _t.

    ///Sets the target node to \c _t.
    ///
    void setTarget(Node _t) { 
      t=_t; 
      if ( flow_prop == GEN_FLOW ) flow_prop=PRE_FLOW;
      status=AFTER_NOTHING; 
    }

    /// Sets the edge map of the capacities to _cap.

    /// Sets the edge map of the capacities to _cap.
    /// 
    void setCap(const CapMap& _cap) { 
      capacity=&_cap; 
      status=AFTER_NOTHING; 
    }

    /// Sets the edge map of the flows to _flow.

    /// Sets the edge map of the flows to _flow.
    /// 
    void setFlow(FlowMap& _flow) { 
      flow=&_flow; 
      flow_prop=NO_FLOW;
      status=AFTER_NOTHING; 
    }


  private:

    int push(Node w, NNMap& next, VecNode& first) {

      int lev=level[w];
      Num exc=excess[w];
      int newlevel=n;       //bound on the next level of w

      for(OutEdgeIt e(*g,w) ; e!=INVALID; ++e) {
	if ( (*flow)[e] >= (*capacity)[e] ) continue;
	Node v=g->head(e);

	if( lev > level[v] ) { //Push is allowed now
	  
	  if ( excess[v]<=0 && v!=t && v!=s ) {
	    next.set(v,first[level[v]]);
	    first[level[v]]=v;
	  }

	  Num cap=(*capacity)[e];
	  Num flo=(*flow)[e];
	  Num remcap=cap-flo;
	  
	  if ( remcap >= exc ) { //A nonsaturating push.
	    
	    flow->set(e, flo+exc);
	    excess.set(v, excess[v]+exc);
	    exc=0;
	    break;

	  } else { //A saturating push.
	    flow->set(e, cap);
	    excess.set(v, excess[v]+remcap);
	    exc-=remcap;
	  }
	} else if ( newlevel > level[v] ) newlevel = level[v];
      } //for out edges wv

      if ( exc > 0 ) {
	for(InEdgeIt e(*g,w) ; e!=INVALID; ++e) {
	  
	  if( (*flow)[e] <= 0 ) continue;
	  Node v=g->tail(e);

	  if( lev > level[v] ) { //Push is allowed now

	    if ( excess[v]<=0 && v!=t && v!=s ) {
	      next.set(v,first[level[v]]);
	      first[level[v]]=v;
	    }

	    Num flo=(*flow)[e];

	    if ( flo >= exc ) { //A nonsaturating push.

	      flow->set(e, flo-exc);
	      excess.set(v, excess[v]+exc);
	      exc=0;
	      break;
	    } else {  //A saturating push.

	      excess.set(v, excess[v]+flo);
	      exc-=flo;
	      flow->set(e,0);
	    }
	  } else if ( newlevel > level[v] ) newlevel = level[v];
	} //for in edges vw

      } // if w still has excess after the out edge for cycle

      excess.set(w, exc);
      
      return newlevel;
    }
    
    
    
    void preflowPreproc(VecNode& first, NNMap& next, 
			VecNode& level_list, NNMap& left, NNMap& right)
    {
      for(NodeIt v(*g); v!=INVALID; ++v) level.set(v,n);
      std::queue<Node> bfs_queue;
      
      if ( flow_prop == GEN_FLOW || flow_prop == PRE_FLOW ) {
	//Reverse_bfs from t in the residual graph,
	//to find the starting level.
	level.set(t,0);
	bfs_queue.push(t);
	
	while ( !bfs_queue.empty() ) {
	  
	  Node v=bfs_queue.front();
	  bfs_queue.pop();
	  int l=level[v]+1;
	  
	  for(InEdgeIt e(*g,v) ; e!=INVALID; ++e) {
	    if ( (*capacity)[e] <= (*flow)[e] ) continue;
	    Node w=g->tail(e);
	    if ( level[w] == n && w != s ) {
	      bfs_queue.push(w);
	      Node z=level_list[l];
	      if ( z!=INVALID ) left.set(z,w);
	      right.set(w,z);
	      level_list[l]=w;
	      level.set(w, l);
	    }
	  }
	  
	  for(OutEdgeIt e(*g,v) ; e!=INVALID; ++e) {
	    if ( 0 >= (*flow)[e] ) continue;
	    Node w=g->head(e);
	    if ( level[w] == n && w != s ) {
	      bfs_queue.push(w);
	      Node z=level_list[l];
	      if ( z!=INVALID ) left.set(z,w);
	      right.set(w,z);
	      level_list[l]=w;
	      level.set(w, l);
	    }
	  }
	} //while
      } //if


      switch (flow_prop) {
	case NO_FLOW:  
	for(EdgeIt e(*g); e!=INVALID; ++e) flow->set(e,0);
	case ZERO_FLOW:
	for(NodeIt v(*g); v!=INVALID; ++v) excess.set(v,0);
	
	//Reverse_bfs from t, to find the starting level.
	level.set(t,0);
	bfs_queue.push(t);
	
	while ( !bfs_queue.empty() ) {
	  
	  Node v=bfs_queue.front();
	  bfs_queue.pop();
	  int l=level[v]+1;
	  
	  for(InEdgeIt e(*g,v) ; e!=INVALID; ++e) {
	    Node w=g->tail(e);
	    if ( level[w] == n && w != s ) {
	      bfs_queue.push(w);
	      Node z=level_list[l];
	      if ( z!=INVALID ) left.set(z,w);
	      right.set(w,z);
	      level_list[l]=w;
	      level.set(w, l);
	    }
	  }
	}
	
	//the starting flow
	for(OutEdgeIt e(*g,s) ; e!=INVALID; ++e) {
	  Num c=(*capacity)[e];
	  if ( c <= 0 ) continue;
	  Node w=g->head(e);
	  if ( level[w] < n ) {
	    if ( excess[w] <= 0 && w!=t ) { //putting into the stack
	      next.set(w,first[level[w]]);
	      first[level[w]]=w;
	    }
	    flow->set(e, c);
	    excess.set(w, excess[w]+c);
	  }
	}
	break;

	case GEN_FLOW:
	for(NodeIt v(*g); v!=INVALID; ++v) excess.set(v,0);
	{
	  Num exc=0;
	  for(InEdgeIt e(*g,t) ; e!=INVALID; ++e) exc+=(*flow)[e];
	  for(OutEdgeIt e(*g,t) ; e!=INVALID; ++e) exc-=(*flow)[e];
	  excess.set(t,exc);
	}

	//the starting flow
	for(OutEdgeIt e(*g,s); e!=INVALID; ++e)	{
	  Num rem=(*capacity)[e]-(*flow)[e];
	  if ( rem <= 0 ) continue;
	  Node w=g->head(e);
	  if ( level[w] < n ) {
	    if ( excess[w] <= 0 && w!=t ) { //putting into the stack
	      next.set(w,first[level[w]]);
	      first[level[w]]=w;
	    }   
	    flow->set(e, (*capacity)[e]);
	    excess.set(w, excess[w]+rem);
	  }
	}
	
	for(InEdgeIt e(*g,s); e!=INVALID; ++e) {
	  if ( (*flow)[e] <= 0 ) continue;
	  Node w=g->tail(e);
	  if ( level[w] < n ) {
	    if ( excess[w] <= 0 && w!=t ) {
	      next.set(w,first[level[w]]);
	      first[level[w]]=w;
	    }  
	    excess.set(w, excess[w]+(*flow)[e]);
	    flow->set(e, 0);
	  }
	}
	break;

	case PRE_FLOW:	
	//the starting flow
	for(OutEdgeIt e(*g,s) ; e!=INVALID; ++e) {
	  Num rem=(*capacity)[e]-(*flow)[e];
	  if ( rem <= 0 ) continue;
	  Node w=g->head(e);
	  if ( level[w] < n ) flow->set(e, (*capacity)[e]);
	}
	
	for(InEdgeIt e(*g,s) ; e!=INVALID; ++e) {
	  if ( (*flow)[e] <= 0 ) continue;
	  Node w=g->tail(e);
	  if ( level[w] < n ) flow->set(e, 0);
	}
	
	//computing the excess
	for(NodeIt w(*g); w!=INVALID; ++w) {
	  Num exc=0;
	  for(InEdgeIt e(*g,w); e!=INVALID; ++e) exc+=(*flow)[e];
	  for(OutEdgeIt e(*g,w); e!=INVALID; ++e) exc-=(*flow)[e];
	  excess.set(w,exc);
	  
	  //putting the active nodes into the stack
	  int lev=level[w];
	    if ( exc > 0 && lev < n && Node(w) != t ) {
	      next.set(w,first[lev]);
	      first[lev]=w;
	    }
	}
	break;
      } //switch
    } //preflowPreproc


    void relabel(Node w, int newlevel, VecNode& first, NNMap& next, 
		 VecNode& level_list, NNMap& left,
		 NNMap& right, int& b, int& k, bool what_heur )
    {

      int lev=level[w];

      Node right_n=right[w];
      Node left_n=left[w];

      //unlacing starts
      if ( right_n!=INVALID ) {
	if ( left_n!=INVALID ) {
	  right.set(left_n, right_n);
	  left.set(right_n, left_n);
	} else {
	  level_list[lev]=right_n;
	  left.set(right_n, INVALID);
	}
      } else {
	if ( left_n!=INVALID ) {
	  right.set(left_n, INVALID);
	} else {
	  level_list[lev]=INVALID;
	}
      }
      //unlacing ends

      if ( level_list[lev]==INVALID ) {

	//gapping starts
	for (int i=lev; i!=k ; ) {
	  Node v=level_list[++i];
	  while ( v!=INVALID ) {
	    level.set(v,n);
	    v=right[v];
	  }
	  level_list[i]=INVALID;
	  if ( !what_heur ) first[i]=INVALID;
	}

	level.set(w,n);
	b=lev-1;
	k=b;
	//gapping ends

      } else {

	if ( newlevel == n ) level.set(w,n);
	else {
	  level.set(w,++newlevel);
	  next.set(w,first[newlevel]);
	  first[newlevel]=w;
	  if ( what_heur ) b=newlevel;
	  if ( k < newlevel ) ++k;      //now k=newlevel
	  Node z=level_list[newlevel];
	  if ( z!=INVALID ) left.set(z,w);
	  right.set(w,z);
	  left.set(w,INVALID);
	  level_list[newlevel]=w;
	}
      }
    } //relabel

  }; 
} //namespace hugo

#endif //HUGO_PREFLOW_H




