#ifndef NODE_MAP_BASE_H
#define NODE_MAP_BASE_H

/**
	Template base class for implementing mapping on nodes.
	\param The first template parameter is the Graph class. The Graph
		must have an \emp node_maps member with \emp NodeMapRegistry class.
	\param The second template parameter is the Node type of the class.
	
*/

template <typename G, typename K>
class NodeMapBase {
public:
	typedef G Graph;

	typedef K KeyType;
	
	/** 
		Default constructor.
	*/	
	
	NodeMapBase() : graph(0) {}

	/** 
		Simple constructor to register into a graph.
	*/
	
	NodeMapBase(Graph& g) : graph(&g) {
		graph->node_maps.add(*this);
	}

	/** 
		Copy constructor with registering into the map.
	*/	
	
	NodeMapBase(const NodeMapBase& copy) : graph(copy.graph) {
		if (graph) {
			graph->node_maps.add(*this);
		}
	} 
	
	/** 
		Assign operator.
	*/	

	const NodeMapBase& operator=(const NodeMapBase& copy) {
		if (graph) {
			graph.node_maps.erase(*this);
		}
		graph = copy.graph;
		if (graph) {
			graph->node_maps.add(*this);
		}
		
	}
	

	/** 
		Destructor.
	*/	

	virtual ~NodeMapBase() {
		if (graph) {
			graph.node_maps.erase(*this);
		}
	}
	
protected:
	
	Graph* graph;

	int graph_index;
	
	/**
		Helper function to implement the default constructor in the subclasses.
	*/
	
	void init() {
		for (Graph::NodeIt it(g); g.valid(it); g.next(it)) {
			add(it);
		}
	}
	
	/**
		Helper function to implement the destructor in the subclasses.
	*/
	
	void destroy() {
		for (Graph::NodeIt it(g); g.valid(it); g.next(it)) {
			erase(it);
		}
	}
	
	/** 
		The add member function should be overloaded in the subclasses.
		\e Add extends the map with the new node.
	*/
	
	virtual void add(const KeyType&) = 0;
	
	/** 
		The erase member function should be overloaded in the subclasses.
		\e Erase removes the node from the map.
	*/
	
	virtual void erase(const KeyType&) = 0;
	
	/**
		Exception class to throw at unsupported operation.
	*/
	
	class NotSupportedOperationException {};

	friend class Graph;
};

#endif
