#include <fstream>
#include <string>

#include "test_tools.h"
#include <hugo/smart_graph.h>
#include <hugo/dimacs.h>
#include <hugo/preflow.h>
#include <hugo/skeletons/graph.h>
#include <hugo/skeletons/maps.h>

using namespace hugo;

void check_Preflow() 
{
  typedef int VType;
  typedef skeleton::StaticGraph Graph;

  typedef Graph::Node Node;
  typedef Graph::Edge Edge;
  typedef skeleton::ReadMap<Edge,VType> CapMap;
  typedef skeleton::ReadWriteMap<Edge,VType> FlowMap;
  typedef skeleton::ReadWriteMap<Node,bool> CutMap;
 
  typedef Preflow<Graph, int, CapMap, FlowMap> PType;

  Graph G;
  Node n;
  CapMap cap;
  FlowMap flow;
  CutMap cut;

  PType preflow_test(G,n,n,cap,flow);

  preflow_test.run();
  preflow_test.flowValue();
  preflow_test.setSource(n);
  preflow_test.setFlow(flow);

  preflow_test.phase1(PType::NO_FLOW);
  preflow_test.minCut(cut);

  preflow_test.phase2();
  preflow_test.setTarget(n);
  preflow_test.setCap(cap);
  preflow_test.minMinCut(cut);
  preflow_test.maxMinCut(cut);
}

int cut_value ( SmartGraph& G, SmartGraph::NodeMap<bool>& cut, 
		SmartGraph::EdgeMap<int>& cap) {
  
  int c=0;
  for(SmartGraph::EdgeIt e(G); e!=INVALID; ++e) {
    if (cut[G.tail(e)] && !cut[G.head(e)]) c+=cap[e];
  }
  return c;
}

int main() {

  typedef SmartGraph Graph;
  
  typedef Graph::Node Node;
  typedef Graph::NodeIt NodeIt;
  typedef Graph::EdgeIt EdgeIt;
  typedef Graph::EdgeMap<int> CapMap;
  typedef Graph::EdgeMap<int> FlowMap;
  typedef Graph::NodeMap<bool> CutMap;

  typedef Preflow<Graph, int> PType;

  std::string f_name;
  if( getenv("srcdir") ) {
    f_name = std::string(getenv("srcdir")) + "/preflow_graph.dim";
  }
  else {
    f_name = "preflow_graph.dim";
  }
  
  std::ifstream file(f_name.c_str());
  
  check(file, "Input file '" << f_name << "' not found.");
  
  Graph G;
  Node s, t;
  CapMap cap(G);
  readDimacs(file, G, cap, s, t);

  FlowMap flow(G,0);

 

  PType preflow_test(G, s, t, cap, flow);
  preflow_test.run(PType::ZERO_FLOW);
    
  CutMap mincut(G,false);
  preflow_test.minCut(mincut); 
  int min_cut_value=cut_value(G,mincut,cap);
   
  CutMap minmincut(G,false);
  preflow_test.minMinCut(minmincut); 
  int min_min_cut_value=cut_value(G,minmincut,cap);
   
  CutMap maxmincut(G,false);
  preflow_test.maxMinCut(maxmincut); 
  int max_min_cut_value=cut_value(G,maxmincut,cap);

  check(preflow_test.flowValue() == min_cut_value &&
	min_cut_value == min_min_cut_value &&
	min_min_cut_value == max_min_cut_value,
	"The max flow value is not equal to the three min cut values.");

  int flow_value=preflow_test.flowValue();



  for(EdgeIt e(G); e!=INVALID; ++e) cap[e]=2*cap[e]; 
  preflow_test.setCap(cap);  

  preflow_test.phase1(PType::PRE_FLOW);

  CutMap mincut1(G,false);
  preflow_test.minCut(mincut1); 
  min_cut_value=cut_value(G,mincut1,cap);
   
  check(preflow_test.flowValue() == min_cut_value &&
	min_cut_value == 2*flow_value,
	"The max flow value or the min cut value is wrong.");

  preflow_test.phase2();

  CutMap mincut2(G,false);
  preflow_test.minCut(mincut2); 
  min_cut_value=cut_value(G,mincut2,cap);
   
  CutMap minmincut2(G,false);
  preflow_test.minMinCut(minmincut2); 
  min_min_cut_value=cut_value(G,minmincut2,cap);
 
  preflow_test.maxMinCut(maxmincut); 
  max_min_cut_value=cut_value(G,maxmincut,cap);

  check(preflow_test.flowValue() == min_cut_value &&
	min_cut_value == min_min_cut_value &&
	min_min_cut_value == max_min_cut_value &&
	min_cut_value == 2*flow_value,
	"The max flow value or the three min cut values were not doubled");



  EdgeIt e(G);
  for( int i=1; i==10; ++i ) {
    flow.set(e,0);
    ++e;
  }

  preflow_test.setFlow(flow); 

  NodeIt tmp1(G,s);
  ++tmp1;
  if ( tmp1 != INVALID ) s=tmp1;

  NodeIt tmp2(G,t);
  ++tmp2;
  if ( tmp2 != INVALID ) t=tmp2;

  preflow_test.setSource(s);
  preflow_test.setTarget(t); 
  
  preflow_test.run();

  CutMap mincut3(G,false);
  preflow_test.minCut(mincut3); 
  min_cut_value=cut_value(G,mincut3,cap);
   
  CutMap minmincut3(G,false);
  preflow_test.minMinCut(minmincut3); 
  min_min_cut_value=cut_value(G,minmincut3,cap);
   
  preflow_test.maxMinCut(maxmincut); 
  max_min_cut_value=cut_value(G,maxmincut,cap);

  check(preflow_test.flowValue() == min_cut_value &&
	min_cut_value == min_min_cut_value &&
	min_min_cut_value == max_min_cut_value,
	"The max flow value or the three min cut values are incorrect.");
}
