/* -*- C++ -*-
 * lemon/concept/graph.h - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2005 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef LEMON_CONCEPT_GRAPH_H
#define LEMON_CONCEPT_GRAPH_H

///\ingroup graph_concepts
///\file
///\brief Declaration of Graph.

#include <lemon/invalid.h>
#include <lemon/utility.h>
#include <lemon/concept/maps.h>
#include <lemon/concept_check.h>
#include <lemon/concept/graph_component.h>

namespace lemon {
  namespace concept {

    
    /**************** The full-featured graph concepts ****************/


    /// \brief Modular static graph class.
    ///     
    /// It should be the same as the \c StaticGraph class.
    class _StaticGraph 
      :  virtual public BaseGraphComponent,
         public IterableGraphComponent, public MappableGraphComponent {
    public:
      ///\e

      ///\todo undocumented
      ///
      typedef False UndirTag;
      
      typedef BaseGraphComponent::Node Node;
      typedef BaseGraphComponent::Edge Edge;

      template <typename _Graph>
      struct Constraints {
        void constraints() {
          checkConcept<IterableGraphComponent, _Graph>();
          checkConcept<MappableGraphComponent, _Graph>();
        }
      };
    };

    /// \brief Modular extendable graph class.
    ///     
    /// It should be the same as the \c ExtendableGraph class.
    class _ExtendableGraph 
      :  virtual public BaseGraphComponent, public _StaticGraph,
         public ExtendableGraphComponent, public ClearableGraphComponent {
    public:
      typedef BaseGraphComponent::Node Node;
      typedef BaseGraphComponent::Edge Edge;

      template <typename _Graph>
      struct Constraints {
        void constraints() {
          checkConcept<_StaticGraph, _Graph >();
          checkConcept<ExtendableGraphComponent, _Graph >();
          checkConcept<ClearableGraphComponent, _Graph >();
        }
      };
    };

    /// \brief Modular erasable graph class.
    ///     
    /// It should be the same as the \c ErasableGraph class.
    class _ErasableGraph 
      :  virtual public BaseGraphComponent, public _ExtendableGraph,
         public ErasableGraphComponent {
    public:
      typedef BaseGraphComponent::Node Node;
      typedef BaseGraphComponent::Edge Edge;

      template <typename _Graph>
      struct Constraints {
        void constraints() {
          checkConcept<_ExtendableGraph, _Graph >();
          checkConcept<ErasableGraphComponent, _Graph >();
        }
      };
    };

    /// \addtogroup graph_concepts
    /// @{

    /// An empty static graph class.
  
    /// This class provides all the common features of a graph structure,
    /// however completely without implementations and real data structures
    /// behind the interface.
    /// All graph algorithms should compile with this class, but it will not
    /// run properly, of course.
    ///
    /// It can be used for checking the interface compatibility,
    /// or it can serve as a skeleton of a new graph structure.
    /// 
    /// Also, you will find here the full documentation of a certain graph
    /// feature, the documentation of a real graph imlementation
    /// like @ref ListGraph or
    /// @ref SmartGraph will just refer to this structure.
    ///
    /// \todo A pages describing the concept of concept description would
    /// be nice.
    class StaticGraph
    {
    public:
      ///\e

      ///\todo undocumented
      ///
      typedef False UndirTag;

      /// Defalult constructor.

      /// Defalult constructor.
      ///
      StaticGraph() { }
      ///Copy consructor.

//       ///\todo It is not clear, what we expect from a copy constructor.
//       ///E.g. How to assign the nodes/edges to each other? What about maps?
//       StaticGraph(const StaticGraph& g) { }

      /// The base type of node iterators, 
      /// or in other words, the trivial node iterator.

      /// This is the base type of each node iterator,
      /// thus each kind of node iterator converts to this.
      /// More precisely each kind of node iterator should be inherited 
      /// from the trivial node iterator.
      class Node {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        Node() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        Node(const Node&) { }

        /// Invalid constructor \& conversion.

        /// This constructor initializes the iterator to be invalid.
        /// \sa Invalid for more details.
        Node(Invalid) { }
        /// Equality operator

        /// Two iterators are equal if and only if they point to the
        /// same object or both are invalid.
        bool operator==(Node) const { return true; }

        /// Inequality operator
        
        /// \sa operator==(Node n)
        ///
        bool operator!=(Node) const { return true; }

	/// Artificial ordering operator.
	
	/// To allow the use of graph descriptors as key type in std::map or
	/// similar associative container we require this.
	///
	/// \note This operator only have to define some strict ordering of
	/// the items; this order has nothing to do with the iteration
	/// ordering of the items.
	///
	/// \bug This is a technical requirement. Do we really need this?
	bool operator<(Node) const { return false; }

      };
    
      /// This iterator goes through each node.

      /// This iterator goes through each node.
      /// Its usage is quite simple, for example you can count the number
      /// of nodes in graph \c g of type \c Graph like this:
      /// \code
      /// int count=0;
      /// for (Graph::NodeIt n(g); n!=INVALID; ++n) ++count;
      /// \endcode
      class NodeIt : public Node {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        NodeIt() { }
        /// Copy constructor.
        
        /// Copy constructor.
        ///
        NodeIt(const NodeIt& n) : Node(n) { }
        /// Invalid constructor \& conversion.

        /// Initialize the iterator to be invalid.
        /// \sa Invalid for more details.
        NodeIt(Invalid) { }
        /// Sets the iterator to the first node.

        /// Sets the iterator to the first node of \c g.
        ///
        NodeIt(const StaticGraph&) { }
        /// Node -> NodeIt conversion.

        /// Sets the iterator to the node of \c the graph pointed by 
	/// the trivial iterator.
        /// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        NodeIt(const StaticGraph&, const Node&) { }
        /// Next node.

        /// Assign the iterator to the next node.
        ///
        NodeIt& operator++() { return *this; }
      };
    
    
      /// The base type of the edge iterators.

      /// The base type of the edge iterators.
      ///
      class Edge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        Edge() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        Edge(const Edge&) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        Edge(Invalid) { }
        /// Equality operator

        /// Two iterators are equal if and only if they point to the
        /// same object or both are invalid.
        bool operator==(Edge) const { return true; }
        /// Inequality operator

        /// \sa operator==(Edge n)
        ///
        bool operator!=(Edge) const { return true; }

	/// Artificial ordering operator.
	
	/// To allow the use of graph descriptors as key type in std::map or
	/// similar associative container we require this.
	///
	/// \note This operator only have to define some strict ordering of
	/// the items; this order has nothing to do with the iteration
	/// ordering of the items.
	///
	/// \bug This is a technical requirement. Do we really need this?
	bool operator<(Edge) const { return false; }
      };
    
      /// This iterator goes trough the outgoing edges of a node.

      /// This iterator goes trough the \e outgoing edges of a certain node
      /// of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of outgoing edges of a node \c n
      /// in graph \c g of type \c Graph as follows.
      /// \code
      /// int count=0;
      /// for (Graph::OutEdgeIt e(g, n); e!=INVALID; ++e) ++count;
      /// \endcode
    
      class OutEdgeIt : public Edge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        OutEdgeIt() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        OutEdgeIt(const OutEdgeIt& e) : Edge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        OutEdgeIt(Invalid) { }
        /// This constructor sets the iterator to the first outgoing edge.
    
        /// This constructor sets the iterator to the first outgoing edge of
        /// the node.
        OutEdgeIt(const StaticGraph&, const Node&) { }
        /// Edge -> OutEdgeIt conversion

        /// Sets the iterator to the value of the trivial iterator.
	/// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        OutEdgeIt(const StaticGraph&, const Edge&) { }
        ///Next outgoing edge
        
        /// Assign the iterator to the next 
        /// outgoing edge of the corresponding node.
        OutEdgeIt& operator++() { return *this; }
      };

      /// This iterator goes trough the incoming edges of a node.

      /// This iterator goes trough the \e incoming edges of a certain node
      /// of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of outgoing edges of a node \c n
      /// in graph \c g of type \c Graph as follows.
      /// \code
      /// int count=0;
      /// for(Graph::InEdgeIt e(g, n); e!=INVALID; ++e) ++count;
      /// \endcode

      class InEdgeIt : public Edge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        InEdgeIt() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        InEdgeIt(const InEdgeIt& e) : Edge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        InEdgeIt(Invalid) { }
        /// This constructor sets the iterator to first incoming edge.
    
        /// This constructor set the iterator to the first incoming edge of
        /// the node.
        InEdgeIt(const StaticGraph&, const Node&) { }
        /// Edge -> InEdgeIt conversion

        /// Sets the iterator to the value of the trivial iterator \c e.
        /// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        InEdgeIt(const StaticGraph&, const Edge&) { }
        /// Next incoming edge

        /// Assign the iterator to the next inedge of the corresponding node.
        ///
        InEdgeIt& operator++() { return *this; }
      };
      /// This iterator goes through each edge.

      /// This iterator goes through each edge of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of edges in a graph \c g of type \c Graph as follows:
      /// \code
      /// int count=0;
      /// for(Graph::EdgeIt e(g); e!=INVALID; ++e) ++count;
      /// \endcode
      class EdgeIt : public Edge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        EdgeIt() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        EdgeIt(const EdgeIt& e) : Edge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        EdgeIt(Invalid) { }
        /// This constructor sets the iterator to the first edge.
    
        /// This constructor sets the iterator to the first edge of \c g.
        ///@param g the graph
        EdgeIt(const StaticGraph&) { }
        /// Edge -> EdgeIt conversion

        /// Sets the iterator to the value of the trivial iterator \c e.
        /// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        EdgeIt(const StaticGraph&, const Edge&) { } 
        ///Next edge
        
        /// Assign the iterator to the next edge.
        EdgeIt& operator++() { return *this; }
      };
      ///Gives back the target node of an edge.

      ///Gives back the target node of an edge.
      ///
      Node target(Edge) const { return INVALID; }
      ///Gives back the source node of an edge.

      ///Gives back the source node of an edge.
      ///
      Node source(Edge) const { return INVALID; }

      /// Gives back the first Node in the iterating order.
      
      /// Gives back the first Node in the iterating order.
      ///     
      void first(Node&) const {}

      /// Gives back the next Node in the iterating order.
      
      /// Gives back the next Node in the iterating order.
      ///     
      void next(Node&) const {}

      /// Gives back the first Edge in the iterating order.
      
      /// Gives back the first Edge in the iterating order.
      ///     
      void first(Edge&) const {}
      /// Gives back the next Edge in the iterating order.
      
      /// Gives back the next Edge in the iterating order.
      ///     
      void next(Edge&) const {}


      /// Gives back the first of the Edges point to the given Node.
      
      /// Gives back the first of the Edges point to the given Node.
      ///     
      void firstIn(Edge&, const Node&) const {}

      /// Gives back the next of the Edges points to the given Node.


      /// Gives back the next of the Edges points to the given Node.
      ///
      void nextIn(Edge&) const {}

      /// Gives back the first of the Edges start from the given Node.
      
      /// Gives back the first of the Edges start from the given Node.
      ///     
      void firstOut(Edge&, const Node&) const {}

      /// Gives back the next of the Edges start from the given Node.
      
      /// Gives back the next of the Edges start from the given Node.
      ///     
      void nextOut(Edge&) const {}

      /// \brief The base node of the iterator.
      ///
      /// Gives back the base node of the iterator.
      /// It is always the target of the pointed edge.
      Node baseNode(const InEdgeIt&) const { return INVALID; }

      /// \brief The running node of the iterator.
      ///
      /// Gives back the running node of the iterator.
      /// It is always the source of the pointed edge.
      Node runningNode(const InEdgeIt&) const { return INVALID; }

      /// \brief The base node of the iterator.
      ///
      /// Gives back the base node of the iterator.
      /// It is always the source of the pointed edge.
      Node baseNode(const OutEdgeIt&) const { return INVALID; }

      /// \brief The running node of the iterator.
      ///
      /// Gives back the running node of the iterator.
      /// It is always the target of the pointed edge.
      Node runningNode(const OutEdgeIt&) const { return INVALID; }

      /// \brief The opposite node on the given edge.
      ///
      /// Gives back the opposite node on the given edge.
      Node oppositeNode(const Node&, const Edge&) const { return INVALID; }

      /// \brief Read write map of the nodes to type \c T.
      /// 
      /// ReadWrite map of the nodes to type \c T.
      /// \sa Reference
      /// \warning Making maps that can handle bool type (NodeMap<bool>)
      /// needs some extra attention!
      template<class T> 
      class NodeMap : public ReadWriteMap< Node, T >
      {
      public:

        ///\e
        NodeMap(const StaticGraph&) { }
        ///\e
        NodeMap(const StaticGraph&, T) { }

        ///Copy constructor
        NodeMap(const NodeMap& nm) : ReadWriteMap< Node, T >(nm) { }
        ///Assignment operator
        NodeMap& operator=(const NodeMap&) { return *this; }
        // \todo fix this concept
      };

      /// \brief Read write map of the edges to type \c T.
      ///
      /// Reference map of the edges to type \c T.
      /// \sa Reference
      /// \warning Making maps that can handle bool type (EdgeMap<bool>)
      /// needs some extra attention!
      template<class T> 
      class EdgeMap : public ReadWriteMap<Edge,T>
      {
      public:

        ///\e
        EdgeMap(const StaticGraph&) { }
        ///\e
        EdgeMap(const StaticGraph&, T) { }
        ///Copy constructor
        EdgeMap(const EdgeMap& em) : ReadWriteMap<Edge,T>(em) { }
        ///Assignment operator
        EdgeMap& operator=(const EdgeMap&) { return *this; }
        // \todo fix this concept    
      };

      template <typename _Graph>
      struct Constraints : public _StaticGraph::Constraints<_Graph> {};

    };

    /// An empty non-static graph class.
    
    /// This class provides everything that \ref StaticGraph does.
    /// Additionally it enables building graphs from scratch.
    class ExtendableGraph : public StaticGraph
    {
    public:
      /// Defalult constructor.

      /// Defalult constructor.
      ///
      ExtendableGraph() { }
      ///Add a new node to the graph.

      /// \return the new node.
      ///
      Node addNode() { return INVALID; }
      ///Add a new edge to the graph.

      ///Add a new edge to the graph with source node \c s
      ///and target node \c t.
      ///\return the new edge.
      Edge addEdge(Node, Node) { return INVALID; }
    
      /// Resets the graph.

      /// This function deletes all edges and nodes of the graph.
      /// It also frees the memory allocated to store them.
      /// \todo It might belong to \ref ErasableGraph.
      void clear() { }

      template <typename _Graph>
      struct Constraints : public _ExtendableGraph::Constraints<_Graph> {};

    };

    /// An empty erasable graph class.
  
    /// This class is an extension of \ref ExtendableGraph. It makes it
    /// possible to erase edges or nodes.
    class ErasableGraph : public ExtendableGraph
    {
    public:
      /// Defalult constructor.

      /// Defalult constructor.
      ///
      ErasableGraph() { }
      /// Deletes a node.

      /// Deletes node \c n node.
      ///
      void erase(Node) { }
      /// Deletes an edge.

      /// Deletes edge \c e edge.
      ///
      void erase(Edge) { }

      template <typename _Graph>
      struct Constraints : public _ErasableGraph::Constraints<_Graph> {};

    };
    
    // @}
  } //namespace concept  
} //namespace lemon



#endif // LEMON_CONCEPT_GRAPH_H
