/* -*- C++ -*-
 *
 * lemon/concept/undir_graph_component.h - Part of LEMON, a generic
 * C++ optimization library
 *
 * Copyright (C) 2005 Egervary Jeno Kombinatorikus Optimalizalasi
 * Kutatocsoport (Egervary Research Group on Combinatorial Optimization,
 * EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

///\ingroup graph_concepts
///\file
///\brief Undirected graphs and components of.


#ifndef LEMON_CONCEPT_UNDIR_GRAPH_H
#define LEMON_CONCEPT_UNDIR_GRAPH_H

#include <lemon/concept/graph_component.h>
#include <lemon/concept/graph.h>
#include <lemon/utility.h>

namespace lemon {
  namespace concept {

    /// Skeleton class which describes an edge with direction in \ref
    /// UndirGraph "undirected graph".
    template <typename UndirGraph>
    class UndirGraphEdge : public UndirGraph::UndirEdge {
      typedef typename UndirGraph::UndirEdge UndirEdge;
      typedef typename UndirGraph::Node Node;
    public:

      /// \e
      UndirGraphEdge() {}

      /// \e
      UndirGraphEdge(const UndirGraphEdge& e) : UndirGraph::UndirEdge(e) {}

      /// \e
      UndirGraphEdge(Invalid) {}

      /// \brief Directed edge from undirected edge and a source node.
      ///
      /// Constructs a directed edge from undirected edge and a source node.
      ///
      /// \note You have to specify the graph for this constructor.
      UndirGraphEdge(const UndirGraph &g,
		     UndirEdge undir_edge, Node n) {
	ignore_unused_variable_warning(undir_edge);
	ignore_unused_variable_warning(g);
	ignore_unused_variable_warning(n);
      }

      /// \e
      UndirGraphEdge& operator=(UndirGraphEdge) { return *this; }

      /// \e
      bool operator==(UndirGraphEdge) const { return true; }
      /// \e
      bool operator!=(UndirGraphEdge) const { return false; }

      /// \e
      bool operator<(UndirGraphEdge) const { return false; }

      template <typename Edge>
      struct Constraints {
	void constraints() {
	  const_constraints();
	}
	void const_constraints() const {
	  /// \bug This should be is_base_and_derived ...
	  UndirEdge ue = e;
	  ue = e;

	  Edge e_with_source(graph,ue,n);
	  ignore_unused_variable_warning(e_with_source);
	}
	Edge e;
	UndirEdge ue;
	UndirGraph graph;
	Node n;
      };
    };
    

    struct BaseIterableUndirGraphConcept {

      template <typename Graph>
      struct Constraints {

	typedef typename Graph::UndirEdge UndirEdge;
	typedef typename Graph::Edge Edge;
	typedef typename Graph::Node Node;

	void constraints() {
	  checkConcept<BaseIterableGraphComponent, Graph>();
	  checkConcept<GraphItem<>, UndirEdge>();
	  //checkConcept<UndirGraphEdge<Graph>, Edge>();

	  graph.first(ue);
	  graph.next(ue);

	  const_constraints();
	}
	void const_constraints() {
	  Node n;
	  n = graph.target(ue);
	  n = graph.source(ue);
	  n = graph.oppositeNode(n0, ue);

	  bool b;
	  b = graph.direction(e);
	  Edge e = graph.direct(UndirEdge(), true);
	  e = graph.direct(UndirEdge(), n);
 
	  ignore_unused_variable_warning(b);
	}

	Graph graph;
	Edge e;
	Node n0;
	UndirEdge ue;
      };

    };


    struct IterableUndirGraphConcept {

      template <typename Graph>
      struct Constraints {
	void constraints() {
	  /// \todo we don't need the iterable component to be base iterable
	  /// Don't we really???
	  //checkConcept< BaseIterableUndirGraphConcept, Graph > ();

	  checkConcept<IterableGraphComponent, Graph> ();

	  typedef typename Graph::UndirEdge UndirEdge;
	  typedef typename Graph::UndirEdgeIt UndirEdgeIt;
	  typedef typename Graph::IncEdgeIt IncEdgeIt;

	  checkConcept<GraphIterator<Graph, UndirEdge>, UndirEdgeIt>();
	  checkConcept<GraphIncIterator<Graph, UndirEdge>, IncEdgeIt>();
	}
      };

    };

    struct MappableUndirGraphConcept {

      template <typename Graph>
      struct Constraints {

	struct Dummy {
	  int value;
	  Dummy() : value(0) {}
	  Dummy(int _v) : value(_v) {}
	};

	void constraints() {
	  checkConcept<MappableGraphComponent, Graph>();

	  typedef typename Graph::template UndirEdgeMap<int> IntMap;
	  checkConcept<GraphMap<Graph, typename Graph::UndirEdge, int>,
	    IntMap >();

	  typedef typename Graph::template UndirEdgeMap<bool> BoolMap;
	  checkConcept<GraphMap<Graph, typename Graph::UndirEdge, bool>,
	    BoolMap >();

	  typedef typename Graph::template UndirEdgeMap<Dummy> DummyMap;
	  checkConcept<GraphMap<Graph, typename Graph::UndirEdge, Dummy>,
	    DummyMap >();
	}
      };

    };

    struct ExtendableUndirGraphConcept {

      template <typename Graph>
      struct Constraints {
	void constraints() {
	  node_a = graph.addNode();
	  uedge = graph.addEdge(node_a, node_b);
	}
	typename Graph::Node node_a, node_b;
	typename Graph::UndirEdge uedge;
	Graph graph;
      };

    };

    struct ErasableUndirGraphConcept {

      template <typename Graph>
      struct Constraints {
	void constraints() {
	  graph.erase(n);
	  graph.erase(e);
	}
	Graph graph;
	typename Graph::Node n;
	typename Graph::UndirEdge e;
      };

    };

    /// \addtogroup graph_concepts
    /// @{


    /// Class describing the concept of Undirected Graphs.

    /// This class describes the common interface of all Undirected
    /// Graphs.
    ///
    /// As all concept describing classes it provides only interface
    /// without any sensible implementation. So any algorithm for
    /// undirected graph should compile with this class, but it will not
    /// run properly, of couse.
    ///
    /// In LEMON undirected graphs also fulfill the concept of directed
    /// graphs (\ref lemon::concept::Graph "Graph Concept"). For
    /// explanation of this and more see also the page \ref undir_graphs,
    /// a tutorial about undirected graphs.
    ///
    /// You can assume that all undirected graph can be handled
    /// as a static directed graph. This way it is fully conform
    /// to the StaticGraph concept.

    class UndirGraph {
    public:
      ///\e

      ///\todo undocumented
      ///
      typedef True UndirTag;

      /// The base type of node iterators, 
      /// or in other words, the trivial node iterator.

      /// This is the base type of each node iterator,
      /// thus each kind of node iterator converts to this.
      /// More precisely each kind of node iterator should be inherited 
      /// from the trivial node iterator.
      class Node {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        Node() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        Node(const Node&) { }

        /// Invalid constructor \& conversion.

        /// This constructor initializes the iterator to be invalid.
        /// \sa Invalid for more details.
        Node(Invalid) { }
        /// Equality operator

        /// Two iterators are equal if and only if they point to the
        /// same object or both are invalid.
        bool operator==(Node) const { return true; }

        /// Inequality operator
        
        /// \sa operator==(Node n)
        ///
        bool operator!=(Node) const { return true; }

	/// Artificial ordering operator.
	
	/// To allow the use of graph descriptors as key type in std::map or
	/// similar associative container we require this.
	///
	/// \note This operator only have to define some strict ordering of
	/// the items; this order has nothing to do with the iteration
	/// ordering of the items.
	///
	/// \bug This is a technical requirement. Do we really need this?
	bool operator<(Node) const { return false; }

      };
    
      /// This iterator goes through each node.

      /// This iterator goes through each node.
      /// Its usage is quite simple, for example you can count the number
      /// of nodes in graph \c g of type \c Graph like this:
      /// \code
      /// int count=0;
      /// for (Graph::NodeIt n(g); n!=INVALID; ++n) ++count;
      /// \endcode
      class NodeIt : public Node {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        NodeIt() { }
        /// Copy constructor.
        
        /// Copy constructor.
        ///
        NodeIt(const NodeIt& n) : Node(n) { }
        /// Invalid constructor \& conversion.

        /// Initialize the iterator to be invalid.
        /// \sa Invalid for more details.
        NodeIt(Invalid) { }
        /// Sets the iterator to the first node.

        /// Sets the iterator to the first node of \c g.
        ///
        NodeIt(const UndirGraph&) { }
        /// Node -> NodeIt conversion.

        /// Sets the iterator to the node of \c the graph pointed by 
	/// the trivial iterator.
        /// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        NodeIt(const UndirGraph&, const Node&) { }
        /// Next node.

        /// Assign the iterator to the next node.
        ///
        NodeIt& operator++() { return *this; }
      };
    
    
      /// The base type of the undirected edge iterators.

      /// The base type of the undirected edge iterators.
      ///
      class UndirEdge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        UndirEdge() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        UndirEdge(const UndirEdge&) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        UndirEdge(Invalid) { }
        /// Equality operator

        /// Two iterators are equal if and only if they point to the
        /// same object or both are invalid.
        bool operator==(UndirEdge) const { return true; }
        /// Inequality operator

        /// \sa operator==(UndirEdge n)
        ///
        bool operator!=(UndirEdge) const { return true; }

	/// Artificial ordering operator.
	
	/// To allow the use of graph descriptors as key type in std::map or
	/// similar associative container we require this.
	///
	/// \note This operator only have to define some strict ordering of
	/// the items; this order has nothing to do with the iteration
	/// ordering of the items.
	///
	/// \bug This is a technical requirement. Do we really need this?
	bool operator<(UndirEdge) const { return false; }
      };

      /// This iterator goes through each undirected edge.

      /// This iterator goes through each undirected edge of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of undirected edges in a graph \c g of type \c Graph as follows:
      /// \code
      /// int count=0;
      /// for(Graph::UndirEdgeIt e(g); e!=INVALID; ++e) ++count;
      /// \endcode
      class UndirEdgeIt : public UndirEdge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        UndirEdgeIt() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        UndirEdgeIt(const UndirEdgeIt& e) : UndirEdge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        UndirEdgeIt(Invalid) { }
        /// This constructor sets the iterator to the first undirected edge.
    
        /// This constructor sets the iterator to the first undirected edge.
        UndirEdgeIt(const UndirGraph&) { }
        /// UndirEdge -> UndirEdgeIt conversion

        /// Sets the iterator to the value of the trivial iterator.
        /// This feature necessitates that each time we
        /// iterate the undirected edge-set, the iteration order is the 
	/// same.
        UndirEdgeIt(const UndirGraph&, const UndirEdge&) { } 
        /// Next undirected edge
        
        /// Assign the iterator to the next undirected edge.
        UndirEdgeIt& operator++() { return *this; }
      };

      /// \brief This iterator goes trough the incident undirected 
      /// edges of a node.
      ///
      /// This iterator goes trough the incident undirected edges
      /// of a certain node
      /// of a graph.
      /// Its usage is quite simple, for example you can compute the
      /// degree (i.e. count the number
      /// of incident edges of a node \c n
      /// in graph \c g of type \c Graph as follows.
      /// \code
      /// int count=0;
      /// for(Graph::IncEdgeIt e(g, n); e!=INVALID; ++e) ++count;
      /// \endcode
      class IncEdgeIt : public UndirEdge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        IncEdgeIt() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        IncEdgeIt(const IncEdgeIt& e) : UndirEdge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        IncEdgeIt(Invalid) { }
        /// This constructor sets the iterator to first incident edge.
    
        /// This constructor set the iterator to the first incident edge of
        /// the node.
        IncEdgeIt(const UndirGraph&, const Node&) { }
        /// UndirEdge -> IncEdgeIt conversion

        /// Sets the iterator to the value of the trivial iterator \c e.
        /// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        IncEdgeIt(const UndirGraph&, const UndirEdge&) { }
        /// Next incident edge

        /// Assign the iterator to the next incident edge
	/// of the corresponding node.
        IncEdgeIt& operator++() { return *this; }
      };

      /// The directed edge type.

      /// The directed edge type. It can be converted to the
      /// undirected edge.
      class Edge : public UndirEdge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        Edge() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        Edge(const Edge& e) : UndirEdge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        Edge(Invalid) { }
        /// Equality operator

        /// Two iterators are equal if and only if they point to the
        /// same object or both are invalid.
        bool operator==(Edge) const { return true; }
        /// Inequality operator

        /// \sa operator==(Edge n)
        ///
        bool operator!=(Edge) const { return true; }

	/// Artificial ordering operator.
	
	/// To allow the use of graph descriptors as key type in std::map or
	/// similar associative container we require this.
	///
	/// \note This operator only have to define some strict ordering of
	/// the items; this order has nothing to do with the iteration
	/// ordering of the items.
	///
	/// \bug This is a technical requirement. Do we really need this?
	bool operator<(Edge) const { return false; }
	
      }; 
      /// This iterator goes through each directed edge.

      /// This iterator goes through each edge of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of edges in a graph \c g of type \c Graph as follows:
      /// \code
      /// int count=0;
      /// for(Graph::EdgeIt e(g); e!=INVALID; ++e) ++count;
      /// \endcode
      class EdgeIt : public Edge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        EdgeIt() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        EdgeIt(const EdgeIt& e) : Edge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        EdgeIt(Invalid) { }
        /// This constructor sets the iterator to the first edge.
    
        /// This constructor sets the iterator to the first edge of \c g.
        ///@param g the graph
        EdgeIt(const UndirGraph&) { }
        /// Edge -> EdgeIt conversion

        /// Sets the iterator to the value of the trivial iterator \c e.
        /// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        EdgeIt(const UndirGraph&, const Edge&) { } 
        ///Next edge
        
        /// Assign the iterator to the next edge.
        EdgeIt& operator++() { return *this; }
      };
   
      /// This iterator goes trough the outgoing directed edges of a node.

      /// This iterator goes trough the \e outgoing edges of a certain node
      /// of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of outgoing edges of a node \c n
      /// in graph \c g of type \c Graph as follows.
      /// \code
      /// int count=0;
      /// for (Graph::OutEdgeIt e(g, n); e!=INVALID; ++e) ++count;
      /// \endcode
    
      class OutEdgeIt : public Edge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        OutEdgeIt() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        OutEdgeIt(const OutEdgeIt& e) : Edge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        OutEdgeIt(Invalid) { }
        /// This constructor sets the iterator to the first outgoing edge.
    
        /// This constructor sets the iterator to the first outgoing edge of
        /// the node.
        ///@param n the node
        ///@param g the graph
        OutEdgeIt(const UndirGraph&, const Node&) { }
        /// Edge -> OutEdgeIt conversion

        /// Sets the iterator to the value of the trivial iterator.
	/// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        OutEdgeIt(const UndirGraph&, const Edge&) { }
        ///Next outgoing edge
        
        /// Assign the iterator to the next 
        /// outgoing edge of the corresponding node.
        OutEdgeIt& operator++() { return *this; }
      };

      /// This iterator goes trough the incoming directed edges of a node.

      /// This iterator goes trough the \e incoming edges of a certain node
      /// of a graph.
      /// Its usage is quite simple, for example you can count the number
      /// of outgoing edges of a node \c n
      /// in graph \c g of type \c Graph as follows.
      /// \code
      /// int count=0;
      /// for(Graph::InEdgeIt e(g, n); e!=INVALID; ++e) ++count;
      /// \endcode

      class InEdgeIt : public Edge {
      public:
        /// Default constructor

        /// @warning The default constructor sets the iterator
        /// to an undefined value.
        InEdgeIt() { }
        /// Copy constructor.

        /// Copy constructor.
        ///
        InEdgeIt(const InEdgeIt& e) : Edge(e) { }
        /// Initialize the iterator to be invalid.

        /// Initialize the iterator to be invalid.
        ///
        InEdgeIt(Invalid) { }
        /// This constructor sets the iterator to first incoming edge.
    
        /// This constructor set the iterator to the first incoming edge of
        /// the node.
        ///@param n the node
        ///@param g the graph
        InEdgeIt(const UndirGraph&, const Node&) { }
        /// Edge -> InEdgeIt conversion

        /// Sets the iterator to the value of the trivial iterator \c e.
        /// This feature necessitates that each time we 
        /// iterate the edge-set, the iteration order is the same.
        InEdgeIt(const UndirGraph&, const Edge&) { }
        /// Next incoming edge

        /// Assign the iterator to the next inedge of the corresponding node.
        ///
        InEdgeIt& operator++() { return *this; }
      };

      /// \brief Read write map of the nodes to type \c T.
      /// 
      /// ReadWrite map of the nodes to type \c T.
      /// \sa Reference
      /// \warning Making maps that can handle bool type (NodeMap<bool>)
      /// needs some extra attention!
      template<class T> 
      class NodeMap : public ReadWriteMap< Node, T >
      {
      public:

        ///\e
        NodeMap(const UndirGraph&) { }
        ///\e
        NodeMap(const UndirGraph&, T) { }

        ///Copy constructor
        NodeMap(const NodeMap& nm) : ReadWriteMap< Node, T >(nm) { }
        ///Assignment operator
        NodeMap& operator=(const NodeMap&) { return *this; }
        // \todo fix this concept
      };

      /// \brief Read write map of the directed edges to type \c T.
      ///
      /// Reference map of the directed edges to type \c T.
      /// \sa Reference
      /// \warning Making maps that can handle bool type (EdgeMap<bool>)
      /// needs some extra attention!
      template<class T> 
      class EdgeMap : public ReadWriteMap<Edge,T>
      {
      public:

        ///\e
        EdgeMap(const UndirGraph&) { }
        ///\e
        EdgeMap(const UndirGraph&, T) { }
        ///Copy constructor
        EdgeMap(const EdgeMap& em) : ReadWriteMap<Edge,T>(em) { }
        ///Assignment operator
        EdgeMap& operator=(const EdgeMap&) { return *this; }
        // \todo fix this concept    
      };

      /// Read write map of the undirected edges to type \c T.

      /// Reference map of the edges to type \c T.
      /// \sa Reference
      /// \warning Making maps that can handle bool type (UndirEdgeMap<bool>)
      /// needs some extra attention!
      template<class T> 
      class UndirEdgeMap : public ReadWriteMap<UndirEdge,T>
      {
      public:

        ///\e
        UndirEdgeMap(const UndirGraph&) { }
        ///\e
        UndirEdgeMap(const UndirGraph&, T) { }
        ///Copy constructor
        UndirEdgeMap(const UndirEdgeMap& em) : ReadWriteMap<UndirEdge,T>(em) {}
        ///Assignment operator
        UndirEdgeMap &operator=(const UndirEdgeMap&) { return *this; }
        // \todo fix this concept    
      };

      /// \brief Direct the given undirected edge.
      ///
      /// Direct the given undirected edge. The returned edge source
      /// will be the given edge.
      Edge direct(const UndirEdge&, const Node&) const {
	return INVALID;
      }

      /// \brief Direct the given undirected edge.
      ///
      /// Direct the given undirected edge. The returned edge source
      /// will be the source of the undirected edge if the given bool
      /// is true.
      Edge direct(const UndirEdge&, bool) const {
	return INVALID;
      }

      /// \brief Returns true if the edge has default orientation.
      ///
      /// Returns whether the given directed edge is same orientation as
      /// the corresponding undirected edge.
      bool direction(Edge) const { return true; }

      /// \brief Returns the opposite directed edge.
      ///
      /// Returns the opposite directed edge.
      Edge oppositeEdge(Edge) const { return INVALID; }

      /// \brief Opposite node on an edge
      ///
      /// \return the opposite of the given Node on the given Edge
      Node oppositeNode(Node, UndirEdge) const { return INVALID; }

      /// \brief First node of the undirected edge.
      ///
      /// \return the first node of the given UndirEdge.
      ///
      /// Naturally undirectected edges don't have direction and thus
      /// don't have source and target node. But we use these two methods
      /// to query the two endnodes of the edge. The direction of the edge
      /// which arises this way is called the inherent direction of the
      /// undirected edge, and is used to define the "default" direction
      /// of the directed versions of the edges.
      /// \sa direction
      Node source(UndirEdge) const { return INVALID; }

      /// \brief Second node of the undirected edge.
      Node target(UndirEdge) const { return INVALID; }

      /// \brief Source node of the directed edge.
      Node source(Edge) const { return INVALID; }

      /// \brief Target node of the directed edge.
      Node target(Edge) const { return INVALID; }

      /// \brief First node of the graph
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void first(Node&) const {}
      /// \brief Next node of the graph
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void next(Node&) const {}

      /// \brief First undirected edge of the graph
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void first(UndirEdge&) const {}
      /// \brief Next undirected edge of the graph
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void next(UndirEdge&) const {}

      /// \brief First directed edge of the graph
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void first(Edge&) const {}
      /// \brief Next directed edge of the graph
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void next(Edge&) const {}

      /// \brief First outgoing edge from a given node
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void firstOut(Edge&, Node) const {}
      /// \brief Next outgoing edge to a node
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void nextOut(Edge&) const {}

      /// \brief First incoming edge to a given node
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void firstIn(Edge&, Node) const {}
      /// \brief Next incoming edge to a node
      ///
      /// \note This method is part of so called \ref
      /// developpers_interface "Developpers' interface", so it shouldn't
      /// be used in an end-user program.
      void nextIn(Edge&) const {}


      /// \brief Base node of the iterator
      ///
      /// Returns the base node (the source in this case) of the iterator
      Node baseNode(OutEdgeIt e) const {
	return source(e);
      }
      /// \brief Running node of the iterator
      ///
      /// Returns the running node (the target in this case) of the
      /// iterator
      Node runningNode(OutEdgeIt e) const {
	return target(e);
      }

      /// \brief Base node of the iterator
      ///
      /// Returns the base node (the target in this case) of the iterator
      Node baseNode(InEdgeIt e) const {
	return target(e);
      }
      /// \brief Running node of the iterator
      ///
      /// Returns the running node (the source in this case) of the
      /// iterator
      Node runningNode(InEdgeIt e) const {
	return source(e);
      }

      /// \brief Base node of the iterator
      ///
      /// Returns the base node of the iterator
      Node baseNode(IncEdgeIt) const {
	return INVALID;
      }
      
      /// \brief Running node of the iterator
      ///
      /// Returns the running node of the iterator
      Node runningNode(IncEdgeIt) const {
	return INVALID;
      }

      template <typename Graph>
      struct Constraints {
	void constraints() {
	  checkConcept<BaseIterableUndirGraphConcept, Graph>();
	  checkConcept<IterableUndirGraphConcept, Graph>();
	  checkConcept<MappableUndirGraphConcept, Graph>();
	}
      };

    };

    /// \brief An empty non-static undirected graph class.
    ///    
    /// This class provides everything that \ref UndirGraph does.
    /// Additionally it enables building graphs from scratch.
    class ExtendableUndirGraph : public UndirGraph {
    public:
      
      /// \brief Add a new node to the graph.
      ///
      /// Add a new node to the graph.
      /// \return the new node.
      Node addNode();

      /// \brief Add a new undirected edge to the graph.
      ///
      /// Add a new undirected edge to the graph.
      /// \return the new edge.
      UndirEdge addEdge(const Node& from, const Node& to);

      /// \brief Resets the graph.
      ///
      /// This function deletes all undirected edges and nodes of the graph.
      /// It also frees the memory allocated to store them.
      void clear() { }

      template <typename Graph>
      struct Constraints {
	void constraints() {
	  checkConcept<BaseIterableUndirGraphConcept, Graph>();
	  checkConcept<IterableUndirGraphConcept, Graph>();
	  checkConcept<MappableUndirGraphConcept, Graph>();

	  checkConcept<UndirGraph, Graph>();
	  checkConcept<ExtendableUndirGraphConcept, Graph>();
	  checkConcept<ClearableGraphComponent, Graph>();
	}
      };

    };

    /// \brief An empty erasable undirected graph class.
    ///
    /// This class is an extension of \ref ExtendableUndirGraph. It makes it
    /// possible to erase undirected edges or nodes.
    class ErasableUndirGraph : public ExtendableUndirGraph {
    public:

      /// \brief Deletes a node.
      ///
      /// Deletes a node.
      ///
      void erase(Node) { }
      /// \brief Deletes an undirected edge.
      ///
      /// Deletes an undirected edge.
      ///
      void erase(UndirEdge) { }

      template <typename Graph>
      struct Constraints {
	void constraints() {
	  checkConcept<ExtendableUndirGraph, Graph>();
	  checkConcept<ErasableUndirGraphConcept, Graph>();
	}
      };

    };

    /// @}

  }

}

#endif
