// -*- c++ -*-
/**
2 dimensional vector (plainvector) implementation

*/
#ifndef HUGO_XY_H
#define HUGO_XY_H

#include <iostream>

namespace hugo {

  template<typename T>
    class xy {

    public:

      T x,y;     
      
      ///Default constructor: both coordinates become 0
      xy() : x(0), y(0) {}

      ///Constructing the instance from coordinates
      xy(T a, T b) : x(a), y(a) { }


      ///Gives back the square of the norm of the vector
      T normSquare(){
	return x*x+y*y;
      };
  
      ///Increments the left hand side by u
      xy<T>& operator +=(const xy<T>& u){
	x += u.x;
	y += u.y;
	return *this;
      };
  
      ///Decrements the left hand side by u
      xy<T>& operator -=(const xy<T>& u){
	x -= u.x;
	y -= u.y;
	return *this;
      };

      ///Multiplying the left hand side with a scalar
      xy<T>& operator *=(const T &u){
	x *= u;
	y *= u;
	return *this;
      };

      ///Dividing the left hand side by a scalar
      xy<T>& operator /=(const T &u){
	x /= u;
	y /= u;
	return *this;
      };
  
      ///Returns the scalar product of two vectors
      T operator *(const xy<T>& u){
	return x*u.x+y*u.y;
      };
  
      ///Returns the sum of two vectors
      xy<T> operator+(const xy<T> &u) const {
	xy<T> b=*this;
	return b+=u;
      };

      ///Returns the difference of two vectors
      xy<T> operator-(const xy<T> &u) const {
	xy<T> b=*this;
	return b-=u;
      };

      ///Returns a vector multiplied by a scalar
      xy<T> operator*(const T &u) const {
	xy<T> b=*this;
	return b*=u;
      };

      ///Returns a vector divided by a scalar
      xy<T> operator/(const T &u) const {
	xy<T> b=*this;
	return b/=u;
      };

      ///Testing equality
      bool operator==(const xy<T> &u){
	return (x==u.x) && (y==u.y);
      };

      ///Testing inequality
      bool operator!=(xy u){
	return  (x!=u.x) || (y!=u.y);
      };

    };

  ///Reading a plainvector from a stream
  template<typename T>
  inline
  std::istream& operator>>(std::istream &is, xy<T> &z)
  {

    is >> z.x >> z.y;
    return is;
  }

  ///Outputting a plainvector to a stream
  template<typename T>
  inline
  std::ostream& operator<<(std::ostream &os, xy<T> z)
  {
    os << "(" << z.x << ", " << z.y << ")";
    return os;
  }

} //namespace hugo

#endif //HUGO_XY_H
