/* -*- C++ -*-
 * src/test/dijkstra_heap_test.cc - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2005 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

//Tests dijsktra.h with two heap implementations:
//the default binary heap of bin_heap.h, and the 
//Fibonacci heap of fib_heap.h.

//The input is a graph in standard dimacs format from the standard input (like
//in /lemon_loc/testfiles/dimacs). It runs dijkstra.h on this graph with both
//heaps, checking two postconditions:

//- if the edges e=uv of the shortest path tree reported by dijkstra.h have
//dist(v)-dist(u)=length(e)

// - if all edges e=uv with u reachable from the root have
//dist(v)-dist(u)>=length(e)
#include <iostream>
#include <math.h>

#include <lemon/smart_graph.h>

#include <lemon/graph_writer.h>
#include <lemon/map_utils.h>


#include <lemon/dimacs.h>
#include <lemon/dijkstra.h>
#include <lemon/time_measure.h>
#include <lemon/graph_utils.h>

#include <lemon/bin_heap.h>
#include <lemon/fib_heap.h>
#include <lemon/radix_heap.h>

using namespace lemon;

typedef ListGraph Graph;

typedef Graph::Edge Edge;
typedef Graph::Node Node;
typedef Graph::EdgeIt EdgeIt;
typedef Graph::NodeIt NodeIt;
typedef Graph::EdgeMap<int> LengthMap;


struct FibTraits : public DijkstraDefaultTraits<Graph, LengthMap> {
  typedef FibHeap<Graph::Node, LengthMap::Value, Graph::NodeMap<int> > Heap;
};

struct RadixTraits : public DijkstraDefaultTraits<Graph, LengthMap> {
  typedef RadixHeap<Graph::Node, Graph::NodeMap<int> > Heap;
};


int main(int argc, const char* argv[]) {
  

  Graph graph;
  Node s, t;
  LengthMap cap(graph);
  readDimacs(std::cin, graph, cap, s, t);
  Timer ts;

  GraphWriter<Graph> writer(std::cout, graph);

  IdMap<Graph, Node> nodeIdMap(graph);
  writer.addNodeMap("id", nodeIdMap);

  IdMap<Graph, Edge> edgeIdMap(graph);
  writer.addEdgeMap("id", edgeIdMap);

  writer.addEdgeMap("cap", cap);

  writer.run();
    
  std::cout <<
    "\n  Testing dijkstra.h with binary heap implementation bin_heap.h,"
	    <<std::endl;
  std::cout<<"  on a graph with " << 
    countNodes(graph) << " nodes and " << countEdges(graph) << " edges..."
	   << std::endl<<std::endl;


  Dijkstra<Graph, LengthMap> 
    dijkstra_test(graph, cap);
  ts.reset();
   dijkstra_test.run(s);
  std::cout << "elapsed time: " << ts << std::endl;

  int error1=0;
  int error2=0;

  for(EdgeIt e(graph); e!=INVALID; ++e) {
    Node u=graph.source(e); 
    Node v=graph.target(e);
    if ( dijkstra_test.dist(v) - dijkstra_test.dist(u) > cap[e] )
      if ( dijkstra_test.reached(u) ) {
	std::cout<<"Error! dist(target)-dist(source)- edge_length= " 
		 <<dijkstra_test.dist(v) - dijkstra_test.dist(u) 
	  - cap[e]<<std::endl;
	++error1;
      }
  }

  NodeIt v;
  for(NodeIt v(graph); v!=INVALID; ++v) {
    if ( dijkstra_test.reached(v) ) {
      Edge e=dijkstra_test.pred(v);
      if(e!=INVALID) {
	Node u=graph.source(e);
	if ( dijkstra_test.dist(v) - dijkstra_test.dist(u) != cap[e] ) {
	  std::cout<<"Error in a shortest path tree edge! Difference: " 
		   <<std::abs(dijkstra_test.dist(v) - dijkstra_test.dist(u) 
			      - cap[e])<<std::endl;
	  ++error2;
	}
      }
    }
  }
  

  
  std::cout << error1 << " non-tree and " << error2 
	    << " shortest path tree edge is erroneous."<<std::endl;



  std::cout <<
    "\n\n  Testing dijkstra.h with Fibonacci heap implementation fib_heap.h,"
	    <<std::endl;
  std::cout<<"  on a graph with " << 
    countNodes(graph) << " nodes and " << countEdges(graph) << " edges..."
	   << std::endl<<std::endl;
  

  Dijkstra<Graph, LengthMap, FibTraits> 
    dijkstra_test2(graph, cap);
  ts.reset();
  dijkstra_test2.run(s);
  std::cout << "elapsed time: " << ts << std::endl;
  
  error1=0;
  error2=0;

  for(EdgeIt e(graph); e!=INVALID; ++e) {
    Node u=graph.source(e);
    Node v=graph.target(e);
    if ( dijkstra_test2.dist(v) - dijkstra_test2.dist(u) > cap[e] )
      if ( dijkstra_test2.reached(u) ) {
	std::cout<<"Error! dist(target)-dist(source)- edge_length= " 
		 <<dijkstra_test2.dist(v) - dijkstra_test2.dist(u) 
	  - cap[e]<<std::endl;
	++error1;
      }
  }

  for(NodeIt n(graph); v!=INVALID; ++v) {
    if ( dijkstra_test2.reached(v) ) {
      Edge e=dijkstra_test2.pred(v);
      if(e!=INVALID) {
	Node u=graph.source(e);
	if ( dijkstra_test2.dist(v) - dijkstra_test2.dist(u) != cap[e] ) {
	  std::cout<<"Error in a shortest path tree edge! Difference: " 
		   <<std::abs(dijkstra_test2.dist(v) - dijkstra_test2.dist(u) 
			      - cap[e])<<std::endl;
	  ++error2;
	}
      }
    }
  }
  

  std::cout << error1 << " non-tree and " << error2 
	    << " shortest path tree edge is erroneous."<<std::endl;

  std::cout <<
    "\n\n  Testing dijkstra.h with Radix heap implementation radix_heap.h,"
	    <<std::endl;
  std::cout<<"  on a graph with " << 
    countNodes(graph) << " nodes and " << countEdges(graph) << " edges..."
	   << std::endl<<std::endl;


  Dijkstra<Graph, LengthMap, RadixTraits> 
    dijkstra_test3(graph, cap);
  ts.reset();
  dijkstra_test3.run(s);
  std::cout << "elapsed time: " << ts << std::endl;
  

  error1=0;
  error2=0;

  for(EdgeIt e(graph); e!=INVALID; ++e) {
    Node u=graph.source(e);
    Node v=graph.target(e);
    if ( dijkstra_test3.dist(v) - dijkstra_test3.dist(u) > cap[e] )
      if ( dijkstra_test3.reached(u) ) {
	std::cout<<"Error! dist(target)-dist(source)- edge_length= " 
		 <<dijkstra_test3.dist(v) - dijkstra_test3.dist(u) 
	  - cap[e]<<std::endl;
	++error1;
      }
  }

  for(NodeIt n(graph); v!=INVALID; ++v) {
    if ( dijkstra_test3.reached(v) ) {
      Edge e=dijkstra_test3.pred(v);
      if(e!=INVALID) {
	Node u=graph.source(e);
	if ( dijkstra_test3.dist(v) - dijkstra_test3.dist(u) != cap[e] ) {
	  std::cout<<"Error in a shortest path tree edge! Difference: " 
		   <<std::abs(dijkstra_test3.dist(v) - dijkstra_test3.dist(u) 
			      - cap[e])<<std::endl;
	  ++error2;
	}
      }
    }
  }
  

  std::cout << error1 << " non-tree and " << error2 
	    << " shortest path tree edge is erroneous."<<std::endl;

  return 0;
}
