/* -*- mode: C++; indent-tabs-mode: nil; -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library.
 *
 * Copyright (C) 2003-2008
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

///\file
///\brief Demonstrating the basic concepts and fetures of LEMON
///
/// This program is intended to be a "Hello World!" program that shows
/// the very basic notions of LEMON: \ref graphs "graphs" and
/// \ref maps "maps".
///
/// \include hello_lemon.cc

#include <iostream>
#include <lemon/list_graph.h>

int main()
{
  // Convenience type definitions
  typedef lemon::ListDigraph Digraph;
  typedef Digraph::Node Node;
  typedef Digraph::Arc Arc;
  typedef Digraph::NodeIt NodeIt;
  typedef Digraph::ArcIt ArcIt;
  typedef Digraph::ArcMap<int> LengthMap;
  using lemon::INVALID;

  // Create a directed graph
  Digraph g;

  // Add nodes to the digraph
  Node v1 = g.addNode();
  Node v2 = g.addNode();
  Node v3 = g.addNode();
  Node v4 = g.addNode();

  // Add arcs to the digraph
  Arc v1_v2 = g.addArc(v1, v2);
  Arc v1_v3 = g.addArc(v1, v3);
  Arc v2_v3 = g.addArc(v2, v3);
  Arc v2_v4 = g.addArc(v2, v4);
  Arc v3_v4 = g.addArc(v3, v4);

  // Create an arc map (length)
  LengthMap length(g);

  // Set the length of each arc
  length[v1_v2]  = 10;
  length[v1_v3]  = 20;
  length[v2_v3] = 5;
  length[v2_v4] = 25;
  length[v3_v4] = 10;

  // Welcome message
  std::cout << "Hello World!" << std::endl;
  std::cout << "This is LEMON library here. We have a direceted graph.";
  std::cout << std::endl << std::endl;

  // Iterate through the nodes and print their IDs
  std::cout << "Nodes:";
  for (NodeIt n(g); n != INVALID; ++n)
    std::cout << " " << g.id(n);
  std::cout << std::endl;

  // Iterate through the arcs and print the IDs of their
  // source and target nodes
  std::cout << "Arcs:";
  for (ArcIt a(g); a != INVALID; ++a)
    std::cout << " (" << g.id(g.source(a)) << ","
              << g.id(g.target(a)) << ")";
  std::cout << std::endl << std::endl;

  // Iterate through the arcs and print their length
  std::cout << "There is a map on the arcs (length):" << std::endl;
  std::cout << std::endl;
  for (ArcIt a(g); a != INVALID; ++a)
    std::cout << "length(" << g.id(g.source(a)) << ","
              << g.id(g.target(a)) << ")=" << length[a] << std::endl;
  std::cout << std::endl;

  return 0;
}
