/* -*- C++ -*-
 * src/lemon/dijkstra.h - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2004 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef LEMON_DIJKSTRA_H
#define LEMON_DIJKSTRA_H

///\ingroup flowalgs
///\file
///\brief Dijkstra algorithm.

#include <lemon/list_graph.h>
#include <lemon/bin_heap.h>
#include <lemon/invalid.h>

namespace lemon {

/// \addtogroup flowalgs
/// @{

  ///Default traits class of Dijkstra class.

  ///Default traits class of Dijkstra class.
  ///\param GR Graph type.
  ///\param LM Type of length map.
  template<class GR, class LM>
  struct DijkstraDefaultTraits
  {
    ///The graph type the algorithm runs on. 
    typedef GR Graph;
    ///The type of the map that stores the edge lengths.

    ///It must meet the \ref concept::ReadMap "ReadMap" concept.
    ///
    typedef LM LengthMap;
    //The type of the length of the edges.
    typedef typename LM::Value Value;
    ///The heap type used by Dijkstra algorithm.

    ///The heap type used by Dijkstra algorithm.
    ///
    ///\sa BinHeap
    ///\sa Dijkstra
    typedef BinHeap<typename Graph::Node,
		    typename LM::Value,
		    typename GR::template NodeMap<int>,
		    std::less<Value> > Heap;

    ///\brief The type of the map that stores the last
    ///edges of the shortest paths.
    /// 
    ///It must meet the \ref concept::WriteMap "WriteMap" concept.
    ///
    typedef typename Graph::template NodeMap<typename GR::Edge> PredMap;
    ///Instantiates a PredMap.
 
    ///\todo Please document...
    ///
    static PredMap *createPredMap(const GR &G) 
    {
      return new PredMap(G);
    }
    ///\brief The type of the map that stores the last but one
    ///nodes of the shortest paths.
    ///
    ///It must meet the \ref concept::WriteMap "WriteMap" concept.
    ///
    typedef typename Graph::template NodeMap<typename GR::Node> PredNodeMap;
    ///Instantiates a PredNodeMap.
 
    ///\todo Please document...
    ///
    static PredNodeMap *createPredNodeMap(const GR &G)
    {
      return new PredNodeMap(G);
    }
    ///The type of the map that stores the dists of the nodes.
 
    ///It must meet the \ref concept::WriteMap "WriteMap" concept.
    ///
    typedef typename Graph::template NodeMap<typename LM::Value> DistMap;
    ///Instantiates a DistMap.
 
    ///\todo Please document...
    ///
    static DistMap *createDistMap(const GR &G)
    {
      return new DistMap(G);
    }
  };
  
  ///%Dijkstra algorithm class.

  ///This class provides an efficient implementation of %Dijkstra algorithm.
  ///The edge lengths are passed to the algorithm using a
  ///\ref concept::ReadMap "ReadMap",
  ///so it is easy to change it to any kind of length.
  ///
  ///The type of the length is determined by the
  ///\ref concept::ReadMap::Value "Value" of the length map.
  ///
  ///It is also possible to change the underlying priority heap.
  ///
  ///\param GR The graph type the algorithm runs on. The default value is
  ///\ref ListGraph. The value of GR is not used directly by Dijkstra, it
  ///is only passed to \ref DijkstraDefaultTraits.
  ///\param LM This read-only
  ///EdgeMap
  ///determines the
  ///lengths of the edges. It is read once for each edge, so the map
  ///may involve in relatively time consuming process to compute the edge
  ///length if it is necessary. The default map type is
  ///\ref concept::StaticGraph::EdgeMap "Graph::EdgeMap<int>".
  ///The value of LM is not used directly by Dijkstra, it
  ///is only passed to \ref DijkstraDefaultTraits.
  ///\param TR Traits class to set various data types used by the algorithm.
  ///The default traits class is
  ///\ref DijkstraDefaultTraits "DijkstraDefaultTraits<GR,LM>".
  ///See \ref DijkstraDefaultTraits for the documentation of
  ///a Dijkstra traits class.
  ///
  ///\author Jacint Szabo and Alpar Juttner
  ///\todo We need a typedef-names should be standardized. (-:

#ifdef DOXYGEN
  template <typename GR,
	    typename LM,
	    typename TR>
#else
  template <typename GR=ListGraph,
	    typename LM=typename GR::template EdgeMap<int>,
	    typename TR=DijkstraDefaultTraits<GR,LM> >
#endif
  class Dijkstra{
  public:
    typedef TR Traits;
    ///The type of the underlying graph.
    typedef typename TR::Graph Graph;
    ///\e
    typedef typename Graph::Node Node;
    ///\e
    typedef typename Graph::NodeIt NodeIt;
    ///\e
    typedef typename Graph::Edge Edge;
    ///\e
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    
    ///The type of the length of the edges.
    typedef typename TR::LengthMap::Value Value;
    ///The type of the map that stores the edge lengths.
    typedef typename TR::LengthMap LengthMap;
    ///\brief The type of the map that stores the last
    ///edges of the shortest paths.
    typedef typename TR::PredMap PredMap;
    ///\brief The type of the map that stores the last but one
    ///nodes of the shortest paths.
    typedef typename TR::PredNodeMap PredNodeMap;
    ///The type of the map that stores the dists of the nodes.
    typedef typename TR::DistMap DistMap;
    ///The heap type used by the dijkstra algorithm.
    typedef typename TR::Heap Heap;
  private:
    /// Pointer to the underlying graph.
    const Graph *G;
    /// Pointer to the length map
    const LengthMap *length;
    ///Pointer to the map of predecessors edges.
    PredMap *predecessor;
    ///Indicates if \ref predecessor is locally allocated (\c true) or not.
    bool local_predecessor;
    ///Pointer to the map of predecessors nodes.
    PredNodeMap *pred_node;
    ///Indicates if \ref pred_node is locally allocated (\c true) or not.
    bool local_pred_node;
    ///Pointer to the map of distances.
    DistMap *distance;
    ///Indicates if \ref distance is locally allocated (\c true) or not.
    bool local_distance;

    ///The source node of the last execution.
    Node source;

    ///Initializes the maps.
    
    ///\todo Error if \c G or are \c NULL. What about \c length?
    ///\todo Better memory allocation (instead of new).
    void init_maps() 
    {
      if(!predecessor) {
	local_predecessor = true;
	predecessor = Traits::createPredMap(*G);
      }
      if(!pred_node) {
	local_pred_node = true;
	pred_node = Traits::createPredNodeMap(*G);
      }
      if(!distance) {
	local_distance = true;
	distance = Traits::createDistMap(*G);
      }
    }
    
  public :

    template <class T>
    struct SetPredMapTraits : public Traits {
      typedef T PredMap;
      ///\todo An exception should be thrown.
      ///
      static PredMap *createPredMap(const Graph &G) 
      {
	std::cerr << __FILE__ ":" << __LINE__ <<
	  ": error: Special maps should be manually created" << std::endl;
	exit(1);
      }
    };
    ///\ref named-templ-param "Named parameter" for setting PredMap type

    ///\ref named-templ-param "Named parameter" for setting PredMap type
    ///
    template <class T>
    class SetPredMap : public Dijkstra< Graph,
					LengthMap,
					SetPredMapTraits<T> > { };
    
    template <class T>
    struct SetPredNodeMapTraits : public Traits {
      typedef T PredNodeMap;
      ///\todo An exception should be thrown.
      ///
      static PredNodeMap *createPredNodeMap(const Graph &G) 
      {
	std::cerr << __FILE__ ":" << __LINE__ <<
	  ": error: Special maps should be manually created" << std::endl;
	exit(1);
      }
    };
    ///\ref named-templ-param "Named parameter" for setting PredNodeMap type

    ///\ref named-templ-param "Named parameter" for setting PredNodeMap type
    ///
    template <class T>
    class SetPredNodeMap : public Dijkstra< Graph,
					    LengthMap,
					    SetPredNodeMapTraits<T> > { };
    
    template <class T>
    struct SetDistMapTraits : public Traits {
      typedef T DistMap;
      ///\todo An exception should be thrown.
      ///
      static DistMap *createDistMap(const Graph &G) 
      {
	std::cerr << __FILE__ ":" << __LINE__ <<
	  ": error: Special maps should be manually created" << std::endl;
	exit(1);
      }
    };
    ///\ref named-templ-param "Named parameter" for setting DistMap type

    ///\ref named-templ-param "Named parameter" for setting DistMap type
    ///
    template <class T>
    class SetDistMap : public Dijkstra< Graph,
					LengthMap,
					SetDistMapTraits<T> > { };
    
    ///Constructor.
    
    ///\param _G the graph the algorithm will run on.
    ///\param _length the length map used by the algorithm.
    Dijkstra(const Graph& _G, const LengthMap& _length) :
      G(&_G), length(&_length),
      predecessor(NULL), local_predecessor(false),
      pred_node(NULL), local_pred_node(false),
      distance(NULL), local_distance(false)
    { }
    
    ///Destructor.
    ~Dijkstra() 
    {
      if(local_predecessor) delete predecessor;
      if(local_pred_node) delete pred_node;
      if(local_distance) delete distance;
    }

    ///Sets the length map.

    ///Sets the length map.
    ///\return <tt> (*this) </tt>
    Dijkstra &setLengthMap(const LengthMap &m) 
    {
      length = &m;
      return *this;
    }

    ///Sets the map storing the predecessor edges.

    ///Sets the map storing the predecessor edges.
    ///If you don't use this function before calling \ref run(),
    ///it will allocate one. The destuctor deallocates this
    ///automatically allocated map, of course.
    ///\return <tt> (*this) </tt>
    Dijkstra &setPredMap(PredMap &m) 
    {
      if(local_predecessor) {
	delete predecessor;
	local_predecessor=false;
      }
      predecessor = &m;
      return *this;
    }

    ///Sets the map storing the predecessor nodes.

    ///Sets the map storing the predecessor nodes.
    ///If you don't use this function before calling \ref run(),
    ///it will allocate one. The destuctor deallocates this
    ///automatically allocated map, of course.
    ///\return <tt> (*this) </tt>
    Dijkstra &setPredNodeMap(PredNodeMap &m) 
    {
      if(local_pred_node) {
	delete pred_node;
	local_pred_node=false;
      }
      pred_node = &m;
      return *this;
    }

    ///Sets the map storing the distances calculated by the algorithm.

    ///Sets the map storing the distances calculated by the algorithm.
    ///If you don't use this function before calling \ref run(),
    ///it will allocate one. The destuctor deallocates this
    ///automatically allocated map, of course.
    ///\return <tt> (*this) </tt>
    Dijkstra &setDistMap(DistMap &m) 
    {
      if(local_distance) {
	delete distance;
	local_distance=false;
      }
      distance = &m;
      return *this;
    }
    
  ///Runs %Dijkstra algorithm from node \c s.

  ///This method runs the %Dijkstra algorithm from a root node \c s
  ///in order to
  ///compute the
  ///shortest path to each node. The algorithm computes
  ///- The shortest path tree.
  ///- The distance of each node from the root.
  ///\todo heap_map's type could also be in the traits class.
    void run(Node s) {
      
      init_maps();
      
      source = s;
      
      for ( NodeIt u(*G) ; u!=INVALID ; ++u ) {
	predecessor->set(u,INVALID);
	pred_node->set(u,INVALID);
      }
      
      typename Graph::template NodeMap<int> heap_map(*G,-1);
      
      Heap heap(heap_map);
      
      heap.push(s,0); 
      
      while ( !heap.empty() ) {
	
	Node v=heap.top(); 
	Value oldvalue=heap[v];
	heap.pop();
	distance->set(v, oldvalue);
	
	
	for(OutEdgeIt e(*G,v); e!=INVALID; ++e) {
	  Node w=G->target(e); 
	  switch(heap.state(w)) {
	  case Heap::PRE_HEAP:
	    heap.push(w,oldvalue+(*length)[e]); 
	    predecessor->set(w,e);
	    pred_node->set(w,v);
	    break;
	  case Heap::IN_HEAP:
	    if ( oldvalue+(*length)[e] < heap[w] ) {
	      heap.decrease(w, oldvalue+(*length)[e]); 
	      predecessor->set(w,e);
	      pred_node->set(w,v);
	    }
	    break;
	  case Heap::POST_HEAP:
	    break;
	  }
	}
      }
    }
    
    ///The distance of a node from the root.

    ///Returns the distance of a node from the root.
    ///\pre \ref run() must be called before using this function.
    ///\warning If node \c v in unreachable from the root the return value
    ///of this funcion is undefined.
    Value dist(Node v) const { return (*distance)[v]; }

    ///Returns the 'previous edge' of the shortest path tree.

    ///For a node \c v it returns the 'previous edge' of the shortest path tree,
    ///i.e. it returns the last edge of a shortest path from the root to \c
    ///v. It is \ref INVALID
    ///if \c v is unreachable from the root or if \c v=s. The
    ///shortest path tree used here is equal to the shortest path tree used in
    ///\ref predNode(Node v).  \pre \ref run() must be called before using
    ///this function.
    ///\todo predEdge could be a better name.
    Edge pred(Node v) const { return (*predecessor)[v]; }

    ///Returns the 'previous node' of the shortest path tree.

    ///For a node \c v it returns the 'previous node' of the shortest path tree,
    ///i.e. it returns the last but one node from a shortest path from the
    ///root to \c /v. It is INVALID if \c v is unreachable from the root or if
    ///\c v=s. The shortest path tree used here is equal to the shortest path
    ///tree used in \ref pred(Node v).  \pre \ref run() must be called before
    ///using this function.
    Node predNode(Node v) const { return (*pred_node)[v]; }
    
    ///Returns a reference to the NodeMap of distances.

    ///Returns a reference to the NodeMap of distances. \pre \ref run() must
    ///be called before using this function.
    const DistMap &distMap() const { return *distance;}
 
    ///Returns a reference to the shortest path tree map.

    ///Returns a reference to the NodeMap of the edges of the
    ///shortest path tree.
    ///\pre \ref run() must be called before using this function.
    const PredMap &predMap() const { return *predecessor;}
 
    ///Returns a reference to the map of nodes of shortest paths.

    ///Returns a reference to the NodeMap of the last but one nodes of the
    ///shortest path tree.
    ///\pre \ref run() must be called before using this function.
    const PredNodeMap &predNodeMap() const { return *pred_node;}

    ///Checks if a node is reachable from the root.

    ///Returns \c true if \c v is reachable from the root.
    ///\note The root node is reported to be reached!
    ///\pre \ref run() must be called before using this function.
    ///
    bool reached(Node v) { return v==source || (*predecessor)[v]!=INVALID; }
    
  };

  ///\e

  ///\e
  ///
  template<class TR>
  class _Dijkstra 
  {
    typedef TR Traits;

    ///The type of the underlying graph.
    typedef typename TR::Graph Graph;
    ///\e
    typedef typename Graph::Node Node;
    ///\e
    typedef typename Graph::NodeIt NodeIt;
    ///\e
    typedef typename Graph::Edge Edge;
    ///\e
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    
    ///The type of the map that stores the edge lengths.
    typedef typename TR::LengthMap LengthMap;
    ///The type of the length of the edges.
    typedef typename LengthMap::Value Value;
    ///\brief The type of the map that stores the last
    ///edges of the shortest paths.
    typedef typename TR::PredMap PredMap;
    ///\brief The type of the map that stores the last but one
    ///nodes of the shortest paths.
    typedef typename TR::PredNodeMap PredNodeMap;
    ///The type of the map that stores the dists of the nodes.
    typedef typename TR::DistMap DistMap;

    ///The heap type used by the dijkstra algorithm.
    typedef typename TR::Heap Heap;

    /// Pointer to the underlying graph.
    const Graph *G;
    /// Pointer to the length map
    const LengthMap *length;
    ///Pointer to the map of predecessors edges.
    PredMap *predecessor;
    ///Pointer to the map of predecessors nodes.
    PredNodeMap *pred_node;
    ///Pointer to the map of distances.
    DistMap *distance;
    
    Node source;
    
public:
    _Dijkstra() : G(0), length(0), predecessor(0), pred_node(0),
		  distance(0), source(INVALID) {}

    _Dijkstra(const Graph &g,const LengthMap &l, Node s) :
      G(&g), length(&l), predecessor(0), pred_node(0),
		  distance(0), source(s) {}

    ~_Dijkstra() 
    {
      Dijkstra<Graph,LengthMap,TR> Dij(*G,*length);
      if(predecessor) Dij.setPredMap(*predecessor);
      if(pred_node) Dij.setPredNodeMap(*pred_node);
      if(distance) Dij.setDistMap(*distance);
      Dij.run(source);
    }

    template<class T>
    struct SetPredMapTraits : public Traits {typedef T PredMap;};
    
    ///\e
    template<class T>
    _Dijkstra<SetPredMapTraits<T> > setPredMap(const T &t) 
    {
      _Dijkstra<SetPredMapTraits<T> > r;
      r.G=G;
      r.length=length;
      r.predecessor=&t;
      r.pred_node=pred_node;
      r.distance=distance;
      r.source=source;
      return r;
    }
    
    template<class T>
    struct SetPredNodeMapTraits :public Traits {typedef T PredNodeMap;};
    ///\e
    template<class T>
    _Dijkstra<SetPredNodeMapTraits<T> > setPredNodeMap(const T &t) 
    {
      _Dijkstra<SetPredNodeMapTraits<T> > r;
      r.G=G;
      r.length=length;
      r.predecessor=predecessor;
      r.pred_node=&t;
      r.distance=distance;
      r.source=source;
      return r;
    }
    
    template<class T>
    struct SetDistMapTraits : public Traits {typedef T DistMap;};
    ///\e
    template<class T>
    _Dijkstra<SetDistMapTraits<T> > setDistMap(const T &t) 
    {
      _Dijkstra<SetPredMapTraits<T> > r;
      r.G=G;
      r.length=length;
      r.predecessor=predecessor;
      r.pred_node=pred_node;
      r.distance=&t;
      r.source=source;
      return r;
    }
    
    ///\e
    _Dijkstra<TR> &setSource(Node s) 
    {
      source=s;
      return *this;
    }
    
  };
  
  ///\e

  ///\todo Please document...
  ///
  template<class GR, class LM>
  _Dijkstra<DijkstraDefaultTraits<GR,LM> >
  dijkstra(const GR &g,const LM &l,typename GR::Node s)
  {
    return _Dijkstra<DijkstraDefaultTraits<GR,LM> >(g,l,s);
  }

/// @}
  
} //END OF NAMESPACE LEMON

#endif


