/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include <algobox.h>
#include <mapstorage.h>
#include <mapselector.h>

enum {N_DEMO1, N_DEMO2, NODE_INPUT_NUM}; // input IDs for nodes;
enum {E_DEMO1, EDGE_INPUT_NUM}; // input IDs for arcs;

AlgoBox::AlgoBox(std::vector<std::string> tabnames)
{
  init(tabnames);
}

void AlgoBox::init(std::vector<std::string> tabnames)
{
  set_spacing(5);

  update_tablist(tabnames);

  //if active tab is changed, the map names in cbt/s have to be updated
  tabcbt.signal_changed().connect(sigc::mem_fun(*this, &AlgoBox::emit_tab_change));

  pack_start(tabcbt);
  build_box();

  show_all_children();
};

void AlgoBox::update_cbt(std::vector< std::string > stringlist, Gtk::ComboBoxText & cbt)
{
  std::string actname=cbt.get_active_text();
  int prev_act=-1;

  cbt.clear();
  int actptr=0;

  std::vector< std::string >::iterator emsi=stringlist.begin();
  for(;emsi!=stringlist.end();emsi++)
    {
      if(actname==*emsi)
	{
	  prev_act=actptr;
	}

      cbt.append_text(*emsi);
      actptr++;
    }

  if(prev_act!=-1)
    {
      cbt.set_active(prev_act);
    }
  else if(actptr>0) //so there is item in the list
    {
      //cbt.set_active(0);
    }
}

void AlgoBox::update_tablist( std::vector< std::string > tl )
{
  update_cbt(tl, tabcbt);
  emit_tab_change();
}

void AlgoBox::update_maplist(MapStorage * ms)
{
  mapstorage=ms;
  std::vector<std::string> n_nml;
  std::vector<std::string> s_nml;
  std::vector<std::string> n_eml;
  std::vector<std::string> s_eml;
  if(mapstorage!=NULL)
    {
      mapstorage->signal_node_map_ch().connect(sigc::mem_fun(*this, &AlgoBox::nodemaplist_changed));
      mapstorage->signal_arc_map_ch().connect(sigc::mem_fun(*this, &AlgoBox::arcmaplist_changed));
      n_nml=mapstorage->getNodeMapList(NUM);
      s_nml=mapstorage->getNodeMapList(STR);
      n_eml=mapstorage->getArcMapList(NUM);
      s_eml=mapstorage->getArcMapList(STR);
    }
  for(int i=0;i<(int)nodemapcbts.size();i++)
    {
      (nodemapcbts[i])->update_list(n_nml, s_nml);
      //update_cbt(nml, *(nodemapcbts[i]));
    }
  for(int i=0;i<(int)arcmapcbts.size();i++)
    {
      (arcmapcbts[i])->update_list(n_eml, s_eml);
      //update_cbt(eml, *(arcmapcbts[i]));
    }
  signal_maplist_updated.emit();
}

void AlgoBox::nodemaplist_changed(std::string newmap, MapValue::Type type)
{
  for(int i=0;i<(int)nodemapcbts.size();i++)
    {
      (nodemapcbts[i])->append_text(newmap, type);
    }
}

void AlgoBox::arcmaplist_changed(std::string newmap, MapValue::Type type)
{
  for(int i=0;i<(int)arcmapcbts.size();i++)
    {
      (arcmapcbts[i])->append_text(newmap, type);
    }
}

void AlgoBox::run()
{
  std::cout << "Start algorithm." << std::endl;
}

void AlgoBox::build_box()
{
  pack_start(*(new Gtk::HSeparator()));

  Gtk::Label * label=new Gtk::Label("Specific part for each algorithm.");
      
  pack_start(*label);
  pack_start(*(new Gtk::HSeparator()));

  label=new Gtk::Label("Maps in chosen tab:");
      
  pack_start(*label);

  for(int i=0;i<NODE_INPUT_NUM;i++)
    {
      std::ostringstream o;
      o << "NodeInput " << i+1 << ":";

      addMapSelector(o.str(), false);
    }

  pack_start(*(new Gtk::HSeparator()));

  for(int i=0;i<EDGE_INPUT_NUM;i++)
    {

      std::ostringstream o;
      o << "ArcInput " << i+1 << ":";

      addMapSelector(o.str(), true);
    }

  pack_start(*(new Gtk::HSeparator()));
}

void AlgoBox::addMapSelector(std::string inputname, bool itisarc, MapType type)
{
  std::vector<std::string> empty_vector;

  MapSelector * msp=new MapSelector(empty_vector,empty_vector,"",inputname,itisarc, false, type);

  if(itisarc)
    {
      arcmapcbts.resize(arcmapcbts.size()+1);
      arcmapcbts[arcmapcbts.size()-1]=msp;
    }
  else
    {
      nodemapcbts.resize(nodemapcbts.size()+1);
      nodemapcbts[nodemapcbts.size()-1]=msp;
    }

  msp->signal_newmapwin_needed().connect(sigc::mem_fun(*this, &AlgoBox::emit_new_map_signal));

  pack_start(*msp);
}

sigc::signal<void, std::string> AlgoBox::signal_maplist_needed()
{
  return signal_maplist_need;
}

void AlgoBox::emit_tab_change()
{
  std::string active_tab=tabcbt.get_active_text();
  if(active_tab!="")
    {
      signal_maplist_need.emit(active_tab);
    }
  else
    {
      std::vector<std::string> empty_vector;
      update_maplist(NULL);
    }
}

void AlgoBox::emit_new_map_signal(bool itisarc)
{
  signal_newmapwin_need.emit(tabcbt.get_active_text(), itisarc);
}
