/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include <graph_displayer_canvas.h>
#include <mapstorage.h>
#include <nbtab.h>
#include <cmath>
#include <set>

bool DigraphDisplayerCanvas::on_expose_event(GdkEventExpose *event)
{
  Gnome::Canvas::CanvasAA::on_expose_event(event);
  //usleep(10000);
  //rezoom();
  return true;
}

void DigraphDisplayerCanvas::changeEditorialTool(int newtool)
{
  if(actual_tool!=newtool)
  {

    actual_handler.disconnect();

    switch(actual_tool)
    {
      case CREATE_EDGE:
        {
          GdkEvent * generated=new GdkEvent();
          generated->type=GDK_BUTTON_RELEASE;
          generated->button.button=3;
          createArcEventHandler(generated);      
          break;
        }
      case MAP_EDIT:
        {
          break;
        }
      default:
        break;
    }

    active_item=NULL; 
    target_item=NULL; 
    active_arc=INVALID;	
    active_node=INVALID;	


    actual_tool=newtool;

    switch(newtool)
    {
      case MOVE:
        actual_handler=signal_event().connect(sigc::mem_fun(*this, &DigraphDisplayerCanvas::moveEventHandler), false);
        break;

      case CREATE_NODE:
        actual_handler=signal_event().connect(sigc::mem_fun(*this, &DigraphDisplayerCanvas::createNodeEventHandler), false);
        break;

      case CREATE_EDGE:
        actual_handler=signal_event().connect(sigc::mem_fun(*this, &DigraphDisplayerCanvas::createArcEventHandler), false);
        break;

      case ERASER:
        actual_handler=signal_event().connect(sigc::mem_fun(*this, &DigraphDisplayerCanvas::eraserEventHandler), false);
        break;

      case MAP_EDIT:
        grab_focus();
        actual_handler=signal_event().connect(sigc::mem_fun(*this, &DigraphDisplayerCanvas::mapEditEventHandler), false);
        break;

      default:
        break;
    }
  }
}

int DigraphDisplayerCanvas::getActualTool()
{
  return actual_tool;
}

bool DigraphDisplayerCanvas::scrollEventHandler(GdkEvent* e)
{
  bool handled=false;
  if(e->type==GDK_SCROLL)
    {

      //pointer shows this win point before zoom
      XY win_coord(((GdkEventScroll*)e)->x, ((GdkEventScroll*)e)->y);

      //the original scroll settings
      int scroll_offset_x, scroll_offset_y;
      get_scroll_offsets(scroll_offset_x, scroll_offset_y);

      //pointer shows this canvas point before zoom
      XY canvas_coord;
      window_to_world(win_coord.x, win_coord.y, canvas_coord.x, canvas_coord.y);

      if(((GdkEventScroll*)e)->direction) //IN
	{
	  zoomIn();
	}
      else
	{
	  zoomOut();
	}

      //pointer shows this window point after zoom
      XY post_win_coord;
      world_to_window(canvas_coord.x, canvas_coord.y, post_win_coord.x, post_win_coord.y);

      //we have to add the difference between new and old window point to original scroll offset
      scroll_to(scroll_offset_x+(int)(post_win_coord.x-win_coord.x),scroll_offset_y+(int)(post_win_coord.y-win_coord.y));
      
      //no other eventhandler is needed
      handled=true;
    }
  return handled;
}

bool DigraphDisplayerCanvas::moveEventHandler(GdkEvent* e)
{
  MapStorage& ms = *mytab.mapstorage;

  static Gnome::Canvas::Text *coord_text = 0;
  switch(e->type)
  {
    case GDK_BUTTON_PRESS:
      //we mark the location of the event to be able to calculate parameters of dragging
      window_to_world (e->button.x, e->button.y, clicked_x, clicked_y);

      active_item=(get_item_at(clicked_x, clicked_y));
      active_node=INVALID;
      for (NodeIt i(ms.digraph); i!=INVALID; ++i)
      {
        if(nodesmap[i]==active_item)
        {
          active_node=i;
        }
      }
      isbutton=e->button.button;
      break;
    case GDK_BUTTON_RELEASE:
      if (coord_text)
      {
        delete coord_text;
        coord_text = 0;
      }
      isbutton=0;
      active_item=NULL;
      active_node=INVALID;
      break;
    case GDK_MOTION_NOTIFY:
      //we only have to do sg. if the mouse button is pressed AND the click was on a node that was found in the set of nodes
      if(active_node!=INVALID)
      {
        ms.setModified();

        //new coordinates will be the old values,
        //because the item will be moved to the
        //new coordinate therefore the new movement
        //has to be calculated from here

        double new_x, new_y;

        window_to_world (e->motion.x, e->motion.y, new_x, new_y);

        double dx=new_x-clicked_x;
        double dy=new_y-clicked_y;

        moveNode(dx, dy);

        clicked_x=new_x;
        clicked_y=new_y;

        // reposition the coordinates text
        std::ostringstream ostr;
        ostr << "(" <<
          ms.getNodeCoords(active_node).x << ", " <<
          ms.getNodeCoords(active_node).y << ")";
        double radius =
          (nodesmap[active_node]->property_x2().get_value() -
           nodesmap[active_node]->property_x1().get_value()) / 2.0;
        if (coord_text)
        {
          coord_text->property_text().set_value(ostr.str());
          coord_text->property_x().set_value(
              ms.getNodeCoords(active_node).x + radius);
          coord_text->property_y().set_value(
              ms.getNodeCoords(active_node).y - radius);
        }
        else
        {
          coord_text = new Gnome::Canvas::Text(
              displayed_graph,
              ms.getNodeCoords(active_node).x + radius,
              ms.getNodeCoords(active_node).y - radius,
              ostr.str());
          coord_text->property_fill_color().set_value("black");
          coord_text->property_anchor().set_value(Gtk::ANCHOR_SOUTH_WEST);
        }


      }
    default: break;
  }

  return false;
}

XY DigraphDisplayerCanvas::calcArrowPos(XY moved_node_1, XY moved_node_2, XY fix_node, XY old_arrow_pos, int move_code)
{
  switch(move_code)
  {
    case 1:
      return XY((moved_node_2.x + fix_node.x) / 2.0, (moved_node_2.y + fix_node.y) / 2.0);
      break;
    case 2:
      return old_arrow_pos;
      break;
    case 3:
      {
        //////////////////////////////////////////////////////////////////////////////////////////////////////
        /////////// keeps shape-with scalar multiplication - version 2.
        //////////////////////////////////////////////////////////////////////////////////////////////////////

        //old vector from one to the other node - a
        XY a_v(moved_node_1.x-fix_node.x,moved_node_1.y-fix_node.y);
        //new vector from one to the other node - b
        XY b_v(moved_node_2.x-fix_node.x,moved_node_2.y-fix_node.y);

        double absa=sqrt(a_v.normSquare());
        double absb=sqrt(b_v.normSquare());

        if ((absa == 0.0) || (absb == 0.0))
        {
          return old_arrow_pos;
        }
        else
        {
          //old vector from one node to the breakpoint - c
          XY c_v(old_arrow_pos.x-fix_node.x,old_arrow_pos.y-fix_node.y);

          //unit vector with the same direction to a_v
          XY a_v_u(a_v.x/absa,a_v.y/absa);

          //normal vector of unit vector with the same direction to a_v
          XY a_v_u_n(((-1)*a_v_u.y),a_v_u.x);

          //unit vector with the same direction to b_v
          XY b_v_u(b_v.x/absb,b_v.y/absb);

          //normal vector of unit vector with the same direction to b_v
          XY b_v_u_n(((-1)*b_v_u.y),b_v_u.x);

          //vector c in a_v_u and a_v_u_n co-ordinate system
          XY c_a(c_v*a_v_u,c_v*a_v_u_n);

          //new vector from one node to the breakpoint - d - we have to calculate this one
          XY d_v=absb/absa*(c_a.x*b_v_u+c_a.y*b_v_u_n);

          return XY(d_v.x+fix_node.x,d_v.y+fix_node.y);
        }
        break;
      }
    default:
      break;
  }
}


bool DigraphDisplayerCanvas::createNodeEventHandler(GdkEvent* e)
{
  MapStorage& ms = *mytab.mapstorage;

  switch(e->type)
  {
    //move the new node
    case GDK_MOTION_NOTIFY:
      {
        GdkEvent * generated=new GdkEvent();
        generated->motion.x=e->motion.x;
        generated->motion.y=e->motion.y;
        generated->type=GDK_MOTION_NOTIFY;
        moveEventHandler(generated);      
        break;
      }

    case GDK_BUTTON_RELEASE:
      ms.setModified();

      is_drawn=true;

      isbutton=1;

      window_to_world (e->button.x, e->button.y, clicked_x, clicked_y);

      active_node = ms.addNode(XY(clicked_x, clicked_y));

      nodesmap[active_node]=new Gnome::Canvas::Ellipse(displayed_graph,
          clicked_x-20, clicked_y-20, clicked_x+20, clicked_y+20);
      active_item=(Gnome::Canvas::Item *)(nodesmap[active_node]);
      *(nodesmap[active_node]) <<
        Gnome::Canvas::Properties::fill_color("blue");
      *(nodesmap[active_node]) <<
        Gnome::Canvas::Properties::outline_color("black");
      active_item->raise_to_top();

      (nodesmap[active_node])->show();

      nodetextmap[active_node]=new Gnome::Canvas::Text(displayed_graph,
          clicked_x+node_property_defaults[N_RADIUS]+5,
          clicked_y+node_property_defaults[N_RADIUS]+5, "");
      nodetextmap[active_node]->property_fill_color().set_value("darkblue");
      nodetextmap[active_node]->raise_to_top();

      //       mapwin.updateNode(active_node);
      propertyUpdate(active_node);

      isbutton=0;
      target_item=NULL;
      active_item=NULL;
      active_node=INVALID;
      break;
    default:
      break;
  }
  return false;
}

bool DigraphDisplayerCanvas::createArcEventHandler(GdkEvent* e)
{
  MapStorage& ms = *mytab.mapstorage;

  switch(e->type)
  {
    case GDK_BUTTON_PRESS:
      //in arc creation right button has special meaning
      if(e->button.button!=3)
      {
        //there is not yet selected node
        if(active_node==INVALID)
        {
          //we mark the location of the event to be able to calculate parameters of dragging

          window_to_world (e->button.x, e->button.y, clicked_x, clicked_y);

          active_item=(get_item_at(clicked_x, clicked_y));
          active_node=INVALID;
          for (NodeIt i(ms.digraph); i!=INVALID; ++i)
          {
            if(nodesmap[i]==active_item)
            {
              active_node=i;
            }
          }
          //the clicked item is really a node
          if(active_node!=INVALID)
          {
            *(nodesmap[active_node]) << Gnome::Canvas::Properties::fill_color("red");
            isbutton=1;
          }
          //clicked item was not a node. It could be e.g. arc.
          else
          {
            active_item=NULL;
          }
        }
        //we only have to do sg. if the mouse button
        // is pressed already once AND the click was
        // on a node that was found in the set of 
        //nodes, and now we only search for the second 
        //node
        else
        {
          window_to_world (e->button.x, e->button.y, clicked_x, clicked_y);
          target_item=(get_item_at(clicked_x, clicked_y));
          Node target_node=INVALID;
          for (NodeIt i(ms.digraph); i!=INVALID; ++i)
          {
            if(nodesmap[i]==target_item)
            {
              target_node=i;
            }
          }
          //the clicked item is a node, the arc can be drawn
          if(target_node!=INVALID)
          {
            ms.setModified();

            *(nodesmap[target_node]) <<
              Gnome::Canvas::Properties::fill_color("red");

            active_arc = ms.addArc(active_node, target_node);

            if(target_node!=active_node)		
            {
              arcsmap[active_arc]=new BrokenArc(displayed_graph, active_arc, *this);
            }
            else
            {
              arcsmap[active_arc]=new LoopArc(displayed_graph, active_arc, *this);
            }

            //initializing arc-text as well, to empty string
            XY text_pos=ms.getArrowCoords(active_arc);
            text_pos+=(XY(10,10));

            arctextmap[active_arc]=new Gnome::Canvas::Text(displayed_graph,
                text_pos.x, text_pos.y, "");
            arctextmap[active_arc]->property_fill_color().set_value(
                "darkgreen");
            arctextmap[active_arc]->raise_to_top();

            propertyUpdate(active_arc);
          }
          //clicked item was not a node. it could be an e.g. arc. we do not
          //deal with it furthermore.
          else
          {
            target_item=NULL;
          }
        }
      }
      break;
    case GDK_BUTTON_RELEASE:
      isbutton=0;
      //we clear settings in two cases
      //1: the arc is ready (target_item has valid value)
      //2: the arc creation is cancelled with right button
      if((target_item)||(e->button.button==3))
      {
        if(active_item)
        {
          propertyUpdate(active_node,N_COLOR);
          active_item=NULL;
        }
        if(target_item)
        {
          propertyUpdate(ms.digraph.target(active_arc),N_COLOR);
          target_item=NULL;
        }
        active_node=INVALID;
        active_arc=INVALID;
      }
      break;
    default:
      break;
  }
  return false;
}

bool DigraphDisplayerCanvas::eraserEventHandler(GdkEvent* e)
{
  MapStorage& ms = *mytab.mapstorage;

  switch(e->type)
  {
    case GDK_BUTTON_PRESS:
      //finding the clicked items
      window_to_world (e->button.x, e->button.y, clicked_x, clicked_y);
      active_item=(get_item_at(clicked_x, clicked_y));
      active_node=INVALID;
      active_arc=INVALID;
      //was it a node?
      for (NodeIt i(ms.digraph); i!=INVALID; ++i)
      {
        if(nodesmap[i]==active_item)
        {
          active_node=i;
        }
      }
      //or was it an arc?
      if(active_node==INVALID)
      {
        for (ArcIt i(ms.digraph); i!=INVALID; ++i)
        {
          if(arcsmap[i]->getLine()==active_item)
          {
            active_arc=i;
          }
        }
      }

      // return if the clicked object is neither an arc nor a node
      if (active_arc == INVALID) return false;

      //recolor activated item
      if(active_item)
      {
        *active_item << Gnome::Canvas::Properties::fill_color("red");
      }
      break;

    case GDK_BUTTON_RELEASE:
      window_to_world (e->button.x, e->button.y, clicked_x, clicked_y);
      if(active_item)
      {
        //the cursor was not moved since pressing it
        if( active_item == ( get_item_at (clicked_x, clicked_y) ) )
        {
          //a node was found
          if(active_node!=INVALID)
          {
            ms.setModified();

            std::set<Digraph::Arc> arcs_to_delete;

            for(OutArcIt e(ms.digraph,active_node);e!=INVALID;++e)
            {
              arcs_to_delete.insert(e);
            }

            for(InArcIt e(ms.digraph,active_node);e!=INVALID;++e)
            {
              arcs_to_delete.insert(e);
            }

            //deleting collected arcs
            for(std::set<Digraph::Arc>::iterator
                arc_set_it=arcs_to_delete.begin();
                arc_set_it!=arcs_to_delete.end();
                ++arc_set_it)
            {
              deleteItem(*arc_set_it);
            }
            deleteItem(active_node);
          }
          //a simple arc was chosen
          else if (active_arc != INVALID)
          {
            deleteItem(active_arc);
          }
        }
        //pointer was moved, deletion is cancelled
        else
        {
          if(active_node!=INVALID)
          {
            *active_item << Gnome::Canvas::Properties::fill_color("blue");
          }
          else if (active_arc != INVALID)
          {
            *active_item << Gnome::Canvas::Properties::fill_color("green");
          }
        }
      }
      //reseting datas
      active_item=NULL;
      active_arc=INVALID;
      active_node=INVALID;
      break;

    case GDK_MOTION_NOTIFY:
      break;

    default:
      break;
  }
  return false;
}

bool DigraphDisplayerCanvas::mapEditEventHandler(GdkEvent* e)
{
  MapStorage& ms = *mytab.mapstorage;

  if(actual_tool==MAP_EDIT)
  {
    switch(e->type)
    {
      case GDK_BUTTON_PRESS:
        {
          //for determine, whether it was an arc
          Arc clicked_arc=INVALID;
          //for determine, whether it was a node
          Node clicked_node=INVALID;

          window_to_world (e->button.x, e->button.y, clicked_x, clicked_y);
          active_item=(get_item_at(clicked_x, clicked_y));

          //find the activated item between text of nodes
          for (NodeIt i(ms.digraph); i!=INVALID; ++i)
          {
            //at the same time only one can be active
            if(nodetextmap[i]==active_item)
            {
              clicked_node=i;
            }
          }

          //if there was not, search for it between nodes
          if(clicked_node==INVALID)
          {
            for (NodeIt i(ms.digraph); i!=INVALID; ++i)
            {
              //at the same time only one can be active
              if(nodesmap[i]==active_item)
              {
                clicked_node=i;
              }
            }
          }

          if(clicked_node==INVALID)
          {
            //find the activated item between texts
            for (ArcIt i(ms.digraph); i!=INVALID; ++i)
            {
              //at the same time only one can be active
              if(arctextmap[i]==active_item)
              {
                clicked_arc=i;
              }
            }

            //if it was not between texts, search for it between arcs
            if(clicked_arc==INVALID)
            {
              for (ArcIt i(ms.digraph); i!=INVALID; ++i)
              {
                //at the same time only one can be active
                if((arcsmap[i]->getLine())==active_item)
                {
                  clicked_arc=i;
                }
              }
            }
          }

          //if it was really a node...
          if(clicked_node!=INVALID)
          {
            // the id map is not editable
            if (nodemap_to_edit == "label") return 0;

            //and there is activated map
            if(nodetextmap[clicked_node]->property_text().get_value()!="")
            {
              //activate the general variable for it
              active_node=clicked_node;

              //create a dialog
              Gtk::Dialog dialog("Edit value", true);
              dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
              dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_ACCEPT);
              Gtk::VBox* vbox = dialog.get_vbox();

              /*
              Gtk::SpinButton spin(0.0, 4);
              spin.set_increments(1.0, 10.0);
              spin.set_range(-1000000.0, 1000000.0);
              spin.set_numeric(true);
              spin.set_value(atof(nodetextmap[active_node]->property_text().get_value().c_str()));
              vbox->add(spin);
              spin.show();
              */
              Gtk::Entry entry;
              entry.set_text(nodetextmap[active_node]->property_text().get_value());
              vbox->add(entry);
              entry.show();

              switch (dialog.run())
              {
                case Gtk::RESPONSE_NONE:
                case Gtk::RESPONSE_CANCEL:
                  break;
                case Gtk::RESPONSE_ACCEPT:
                  switch (ms.getNodeMapElementType(nodemap_to_edit))
                  {
                    case MapValue::NUMERIC:
                      ms.set(nodemap_to_edit, active_node,
                          atof(entry.get_text().c_str()));
                      break;
                    case MapValue::STRING:
                      ms.set(nodemap_to_edit, active_node,
                          static_cast<std::string>(entry.get_text()));
                      break;
                  }
                  nodetextmap[active_node]->property_text().set_value(
                      static_cast<std::string>(ms.get(nodemap_to_edit, active_node)));

                  //mapwin.updateNode(active_node);
                  //mapwin.updateNode(Node(INVALID));
                  propertyUpdate(Node(INVALID));
              }
            }
          }
          else
            //if it was really an arc...
            if(clicked_arc!=INVALID)
            {
              // the id map is not editable
              if (arcmap_to_edit == "label") return 0;

              //and there is activated map
              if(arctextmap[clicked_arc]->property_text().get_value()!="")
              {
                //activate the general variable for it
                active_arc=clicked_arc;

                //create a dialog
                Gtk::Dialog dialog("Edit value", true);
                dialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
                dialog.add_button(Gtk::Stock::OK, Gtk::RESPONSE_ACCEPT);
                Gtk::VBox* vbox = dialog.get_vbox();

                /*
                Gtk::SpinButton spin(0.0, 4);
                spin.set_increments(1.0, 10.0);
                spin.set_range(-1000000.0, 1000000.0);
                spin.set_numeric(true);
                spin.set_value(atof(arctextmap[active_arc]->property_text().get_value().c_str()));
                vbox->add(spin);
                spin.show();
                */
                Gtk::Entry entry;
                entry.set_text(arctextmap[active_arc]->property_text().get_value());
                vbox->add(entry);
                entry.show();

                std::cout << arcmap_to_edit << std::endl;
                switch (dialog.run())
                {
                  case Gtk::RESPONSE_NONE:
                  case Gtk::RESPONSE_CANCEL:
                    break;
                  case Gtk::RESPONSE_ACCEPT:
                    switch (ms.getArcMapElementType(arcmap_to_edit))
                    {
                      case MapValue::NUMERIC:
                        ms.set(arcmap_to_edit, active_arc,
                            atof(entry.get_text().c_str()));
                        break;
                      case MapValue::STRING:
                        ms.set(arcmap_to_edit, active_arc,
                            static_cast<std::string>(entry.get_text()));
                        break;
                    }
                    arctextmap[active_arc]->property_text().set_value(
                        static_cast<std::string>(ms.get(arcmap_to_edit, active_arc)));

                    //mapwin.updateArc(active_arc);
                    //                   mapwin.updateArc(Arc(INVALID));
                    propertyUpdate(Arc(INVALID));
                }
              }
            }
          break;
        }
      default:
        break;
    }
  }
  return false;  
}

void DigraphDisplayerCanvas::deleteItem(Node node_to_delete)
{
  delete(nodetextmap[node_to_delete]);
  delete(nodesmap[node_to_delete]);
  mytab.mapstorage->digraph.erase(node_to_delete);
}

void DigraphDisplayerCanvas::deleteItem(Arc arc_to_delete)
{
  delete(arctextmap[arc_to_delete]);
  delete(arcsmap[arc_to_delete]);
  mytab.mapstorage->digraph.erase(arc_to_delete);
}

void DigraphDisplayerCanvas::textReposition(XY new_place)
{
  new_place+=(XY(10,10));
  arctextmap[forming_arc]->property_x().set_value(new_place.x);
  arctextmap[forming_arc]->property_y().set_value(new_place.y);
}

void DigraphDisplayerCanvas::toggleArcActivity(ArcBase* active_bre, bool on)
{
  if(on)
  {
    if(forming_arc!=INVALID)
    {
      std::cerr << "ERROR!!!! Valid arc found!" << std::endl;
    }
    else
    {
      for (ArcIt i(mytab.mapstorage->digraph); i!=INVALID; ++i)
      {
        if(arcsmap[i]==active_bre)
        {
          forming_arc=i;
        }
      }
    }
  }
  else
  {
    if(forming_arc!=INVALID)
    {
      forming_arc=INVALID;
    }
    else
    {
      std::cerr << "ERROR!!!! Invalid arc found!" << std::endl;
    }
  }
}

void DigraphDisplayerCanvas::moveNode(double dx, double dy, Gnome::Canvas::Item * item, Node node)
{
  MapStorage& ms = *mytab.mapstorage;

  Gnome::Canvas::Item * moved_item=item;
  Node moved_node=node;

  if(item==NULL && node==INVALID)
  {
    moved_item=active_item;
    moved_node=active_node;
  }
  else
  {
    isbutton=1;
  }

  //repositioning node and its text
  moved_item->move(dx, dy);
  nodetextmap[moved_node]->move(dx, dy);

  // the new coordinates of the centre of the node 
  double coord_x = dx + ms.getNodeCoords(moved_node).x;
  double coord_y = dy + ms.getNodeCoords(moved_node).y;

  // write back the new coordinates to the coords map
  ms.setNodeCoords(moved_node, XY(coord_x, coord_y));

  //all the arcs connected to the moved point has to be redrawn
  for(OutArcIt ei(ms.digraph,moved_node);ei!=INVALID;++ei)
  {
    XY arrow_pos;

    if (ms.digraph.source(ei) == ms.digraph.target(ei))
    {
      arrow_pos = ms.getArrowCoords(ei) + XY(dx, dy);
    }
    else
    {
      XY moved_node_1(coord_x - dx, coord_y - dy);
      XY moved_node_2(coord_x, coord_y);
      Node target = ms.digraph.target(ei);
      XY fix_node = ms.getNodeCoords(target);
      XY old_arrow_pos(ms.getArrowCoords(ei));

      arrow_pos = calcArrowPos(moved_node_1, moved_node_2, fix_node, old_arrow_pos, isbutton);
    }

    ms.setArrowCoords(ei, arrow_pos);
    arcsmap[ei]->draw();

    //reposition of arctext
    XY text_pos=ms.getArrowCoords(ei);
    text_pos+=(XY(10,10));
    arctextmap[ei]->property_x().set_value(text_pos.x);
    arctextmap[ei]->property_y().set_value(text_pos.y);
  }

  for(InArcIt ei(ms.digraph,moved_node);ei!=INVALID;++ei)
  {
    if (ms.digraph.source(ei) != ms.digraph.target(ei))
    {
      XY moved_node_1(coord_x - dx, coord_y - dy);
      XY moved_node_2(coord_x, coord_y);
      Node source = ms.digraph.source(ei);
      XY fix_node = ms.getNodeCoords(source);
      XY old_arrow_pos(ms.getArrowCoords(ei));

      XY arrow_pos;
      arrow_pos = calcArrowPos(moved_node_1, moved_node_2, fix_node, old_arrow_pos, isbutton);

      ms.setArrowCoords(ei, arrow_pos);
      arcsmap[ei]->draw();

      //reposition of arctext
      XY text_pos=ms.getArrowCoords(ei);
      text_pos+=(XY(10,10));
      arctextmap[ei]->property_x().set_value(text_pos.x);
      arctextmap[ei]->property_y().set_value(text_pos.y);
    }
  }
}

Gdk::Color DigraphDisplayerCanvas::rainbowColorCounter(double min, double max, double w)
{
  Gdk::Color color;

  double pos=(w-min)/(max-min);
  int phase=0;

  //rainbow transitions contain 6 phase
  //in each phase only one color is changed
  //first we determine the phase, in which
  //the actual value belongs to
  for (int i=0;i<=5;i++)
  {
    if(((double)i/6<pos)&&(pos<=(double(i+1)/6)))
    {
      phase=i;
    }
  }
  if(phase<6)
  {
    //within its 1/6 long phase the relativ position
    //determines the power of the color changed in
    //that phase
    //we normalize that to one, to be able to give percentage
    //value for the function
    double rel_pos=(pos-(phase/6.0))*6.0;

    switch(phase)
    {
      case 0:
        color.set_rgb_p (1, 0, 1-rel_pos);
        break;
      case 1:
        color.set_rgb_p (1, rel_pos, 0);
        break;
      case 2:
        color.set_rgb_p (1-rel_pos, 1, 0);
        break;
      case 3:
        color.set_rgb_p (0, 1, rel_pos);
        break;
      case 4:
        color.set_rgb_p (0, 1-rel_pos, 1);
        break;
      case 5:
        color.set_rgb_p ((rel_pos/3.0), 0, 1);
        break;
      default:
        std::cout << "Wrong phase: " << phase << " " << pos << std::endl;
    }
  }
  else
  {
    std::cout << "Wrong phase: " << phase << " " << pos << std::endl;
  }
  return color;
}
