/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include "gui_writer.h"
#include "io_helper.h"
#include "mapstorage.h"
#include "xml.h"
#include <lemon/dim2.h>
#include <vector>

#include <gui_writer.h>
#include <mapstorage.h>

// std::string GuiWriter::header()
// {
//   return "@gui";
// }

void GuiWriter::write(std::ostream& os)
{
  using std::vector;
  using std::string;
  using std::map;
  using std::string;

  os << "@gui" << std::endl;

  XmlIo x(os);

  vector<string> all_node_map_names = mapstorage->getNodeMapList();
  // name of the maps saved to the nodeset section
  vector<string> main_node_map_names;
  // name of the maps saved to the gui section
  vector<string> gui_node_map_names;

  for (vector<string>::const_iterator it = all_node_map_names.begin();
      it != all_node_map_names.end(); ++it)
  {
    if (mapstorage->getNodeMapSaveDest(*it) == MapStorage::NESET_SECT)
      main_node_map_names.push_back(*it);
    else if (mapstorage->getNodeMapSaveDest(*it) == MapStorage::GUI_SECT)
      gui_node_map_names.push_back(*it);
  }

  { x("main_node_map_names", main_node_map_names); }
  { x("gui_node_map_names", gui_node_map_names); }

  map<string, MapValue::Type> node_map_types;
  for (vector<string>::const_iterator it = main_node_map_names.begin();
      it != main_node_map_names.end(); ++it)
  {
    node_map_types[*it] = mapstorage->getNodeMapElementType(*it);
  }
  for (vector<string>::const_iterator it = gui_node_map_names.begin();
      it != gui_node_map_names.end(); ++it)
  {
    node_map_types[*it] = mapstorage->getNodeMapElementType(*it);
  }

  { x("node_map_types", node_map_types); }


  vector<string> all_arc_map_names = mapstorage->getArcMapList();
  // name of the maps saved to the arcset section
  vector<string> main_arc_map_names;
  // name of the maps saved to the gui section
  vector<string> gui_arc_map_names;

  for (vector<string>::const_iterator it = all_arc_map_names.begin();
      it != all_arc_map_names.end(); ++it)
  {
    if (mapstorage->getArcMapSaveDest(*it) == MapStorage::NESET_SECT)
      main_arc_map_names.push_back(*it);
    if (mapstorage->getArcMapSaveDest(*it) == MapStorage::GUI_SECT)
      gui_arc_map_names.push_back(*it);
  }

  { x("main_arc_map_names", main_arc_map_names); }
  { x("gui_arc_map_names", gui_arc_map_names); }

  map<string, MapValue::Type> arc_map_types;
  for (vector<string>::const_iterator it = main_arc_map_names.begin();
      it != main_arc_map_names.end(); ++it)
  {
    arc_map_types[*it] = mapstorage->getArcMapElementType(*it);
  }
  for (vector<string>::const_iterator it = gui_arc_map_names.begin();
      it != gui_arc_map_names.end(); ++it)
  {
    arc_map_types[*it] = mapstorage->getArcMapElementType(*it);
  }

  { x("arc_map_types", arc_map_types); }

  // write the gui node maps
  for (vector<string>::const_iterator it = gui_node_map_names.begin();
      it != gui_node_map_names.end(); ++it)
  {
    MapValue::Type type = mapstorage->getNodeMapElementType(*it);
    const MapStorage::NodeLabelMap& labels = mapstorage->getNodeLabelMap();
    switch (type)
    {
      case MapValue::NUMERIC:
        {
          std::map<int, double> map_data;
          MapStorage::NumericNodeMap& map =
            mapstorage->getNumericNodeMap(*it);
          for (NodeIt n(mapstorage->getDigraph()); n != INVALID; ++n)
          {
            map_data[labels[n]] = map[n];
          }
          { x(*it, map_data); }
        }
        break;
      case MapValue::STRING:
        {
          std::map<int, std::string> map_data;
          MapStorage::StringNodeMap& map =
            mapstorage->getStringNodeMap(*it);
          for (NodeIt n(mapstorage->getDigraph()); n != INVALID; ++n)
          {
            map_data[labels[n]] = map[n];
          }
          { x(*it, map_data); }
        }
        break;
    }
  }

  // write the gui arc maps
  for (vector<string>::const_iterator it = gui_arc_map_names.begin();
      it != gui_arc_map_names.end(); ++it)
  {
    MapValue::Type type = mapstorage->getArcMapElementType(*it);
    const MapStorage::ArcLabelMap& labels = mapstorage->getArcLabelMap();
    switch (type)
    {
      case MapValue::NUMERIC:
        {
          std::map<int, double> map_data;
          MapStorage::NumericArcMap& map =
            mapstorage->getNumericArcMap(*it);
          for (ArcIt e(mapstorage->getDigraph()); e != INVALID; ++e)
          {
            map_data[labels[e]] = map[e];
          }
          { x(*it, map_data); }
        }
        break;
      case MapValue::STRING:
        {
          std::map<int, std::string> map_data;
          MapStorage::StringArcMap& map =
            mapstorage->getStringArcMap(*it);
          for (ArcIt e(mapstorage->getDigraph()); e != INVALID; ++e)
          {
            map_data[labels[e]] = map[e];
          }
          { x(*it, map_data); }
        }
        break;
    }
  }

  {
    switch (mapstorage->getNodeCoordsSaveDest())
    {
      case MapStorage::SpecMapSaveOpts::GUI_SECT:
        { x("node_coords_save_dest", string("gui_sect")); }
        // write the node coorinates
        {
          const MapStorage::NodeLabelMap& labels =
            mapstorage->getNodeLabelMap();
          std::map<int, XY> node_coord_map;
          MapStorage::NodeCoordMap& map = mapstorage->getNodeCoordMap();
          for (NodeIt n(mapstorage->getDigraph()); n != INVALID; ++n)
          {
            node_coord_map[labels[n]] = map[n];
          }
          { x("node_coord_map", node_coord_map); }
        }
        break;
      case MapStorage::SpecMapSaveOpts::NESET_SECT:
        switch (mapstorage->getNodeCoordsSaveMapNum())
        {
          case MapStorage::SpecMapSaveOpts::ONE_MAP:
            { x("node_coords_save_dest", string("nodeset_sect_1_map")); }
            { x("map_name", mapstorage->getNodeCoordsOneMapName()); }
            break;
          case MapStorage::SpecMapSaveOpts::TWO_MAPS:
            { x("node_coords_save_dest", string("nodeset_sect_2_maps")); }
            { x("map1_name", mapstorage->getNodeCoordsTwoMaps1Name()); }
            { x("map2_name", mapstorage->getNodeCoordsTwoMaps2Name()); }
            break;
        }
        break;
    }
  }

  {
    switch (mapstorage->getArrowCoordsSaveDest())
    {
      case MapStorage::SpecMapSaveOpts::GUI_SECT:
        { x("arrow_coords_save_dest", string("gui_sect")); }
        // write the arrow coorinates
        {
          const MapStorage::ArcLabelMap& labels =
            mapstorage->getArcLabelMap();
          std::map<int, XY> arrow_coord_map;
          MapStorage::ArrowCoordMap& map = mapstorage->getArrowCoordMap();
          for (ArcIt e(mapstorage->getDigraph()); e != INVALID; ++e)
          {
            arrow_coord_map[labels[e]] = map[e];
          }
          { x("arrow_coord_map", arrow_coord_map); }
        }
        break;
      case MapStorage::SpecMapSaveOpts::NESET_SECT:
        switch (mapstorage->getArrowCoordsSaveMapNum())
        {
          case MapStorage::SpecMapSaveOpts::ONE_MAP:
            { x("arrow_coords_save_dest", string("arcset_sect_1_map")); }
            { x("map_name", mapstorage->getArrowCoordsOneMapName()); }
            break;
          case MapStorage::SpecMapSaveOpts::TWO_MAPS:
            { x("arrow_coords_save_dest", string("arcset_sect_2_maps")); }
            { x("map1_name", mapstorage->getArrowCoordsTwoMaps1Name()); }
            { x("map2_name", mapstorage->getArrowCoordsTwoMaps2Name()); }
            break;
        }
        break;
    }
  }


  std::map<int, std::string> nm;
  for(int i=0;i<NODE_PROPERTY_NUM;i++)
    {
      nm[i]=mapstorage->active_nodemaps[i];
    }
  { x("active_nodemaps", nm); }

  std::map<int, std::string> em;
  for(int i=0;i<EDGE_PROPERTY_NUM;i++)
    {
      em[i]=mapstorage->active_arcmaps[i];
    }
  { x("active_arcmaps", em); }

  double attraction;
  double propulsation;
  int iteration;

  mapstorage->get_design_data(attraction, propulsation, iteration);

  { x("redesign-attraction", attraction); }
  { x("redesign-propulsation", propulsation); }
  { x("redesign-iteration", iteration); }
}

//GuiWriter::GuiWriter(LemonWriter& writer, MapStorage* ms) :
GuiWriter::GuiWriter(MapStorage* ms) :
  //Parent(writer),
  mapstorage(ms)
{
}
