/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include <nbtab.h>
#include <mapstorage.h>
#include <eps_win.h>
#include <map_win.h>
#include <design_win.h>
#include <graph_displayer_canvas.h>



NoteBookTab::NoteBookTab():mapwinexists(false), designwinexists(false), epswinexists(false)
{
  mapstorage=new MapStorage();

  Gtk::ScrolledWindow *pScrolledWindow = Gtk::manage(new Gtk::ScrolledWindow);
  pScrolledWindow->set_shadow_type(Gtk::SHADOW_ETCHED_IN);
  gd_canvas=Gtk::manage(new DigraphDisplayerCanvas(*this));
  pScrolledWindow->add(*gd_canvas);
  add(*pScrolledWindow);

  //connecting signals - controller character
  mapstorage->signal_prop_ch().connect(sigc::mem_fun(*gd_canvas, &DigraphDisplayerCanvas::propertyChange));
  mapstorage->signal_node_map_ch().connect(sigc::mem_fun(*this, &NoteBookTab::registerNewNodeMap));
  mapstorage->signal_arc_map_ch().connect(sigc::mem_fun(*this, &NoteBookTab::registerNewArcMap));
  mapstorage->signal_background_ch().connect(sigc::mem_fun(*gd_canvas, &DigraphDisplayerCanvas::setBackground));
  show_all_children();
  show();
}

NoteBookTab::~NoteBookTab()
{
  delete mapstorage;
}

void NoteBookTab::readFile(const std::string &file)
{
  mapstorage->readFromFile(file);
  mapstorage->setFileName(file);
  mapstorage->setModified(false);
  gd_canvas->drawDigraph();
  if(mapwinexists)
    {
      mapwin->update(
          mapstorage->getArcMapList(NUM),
          mapstorage->getArcMapList(STR),
          mapstorage->getNodeMapList(NUM),
          mapstorage->getNodeMapList(STR));
    }
  title_changed(Glib::filename_display_basename(file));
}

void NoteBookTab::newFile()
{
  if (mapstorage->getModified())
  {
    Gtk::MessageDialog mdialog("<b>Save changes before closing?</b>", true,
        Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE);
    mdialog.add_button("Close file _without Saving", Gtk::RESPONSE_REJECT);
    mdialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    mdialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
    switch (mdialog.run())
    {
      case Gtk::RESPONSE_CANCEL:
        return;
      case Gtk::RESPONSE_REJECT:
        break;
      case Gtk::RESPONSE_ACCEPT:
        saveFile();
        break;
    }
  }
  gd_canvas->clear();
  mapstorage->clear();
  if(mapwinexists)
    {
      mapwin->update(
          mapstorage->getArcMapList(NUM),
          mapstorage->getArcMapList(STR),
          mapstorage->getNodeMapList(NUM),
          mapstorage->getNodeMapList(STR));
    }
  title_changed("unsaved file");
}

void NoteBookTab::openFile()
{
  if (mapstorage->getModified())
  {
    Gtk::MessageDialog mdialog("<b>Save changes before closing?</b>", true, 
        Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE);
    mdialog.add_button("Close file _without Saving", Gtk::RESPONSE_REJECT);
    mdialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    mdialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
    switch (mdialog.run())
    {
      case Gtk::RESPONSE_CANCEL:
        return;
      case Gtk::RESPONSE_REJECT:
        break;
      case Gtk::RESPONSE_ACCEPT:
        saveFile();
        break;
    }
  }
  Gtk::FileChooserDialog fcdialog("Open File");
  fcdialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  fcdialog.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
  if (fcdialog.run() == Gtk::RESPONSE_ACCEPT)
  {
    gd_canvas->clear();
    std::cout << mapstorage << std::endl;
    mapstorage->clear();
    Glib::ustring filename = fcdialog.get_filename();
    if (!mapstorage->readFromFile(filename))
    {
      mapstorage->setFileName(filename);
      mapstorage->setModified(false);
      gd_canvas->drawDigraph();
      if(mapwinexists)
	{
          mapwin->update(
              mapstorage->getArcMapList(NUM),
              mapstorage->getArcMapList(STR),
              mapstorage->getNodeMapList(NUM),
              mapstorage->getNodeMapList(STR));
	}
      title_changed(Glib::filename_display_basename(filename));
    }
  }
}

void NoteBookTab::saveFile()
{
  if (mapstorage->getFileName() == "") {
    saveFileAs();
  }
  else
  {
    mapstorage->writeToFile(mapstorage->getFileName());
    mapstorage->setModified(false);
    title_changed(Glib::filename_display_basename(mapstorage->getFileName()));
  }
}

void NoteBookTab::saveFileAs()
{
  Gtk::FileChooserDialog fcdialog("Save File", Gtk::FILE_CHOOSER_ACTION_SAVE);
  fcdialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  fcdialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
  if (fcdialog.run() == Gtk::RESPONSE_ACCEPT)
  {
    Glib::ustring filename = fcdialog.get_filename();
    mapstorage->setFileName(filename);
    mapstorage->writeToFile(filename);
    mapstorage->setModified(false);
    title_changed(Glib::filename_display_basename(filename));
  }
}

void NoteBookTab::close()
{
  if (mapstorage->getModified())
  {
    Gtk::MessageDialog mdialog("<b>Save changes before closing?</b>", true,
        Gtk::MESSAGE_WARNING, Gtk::BUTTONS_NONE);
    mdialog.add_button("Close _without Saving", Gtk::RESPONSE_REJECT);
    mdialog.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    mdialog.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
    switch (mdialog.run())
    {
      case Gtk::RESPONSE_CANCEL:
        return;
      case Gtk::RESPONSE_REJECT:
        break;
      case Gtk::RESPONSE_ACCEPT:
        saveFile();
        break;
    }
  }
  gd_canvas->clear();
  mapstorage->clear();
  if(mapwinexists)
    {
      mapwin->update(
          mapstorage->getArcMapList(NUM),
          mapstorage->getArcMapList(STR),
          mapstorage->getNodeMapList(NUM),
          mapstorage->getNodeMapList(STR));
    }
  title_changed("unsaved file");
}

void NoteBookTab::propertyChange(bool itisarc, int prop, std::string mapname)
{
  mapstorage->changeActiveMap(itisarc, prop, mapname);
}

sigc::signal<void, NoteBookTab *, bool> NoteBookTab::signal_newmap_needed()
{
  return signal_newmap;
}

void NoteBookTab::popupNewMapWin(bool itisarc)
{
  signal_newmap.emit(this, itisarc);
}

std::string NoteBookTab::getActiveArcMap(int prop)
{
  return mapstorage->getActiveArcMap(prop);
}

std::string NoteBookTab::getActiveNodeMap(int prop)
{
  return mapstorage->getActiveNodeMap(prop);
}

void NoteBookTab::registerNewArcMap(std::string mapname, MapValue::Type type)
{
  if(mapwinexists)
    {
      mapwin->registerNewArcMap(mapname, type);
    }
}

void NoteBookTab::registerNewNodeMap(std::string mapname, MapValue::Type type)
{
  if(mapwinexists)
    {
      mapwin->registerNewNodeMap(mapname, type);
    }
  if(epswinexists)
    {
      epswin->registerNewNodeMap(mapname, type);
    }
}

void NoteBookTab::createMapWin(std::string name)
{
  if(!mapwinexists)
    {
      mapwin=new MapWin("Map Setup - "+name,
          mapstorage->getArcMapList(NUM),
          mapstorage->getArcMapList(STR),
          mapstorage->getNodeMapList(NUM),
          mapstorage->getNodeMapList(STR),
          *this);
      mapst2mapwin=mapstorage->signal_map_win_ch().connect(sigc::mem_fun(*mapwin, &MapWin::changeEntry));
      mapwin->show();
      mapwinexists=true;
    }
}

void NoteBookTab::createExportToEPSWin(std::string name)
{
  if(!epswinexists)
    {
      epswin=new EpsWin("Export to EPS - "+name, mapstorage->getNodeMapList(NUM), mapstorage->getNodeMapList(STR));
      epswin->show();
      epswinexists=true;
      epswin->signal_eps_details_ch().connect(sigc::mem_fun(*this, &NoteBookTab::exportDigraphToEPS));
      epswin->signal_eps_close_ch().connect(sigc::mem_fun(*this, &NoteBookTab::closeEpsWin));
      epswin->signal_new_map_ch().connect(sigc::mem_fun(*this, &NoteBookTab::popupNewMapWin));
    }
}


void NoteBookTab::createDesignWin(std::string name)
{
  if(!designwinexists)
    {
      double attraction, propulsation;
      int iterations;
      mapstorage->get_design_data(attraction, propulsation, iterations);
      designwin=new DesignWin("Design Setup - "+name, attraction, propulsation, iterations);

      designwin->signal_attraction().connect(sigc::mem_fun(*mapstorage, &MapStorage::set_attraction));
      designwin->signal_propulsation().connect(sigc::mem_fun(*mapstorage, &MapStorage::set_propulsation));
      designwin->signal_iteration().connect(sigc::mem_fun(*mapstorage, &MapStorage::set_iteration));
      designwin->close_run().connect(sigc::mem_fun(*gd_canvas, &DigraphDisplayerCanvas::reDesignDigraph));

      designwin->signal_delete_event().connect(sigc::mem_fun(*this, &NoteBookTab::closeDesignWin));

      mapst2designwin=mapstorage->signal_design_win_ch().connect(sigc::mem_fun(*designwin, &DesignWin::set_data));

      designwin->show();
      designwinexists=true;
    }
}

void NoteBookTab::closeMapWin()
{
  mapst2mapwin.disconnect();
  mapwinexists=false;
  delete mapwin;
}

void NoteBookTab::closeEpsWin()
{
  epswinexists=false;
  delete epswin;
}

bool NoteBookTab::closeDesignWin(GdkEventAny * e)
{
  if(e->type==GDK_DELETE)
    {
      designwinexists=false;
      mapst2designwin.disconnect();
      delete designwin;
    }
}

sigc::signal<void, std::string> NoteBookTab::signal_title_ch()
{
  return signal_title;
}

void NoteBookTab::setView(bool autoscale, bool zoomtrack, double width, double radius)
{
  gd_canvas->setView(autoscale, zoomtrack, width, radius);
}

void NoteBookTab::getView(bool & autoscale, bool & zoomtrack, double& width, double& radius)
{
  gd_canvas->getView(autoscale, zoomtrack, width, radius);
}

void NoteBookTab::reDesignDigraph()
{
  gd_canvas->reDesignDigraph();
}

void NoteBookTab::active_maps_needed()
{
  mapstorage->broadcastActiveMaps();
}

void NoteBookTab::exportDigraphToEPS(std::vector<bool> options, std::string filename, std::string shapemap)
{
  mapstorage->exportDigraphToEPS(options, filename, shapemap);
}

void NoteBookTab::title_changed(std::string newtitle)
{
  signal_title.emit(newtitle);
  if(epswinexists)
    {
      epswin->set_title(newtitle);
    }
  if(designwinexists)
    {
      designwin->set_title(newtitle);
    }
}
