// -*- c++ -*-
#ifndef HUGO_SYM_MAP_H
#define HUGO_SYM_MAP_H

///\ingroup graphmaps
///\file
///\brief Graph maps that construates and destruates
///their elements dynamically.

namespace hugo {

/// \addtogroup graphmaps
/// @{

  /** The SymEdgeIt is wrapper class for the EdgeIt. It can be used to
   *  iterate on the symmetric maps when all of the edge - reverse edge pair
   *  has different parity.
   */


  template <typename Graph, typename Edge, typename EdgeIt>
  class SymEdgeIt : public EdgeIt {
  public:

    /** Default constructor.
     */
    SymEdgeIt() 
      : EdgeIt() {}

    /** Graph initialized constructor.
     */
    SymEdgeIt(const Graph& graph) 
      : EdgeIt(graph) {
      while ( (EdgeIt::n & 1) && EdgeIt::n != -1) {
	EdgeIt::operator++();
      }
    }

    /** Creating invelid SymEdgeIt.
     */
    SymEdgeIt(Invalid invalid) 
      : EdgeIt(invalid) {}

    /** SymEdgeIt from the given Edge.
     */
    SymEdgeIt(const Graph& graph, const Edge& edge)
      : EdgeIt(graph, edge) {
      while ( (EdgeIt::n & 1) && EdgeIt::n != -1) {
	EdgeIt::operator++();
      }
    }

    /** Increase operator.
     */
    SymEdgeIt& operator++() {
      EdgeIt::operator++();
      while ( (EdgeIt::n & 1) && EdgeIt::n != -1) {
	EdgeIt::operator++();
      }
      return *this;
    }
  };

  /** The SymMap template class is graph map structure what
   *  wraps an other map structure to use as symmetric map structure.
   *
   *  The template parameter is the MapRegistry that the maps
   *  will belong to and the ValueType.
   */
  template <template <typename, typename> class DynMap, 
	    typename MapRegistry, typename Value>
  class SymMap : public DynMap<MapRegistry, Value>{

  private:

    typedef DynMap<MapRegistry, Value> MapImpl;

  public:
		
    /// The graph type of the maps. 
    typedef typename MapRegistry::Graph Graph;

    typedef typename MapImpl::KeyType KeyType;

  public:


    /** Graph and Registry initialized map constructor.
     */
    SymMap(const Graph& g, MapRegistry& r) : MapImpl(g, r) {}

    /** Constructor to use default value to initialize the map. 
     */
    SymMap(const Graph& g, MapRegistry& r, const Value& v) 
      : MapImpl(g, r, v) {}

    /** Constructor to copy a map of the same map type.
     */
    SymMap(const SymMap& copy) 
      : MapImpl(static_cast<const MapImpl&>(copy)) {}

    /** Assign operator to copy a map of the same map type.
     */
    SymMap& operator=(const SymMap& copy) {
      MapImpl::operator=(static_cast<const MapImpl&>(copy));
      return *this;
    }

    /** Add a new key to the map. It called by the map registry.
     */
    void add(const KeyType& key) {
      int id = MapImpl::getGraph()->id(key);
      if (id & 1) return;
      MapImpl::add(key);
    }
		
    /** Erase a key from the map. It called by the map registry.
     */
    void erase(const KeyType& key) {
      int id = MapImpl::getGraph()->id(key);
      if (id & 1) return;
      MapImpl::add(key);
    }
  };

  /// @}
}

#endif
