// -*- C++ -*-
#ifndef HUGO_BFS_H
#define HUGO_BFS_H

///\ingroup flowalgs
///\file
///\brief Bfs algorithm.
///
///\todo Revise Manual.

#include <hugo/bin_heap.h>
#include <hugo/invalid.h>

namespace hugo {

/// \addtogroup flowalgs
/// @{

  ///%Bfs algorithm class.

  ///This class provides an efficient implementation of %Bfs algorithm.
  ///The edge lengths are passed to the algorithm using a
  ///\ref ReadMapSkeleton "readable map",
  ///so it is easy to change it to any kind of length.
  ///
  ///The type of the length is determined by the \c ValueType of the length map.
  ///
  ///It is also possible to change the underlying priority heap.
  ///
  ///\param GR The graph type the algorithm runs on.
  ///\param LM This read-only
  ///EdgeMap
  ///determines the
  ///lengths of the edges. It is read once for each edge, so the map
  ///may involve in relatively time consuming process to compute the edge
  ///length if it is necessary. The default map type is
  ///\ref GraphSkeleton::EdgeMap "Graph::EdgeMap<int>"
  ///\param Heap The heap type used by the %Bfs
  ///algorithm. The default
  ///is using \ref BinHeap "binary heap".
  ///
  ///\author Jacint Szabo and Alpar Juttner
  ///\todo We need a typedef-names should be standardized. (-:
  ///\todo Type of \c PredMap, \c PredNodeMap and \c DistMap
  ///should not be fixed. (Problematic to solve).

#ifdef DOXYGEN
  template <typename GR>
#else
  template <typename GR>
#endif
  class Bfs{
  public:
    ///The type of the underlying graph.
    typedef GR Graph;
    typedef typename Graph::Node Node;
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::Edge Edge;
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    
    ///\brief The type of the map that stores the last
    ///edges of the shortest paths.
    typedef typename Graph::template NodeMap<Edge> PredMap;
    ///\brief The type of the map that stores the last but one
    ///nodes of the shortest paths.
    typedef typename Graph::template NodeMap<Node> PredNodeMap;
    ///The type of the map that stores the dists of the nodes.
    typedef typename Graph::template NodeMap<int> DistMap;

  private:
    const Graph *G;
    PredMap *predecessor;
    bool local_predecessor;
    PredNodeMap *pred_node;
    bool local_pred_node;
    DistMap *distance;
    bool local_distance;

    //The source node of the last execution.
    Node source;


    ///Initialize maps.
    
    ///\todo Error if \c G or are \c NULL.
    ///\todo Better memory allocation (instead of new).
    void init_maps() 
    {
//       if(!length) {
// 	local_length = true;
// 	length = new LM(G);
//       }
      if(!predecessor) {
	local_predecessor = true;
	predecessor = new PredMap(*G);
      }
      if(!pred_node) {
	local_pred_node = true;
	pred_node = new PredNodeMap(*G);
      }
      if(!distance) {
	local_distance = true;
	distance = new DistMap(*G);
      }
    }
    
  public :    
    Bfs(const Graph& _G) :
      G(&_G),
      predecessor(NULL), local_predecessor(false),
      pred_node(NULL), local_pred_node(false),
      distance(NULL), local_distance(false)
    { }
    
    ~Bfs() 
    {
      //      if(local_length) delete length;
      if(local_predecessor) delete predecessor;
      if(local_pred_node) delete pred_node;
      if(local_distance) delete distance;
    }

    ///Sets the graph the algorithm will run on.

    ///Sets the graph the algorithm will run on.
    ///\return <tt> (*this) </tt>
    Bfs &setGraph(const Graph &_G) 
    {
      G = &_G;
      return *this;
    }
    ///Sets the length map.

    ///Sets the map storing the predecessor edges.

    ///Sets the map storing the predecessor edges.
    ///If you don't use this function before calling \ref run(),
    ///it will allocate one. The destuctor deallocates this
    ///automatically allocated map, of course.
    ///\return <tt> (*this) </tt>
    Bfs &setPredMap(PredMap &m) 
    {
      if(local_predecessor) {
	delete predecessor;
	local_predecessor=false;
      }
      predecessor = &m;
      return *this;
    }

    ///Sets the map storing the predecessor nodes.

    ///Sets the map storing the predecessor nodes.
    ///If you don't use this function before calling \ref run(),
    ///it will allocate one. The destuctor deallocates this
    ///automatically allocated map, of course.
    ///\return <tt> (*this) </tt>
    Bfs &setPredNodeMap(PredNodeMap &m) 
    {
      if(local_pred_node) {
	delete pred_node;
	local_pred_node=false;
      }
      pred_node = &m;
      return *this;
    }

    ///Sets the map storing the distances calculated by the algorithm.

    ///Sets the map storing the distances calculated by the algorithm.
    ///If you don't use this function before calling \ref run(),
    ///it will allocate one. The destuctor deallocates this
    ///automatically allocated map, of course.
    ///\return <tt> (*this) </tt>
    Bfs &setDistMap(DistMap &m) 
    {
      if(local_distance) {
	delete distance;
	local_distance=false;
      }
      distance = &m;
      return *this;
    }
    
  ///Runs %BFS algorithm from node \c s.

  ///This method runs the %BFS algorithm from a root node \c s
  ///in order to
  ///compute the
  ///shortest path to each node. The algorithm computes
  ///- The shortest path tree.
  ///- The distance of each node from the root.
 
    void run(Node s) {
      
      init_maps();
      
      source = s;
      
      for ( NodeIt u(*G) ; u!=INVALID ; ++u ) {
	predecessor->set(u,INVALID);
	pred_node->set(u,INVALID);
      }
      
      int N=G->nodeNum();
      std::vector<typename Graph::Node> Q(N);
      int Qh=0;
      int Qt=0;
      
      Q[Qh++]=source;
      distance->set(s, 0);
      do {
	Node m;
	Node n=Q[Qt++];
	int d= (*distance)[n]+1;
	
	for(OutEdgeIt e(*G,n);e!=INVALID;++e)
	  if((m=G->head(e))!=s && (*predecessor)[m]==INVALID) {
	    Q[Qh++]=m;
	    predecessor->set(m,e);
	    pred_node->set(m,n);
	    distance->set(m,d);
	  }
      } while(Qt!=Qh);
    }
    
    ///The distance of a node from the root.

    ///Returns the distance of a node from the root.
    ///\pre \ref run() must be called before using this function.
    ///\warning If node \c v in unreachable from the root the return value
    ///of this funcion is undefined.
    int dist(Node v) const { return (*distance)[v]; }

    ///Returns the 'previous edge' of the shortest path tree.

    ///For a node \c v it returns the 'previous edge' of the shortest path tree,
    ///i.e. it returns the last edge from a shortest path from the root to \c
    ///v. It is \ref INVALID
    ///if \c v is unreachable from the root or if \c v=s. The
    ///shortest path tree used here is equal to the shortest path tree used in
    ///\ref predNode(Node v).  \pre \ref run() must be called before using
    ///this function.
    Edge pred(Node v) const { return (*predecessor)[v]; }

    ///Returns the 'previous node' of the shortest path tree.

    ///For a node \c v it returns the 'previous node' of the shortest path tree,
    ///i.e. it returns the last but one node from a shortest path from the
    ///root to \c /v. It is INVALID if \c v is unreachable from the root or if
    ///\c v=s. The shortest path tree used here is equal to the shortest path
    ///tree used in \ref pred(Node v).  \pre \ref run() must be called before
    ///using this function.
    Node predNode(Node v) const { return (*pred_node)[v]; }
    
    ///Returns a reference to the NodeMap of distances.
    
    ///Returns a reference to the NodeMap of distances. \pre \ref run() must
    ///be called before using this function.
    const DistMap &distMap() const { return *distance;}
 
    ///Returns a reference to the shortest path tree map.

    ///Returns a reference to the NodeMap of the edges of the
    ///shortest path tree.
    ///\pre \ref run() must be called before using this function.
    const PredMap &predMap() const { return *predecessor;}
 
    ///Returns a reference to the map of nodes of shortest paths.

    ///Returns a reference to the NodeMap of the last but one nodes of the
    ///shortest path tree.
    ///\pre \ref run() must be called before using this function.
    const PredNodeMap &predNodeMap() const { return *pred_node;}

    ///Checks if a node is reachable from the root.

    ///Returns \c true if \c v is reachable from the root.
    ///\warning The root node is reported to be reached!
    ///
    ///\pre \ref run() must be called before using this function.
    ///
    bool reached(Node v) { return v==source || (*predecessor)[v]==INVALID; }
    
  };
  

  // **********************************************************************
  //  IMPLEMENTATIONS
  // **********************************************************************

/// @}
  
} //END OF NAMESPACE HUGO

#endif


