// -*- c++ -*-
#ifndef HUGO_LIST_GRAPH_H
#define HUGO_LIST_GRAPH_H

#include <iostream>
#include <vector>

#include "invalid.h"

#include "map_registry.h"
#include "array_map_factory.h"

#include "map_defines.h"

namespace hugo {

  template <typename It>
  int count(It it) { 
    int i=0;
    for( ; it.valid(); ++it) { ++i; } 
    return i;
  }

  class ListGraph {
    class node_item;
    class edge_item;
  public:
    class Node;
    class NodeIt;
    class Edge;
    class EdgeIt;
    class OutEdgeIt;
    class InEdgeIt;
    class SymEdgeIt;
    
    typedef ListGraph Graph;
    
    CREATE_MAP_REGISTRIES;
    CREATE_MAPS(ArrayMapFactory);
    
  private:
    
    int node_id;
    int edge_id;
    int _node_num;
    int _edge_num;
    
    node_item* _first_node;
    node_item* _last_node;

    class node_item {
      friend class ListGraph;
      template <typename T> friend class NodeMap;
      
      friend class Node;
      friend class NodeIt;
      friend class Edge;
      friend class EdgeIt;
      friend class OutEdgeIt;
      friend class InEdgeIt;
      friend class SymEdgeIt;
      friend std::ostream& operator<<(std::ostream& os, const Node& i);
      friend std::ostream& operator<<(std::ostream& os, const Edge& i);
      //ListGraph* G;
      int id;
      edge_item* _first_out_edge;
      edge_item* _last_out_edge;
      edge_item* _first_in_edge;
      edge_item* _last_in_edge;
      node_item* _next_node;
      node_item* _prev_node;
    public:
      node_item() { }
    };

    class edge_item {
      friend class ListGraph;
      template <typename T> friend class EdgeMap;

      friend class Node;
      friend class NodeIt;
      friend class Edge;
      friend class EdgeIt;
      friend class OutEdgeIt;
      friend class InEdgeIt;
      friend class SymEdgeIt;
      friend std::ostream& operator<<(std::ostream& os, const Edge& i);
      //ListGraph* G;
      int id;
      node_item* _tail;
      node_item* _head;
      edge_item* _next_out;
      edge_item* _prev_out;
      edge_item* _next_in;
      edge_item* _prev_in;
    public:
      edge_item() { }
    };

    node_item* _add_node() { 
      node_item* p=new node_item;
      p->id=node_id++;
      p->_first_out_edge=0;
      p->_last_out_edge=0;
      p->_first_in_edge=0;
      p->_last_in_edge=0;
      p->_prev_node=_last_node;
      p->_next_node=0;
      if (_last_node) _last_node->_next_node=p;
      _last_node=p;
      if (!_first_node) _first_node=p;

      ++_node_num;
      return p;
    }

    edge_item* _add_edge(node_item* _tail, node_item* _head) {
      edge_item* e=new edge_item;
      e->id=edge_id++;
      e->_tail=_tail;
      e->_head=_head;
      
      e->_prev_out=_tail->_last_out_edge;
      if (_tail->_last_out_edge) (_tail->_last_out_edge)->_next_out=e;
      _tail->_last_out_edge=e;
      if (!_tail->_first_out_edge) _tail->_first_out_edge=e; 
      e->_next_out=0;
 
      e->_prev_in=_head->_last_in_edge;
      if (_head->_last_in_edge) (_head->_last_in_edge)->_next_in=e;
      _head->_last_in_edge=e;
      if (!_head->_first_in_edge) { _head->_first_in_edge=e; } 
      e->_next_in=0;

      ++_edge_num;
      return e;
    }

    //deletes a node which has no out edge and no in edge
    void _delete_node(node_item* v) {
      if (v->_next_node) (v->_next_node)->_prev_node=v->_prev_node; else 
	_last_node=v->_prev_node;
      if (v->_prev_node) (v->_prev_node)->_next_node=v->_next_node; else 
	_first_node=v->_next_node;

      delete v;
      --_node_num;
    }

    void _delete_edge(edge_item* e) {
      if (e->_next_out) (e->_next_out)->_prev_out=e->_prev_out; else 
	(e->_tail)->_last_out_edge=e->_prev_out;
      if (e->_prev_out) (e->_prev_out)->_next_out=e->_next_out; else 
	(e->_tail)->_first_out_edge=e->_next_out;
      if (e->_next_in) (e->_next_in)->_prev_in=e->_prev_in; else 
	(e->_head)->_last_in_edge=e->_prev_in;
      if (e->_prev_in) (e->_prev_in)->_next_in=e->_next_in; else 
	(e->_head)->_first_in_edge=e->_next_in;

      delete e;
      --_edge_num;
    }

    void _set_tail(edge_item* e, node_item* _tail) {
      if (e->_next_out) (e->_next_out)->_prev_out=e->_prev_out; else 
	(e->_tail)->_last_out_edge=e->_prev_out;
      if (e->_prev_out) (e->_prev_out)->_next_out=e->_next_out; else 
	(e->_tail)->_first_out_edge=e->_next_out;
      
      e->_tail=_tail;
      
      e->_prev_out=_tail->_last_out_edge;
      if (_tail->_last_out_edge) (_tail->_last_out_edge)->_next_out=e;
      _tail->_last_out_edge=e;
      if (!_tail->_first_out_edge) _tail->_first_out_edge=e; 
      e->_next_out=0;
    }

    void _set_head(edge_item* e, node_item* _head) {
      if (e->_next_in) (e->_next_in)->_prev_in=e->_prev_in; else 
	(e->_head)->_last_in_edge=e->_prev_in;
      if (e->_prev_in) (e->_prev_in)->_next_in=e->_next_in; else 
	(e->_head)->_first_in_edge=e->_next_in;
      
      e->_head=_head;
      
      e->_prev_in=_head->_last_in_edge;
      if (_head->_last_in_edge) (_head->_last_in_edge)->_next_in=e;
      _head->_last_in_edge=e;
      if (!_head->_first_in_edge) { _head->_first_in_edge=e; } 
      e->_next_in=0;
    }

  public:

    /* default constructor */

    ListGraph() : node_id(0), edge_id(0), _node_num(0), _edge_num(0), _first_node(0), _last_node(0){ }
    
    ~ListGraph() { 
      while (first<NodeIt>().valid()) erase(first<NodeIt>());
    }

    int nodeNum() const { return _node_num; }
    int edgeNum() const { return _edge_num; }

    /* functions to construct iterators from the graph, or from each other */

    //NodeIt firstNode() const { return NodeIt(*this); }
    //EdgeIt firstEdge() const { return EdgeIt(*this); }
    
    //OutEdgeIt firstOutEdge(const Node v) const { return OutEdgeIt(v); }
    //InEdgeIt firstInEdge(const Node v) const { return InEdgeIt(v); }
    //SymEdgeIt firstSymEdge(const Node v) const { return SymEdgeIt(v); }
    Node tail(Edge e) const { return e.tailNode(); }
    Node head(Edge e) const { return e.headNode(); }

    Node aNode(const OutEdgeIt& e) const { return e.aNode(); }
    Node aNode(const InEdgeIt& e) const { return e.aNode(); }
    Node aNode(const SymEdgeIt& e) const { return e.aNode(); }

    Node bNode(const OutEdgeIt& e) const { return e.bNode(); }
    Node bNode(const InEdgeIt& e) const { return e.bNode(); }
    Node bNode(const SymEdgeIt& e) const { return e.bNode(); }

    //Node invalid_node() { return Node(); }
    //Edge invalid_edge() { return Edge(); }
    //OutEdgeIt invalid_out_edge() { return OutEdgeIt(); }
    //InEdgeIt invalid_in_edge() { return InEdgeIt(); }
    //SymEdgeIt invalid_sym_edge() { return SymEdgeIt(); }

    /* same methods in other style */
    /* for experimental purpose */

    NodeIt& /*getF*/first(NodeIt& v) const { 
      v=NodeIt(*this); return v; }
    EdgeIt& /*getF*/first(EdgeIt& e) const { 
      e=EdgeIt(*this); return e; }
    OutEdgeIt& /*getF*/first(OutEdgeIt& e, Node v) const { 
      e=OutEdgeIt(*this, v); return e; }
    InEdgeIt& /*getF*/first(InEdgeIt& e, Node v) const { 
      e=InEdgeIt(*this, v); return e; }
    SymEdgeIt& /*getF*/first(SymEdgeIt& e, Node v) const { 
      e=SymEdgeIt(*this, v); return e; }
    //void getTail(Node& n, const Edge& e) const { n=tail(e); }
    //void getHead(Node& n, const Edge& e) const { n=head(e); }

    //void getANode(Node& n, const OutEdgeIt& e) const { n=e.aNode(); }
    //void getANode(Node& n, const InEdgeIt& e) const { n=e.aNode(); }
    //void getANode(Node& n, const SymEdgeIt& e) const { n=e.aNode(); }
    //void getBNode(Node& n, const OutEdgeIt& e) const { n=e.bNode(); }
    //void getBNode(Node& n, const InEdgeIt& e) const { n=e.bNode(); }
    //void getBNode(Node& n, const SymEdgeIt& e) const { n=e.bNode(); }
    //void get_invalid(Node& n) { n=Node(); }
    //void get_invalid(Edge& e) { e=Edge(); }
    //void get_invalid(OutEdgeIt& e) { e=OutEdgeIt(); }
    //void get_invalid(InEdgeIt& e) { e=InEdgeIt(); }
    //void get_invalid(SymEdgeIt& e) { e=SymEdgeIt(); }

    template< typename It >
    It first() const { 
      It e;
      /*getF*/first(e);
      return e; 
    }

    template< typename It >
    It first(Node v) const { 
      It e;
      /*getF*/first(e, v);
      return e; 
    }

    bool valid(Node n) const { return n.valid(); }
    bool valid(Edge e) const { return e.valid(); }
    
    template <typename It> It getNext(It it) const { 
      It tmp(it); return next(tmp); }
    template <typename It> It& next(It& it) const { return ++it; }
   

    /* for getting id's of graph objects */
    /* these are important for the implementation of property vectors */

    int id(Node v) const { return v.node->id; }
    int id(Edge e) const { return e.edge->id; }

    /* adding nodes and edges */

    Node addNode() { 
      Node n = _add_node();
      node_maps.add(n);
      return n; 
    }
    Edge addEdge(Node u, Node v) {
      Edge e = _add_edge(u.node, v.node);
      edge_maps.add(e);
      return e; 
    }

    void erase(Node i) { 
      node_maps.erase(i);
      while (first<OutEdgeIt>(i).valid()) erase(first<OutEdgeIt>(i));
      while (first<InEdgeIt>(i).valid()) erase(first<InEdgeIt>(i));
      _delete_node(i.node); 
    }
  
    void erase(Edge e) { 
      edge_maps.erase(e);
      _delete_edge(e.edge); 
    }

    void clear() { 
      while (first<NodeIt>().valid()) erase(first<NodeIt>());
    }

    void setTail(Edge e, Node tail) {
      _set_tail(e.edge, tail.node); 
    }

    void setHead(Edge e, Node head) {
      _set_head(e.edge, head.node); 
    }

    /* stream operations, for testing purpose */

    friend std::ostream& operator<<(std::ostream& os, const Node& i) { 
      os << i.node->id; return os; 
    }
    friend std::ostream& operator<<(std::ostream& os, const Edge& i) { 
      os << "(" << i.edge->_tail->id << "--" << i.edge->id << "->" << i.edge->_head->id << ")"; 
      return os; 
    }

    class Node {
      friend class ListGraph;
      template <typename T> friend class NodeMap;

      friend class Edge;
      friend class OutEdgeIt;
      friend class InEdgeIt;
      friend class SymEdgeIt;
      //public:  //FIXME: It is required by op= of NodeIt
    protected:
      node_item* node;
    protected:
      friend int ListGraph::id(Node v) const; 
    public:
      Node() /*: node(0)*/ { }
      Node(const Invalid&) : node(0) { }
    protected:
      Node(node_item* _node) : node(_node) { }
      bool valid() const { return (node); }
    public:
      //void makeInvalid() { node=0; }
      friend bool operator==(Node u, Node v) { return v.node==u.node; } 
      friend bool operator!=(Node u, Node v) { return v.node!=u.node; } 
      friend std::ostream& operator<<(std::ostream& os, const Node& i);
    };
    
    class NodeIt : public Node {
      friend class ListGraph;
      //protected:
    public: //for everybody but marci
      NodeIt(const ListGraph& G) : Node(G._first_node) { }
    public:
      NodeIt() : Node() { }
      NodeIt(const Invalid& i) : Node(i) { }
    protected:
      NodeIt(node_item* v) : Node(v) { }
      NodeIt& operator++() { node=node->_next_node; return *this; }
      //FIXME::
      //      NodeIt& operator=(const Node& e)
      //      { node=e.node; return *this; }
    };

    class Edge {
      friend class ListGraph;
      template <typename T> friend class EdgeMap;
      
      friend class Node;
      friend class NodeIt;
    protected:
      edge_item* edge;
      friend int ListGraph::id(Edge e) const;
    public:
      Edge() /*: edge(0)*/ { }
      Edge(const Invalid&) : edge(0) { }
      //Edge() { }
    protected:
      Edge(edge_item* _edge) : edge(_edge) { }
      bool valid() const { return (edge); }
    public:
      //void makeInvalid() { edge=0; }
      friend bool operator==(Edge u, Edge v) { return v.edge==u.edge; } 
      friend bool operator!=(Edge u, Edge v) { return v.edge!=u.edge; } 
    protected:
      Node tailNode() const { return Node(edge->_tail); }
      Node headNode() const { return Node(edge->_head); }
    public:
      friend std::ostream& operator<<(std::ostream& os, const Edge& i);
    };
    
    class EdgeIt : public Edge {
      friend class ListGraph;
      //protected: 
    public: //for alpar
      EdgeIt(const ListGraph& G) {
	node_item* v=G._first_node;
	if (v) edge=v->_first_out_edge; else edge=0;
	while (v && !edge) { v=v->_next_node; if (v) edge=v->_first_out_edge; }
      }
    public:
      EdgeIt() : Edge() { }
      EdgeIt(const Invalid& i) : Edge(i) { }
    protected:
      EdgeIt(edge_item* _e) : Edge(_e) { }
      EdgeIt& operator++() { 
	node_item* v=edge->_tail;
	edge=edge->_next_out; 
	while (v && !edge) { v=v->_next_node; if (v) edge=v->_first_out_edge; }
	return *this;
      }
    };
    
    class OutEdgeIt : public Edge {
      friend class ListGraph;
      //node_item* v;
      //protected: 
    protected: //for alpar
      OutEdgeIt(const Node& _v) /*: v(_v.node)*/ { edge=_v.node->_first_out_edge; }
    public:
      OutEdgeIt() : Edge()/*, v(0)*/ { }
      OutEdgeIt(const Invalid& i) : Edge(i) { }
      OutEdgeIt(const ListGraph&, Node _v) /*: v(_v.node)*/ { edge=_v.node->_first_out_edge; }
    protected:
      OutEdgeIt& operator++() { edge=edge->_next_out; return *this; }
    protected:
      Node aNode() const { return Node(edge->_tail); }
      Node bNode() const { return Node(edge->_head); }
    };
    
    class InEdgeIt : public Edge {
      friend class ListGraph;
      //node_item* v;
      //protected:
    protected: //for alpar
      InEdgeIt(const Node& _v) /*: v(_v.node)*/ { edge=_v.node->_first_in_edge; }
    public:
      InEdgeIt() : Edge()/*, v(0)*/ { }
      InEdgeIt(const Invalid& i) : Edge(i) { }
      InEdgeIt(const ListGraph&, Node _v) /*: v(_v.node)*/ { edge=_v.node->_first_in_edge; }
    protected:
      InEdgeIt& operator++() { edge=edge->_next_in; return *this; }
    protected:
      Node aNode() const { return Node(edge->_head); }
      Node bNode() const { return Node(edge->_tail); }
    };

    class SymEdgeIt : public Edge {
      friend class ListGraph;
      bool out_or_in; //1 iff out, 0 iff in
      //node_item* v;
      //protected:
    public: //for alpar
      SymEdgeIt(const Node& _v) /*: v(_v.node)*/ { 
	out_or_in=1;
	edge=_v.node->_first_out_edge; 
	if (!edge) { edge=_v.node->_first_in_edge; out_or_in=0; }
      }
    public:
      SymEdgeIt() : Edge() /*, v(0)*/ { }
      SymEdgeIt(const Invalid& i) : Edge(i) { }
      SymEdgeIt(const ListGraph&, Node _v) /*: v(_v.node)*/ { 
	out_or_in=1;
	edge=_v.node->_first_out_edge; 
	if (!edge) { edge=_v.node->_first_in_edge; out_or_in=0; }
      }
    protected:
      SymEdgeIt& operator++() { 
	if (out_or_in) { 
	  node_item* v=edge->_tail;
	  edge=edge->_next_out; 
	  if (!edge) { out_or_in=0; edge=v->_first_in_edge; }
	} else {
	  edge=edge->_next_in; 
	}
	return *this;
      }
    protected:
      Node aNode() const { 
	return (out_or_in) ? Node(edge->_tail) : Node(edge->_head); }
      Node bNode() const { 
	return (out_or_in) ? Node(edge->_head) : Node(edge->_tail); }
    };

  };

  //   template< typename T >
  //   T ListGraph::first() const { 
  //     std::cerr << "Invalid use of template<typemane T> T ListGraph::first<T>();" << std::endl; 
  //     return T(); 
  //   }

  //   template<>
  //   ListGraph::NodeIt ListGraph::first<ListGraph::NodeIt>() const { 
  //     return firstNode(); 
  //   }

  //   template<>
  //   ListGraph::EdgeIt ListGraph::first<ListGraph::EdgeIt>() const { 
  //     return firstEdge(); 
  //   }

  //   template< typename T >
  //   T ListGraph::first(ListGraph::Node v) const {
  //     std::cerr << "Invalid use of template<typemane T> T ListGraph::first<T>(ListGRaph::Node);" << std::endl; 
  //     return T(); 
  //   } 

  //   template<>
  //   ListGraph::OutEdgeIt ListGraph::first<ListGraph::OutEdgeIt>(const ListGraph::Node v) const { 
  //     return firstOutEdge(v); 
  //   }

  //   template<>
  //   ListGraph::InEdgeIt ListGraph::first<ListGraph::InEdgeIt>(const ListGraph::Node v) const { 
  //     return firstInEdge(v); 
  //   }

  //   template<>
  //   ListGraph::SymEdgeIt ListGraph::first<ListGraph::SymEdgeIt>(const ListGraph::Node v) const { 
  //     return firstSymEdge(v); 
  //   }


} //namespace hugo

#endif //HUGO_LIST_GRAPH_H
