// -*- c++ -*-
#ifndef LEMON_NET_GRAPH_H
#define LEMON_NET_GRAPH_H

///\file
///\brief Declaration of HierarchyGraph.

#include <lemon/invalid.h>
#include <lemon/maps.h>

/// The namespace of LEMON
namespace lemon
{

  // @defgroup empty_graph The HierarchyGraph class
  // @{

  /// A graph class in that a simple edge can represent a path.

  /// This class provides common features of a graph structure
  /// that represents a network. You can handle with it layers. This
  /// means that a node in one layer can be a complete network in a nother
  /// layer.

  template < class Gact, class Gsub > class HierarchyGraph
  {

  public:

    /// The actual layer
    Gact actuallayer;


    /// Map of the subnetworks in the sublayer
    /// The appropriate edge nodes are also stored here

    class SubNetwork
    {

      struct actedgesubnodestruct
      {
	typename Gact::Edge actedge;
	typename Gsub::Node subnode;
      };

      int edgenumber;
      bool connectable;
      Gact *actuallayer;
      typename Gact::Node * actuallayernode;
      Gsub *subnetwork;
      actedgesubnodestruct *assignments;

    public:

      int addAssignment (typename Gact::Edge actedge,
			 typename Gsub::Node subnode)
      {
	if (!(actuallayer->valid (actedge)))
	  {
	    cerr << "The given edge is not in the given network!" << endl;
	    return -1;
	  }
	else if ((actuallayer->id (actuallayer->source (actedge)) !=
		  actuallayer->id (*actuallayernode))
		 && (actuallayer->id (actuallayer->target (actedge)) !=
		     actuallayer->id (*actuallayernode)))
	  {
	    cerr << "The given edge does not connect to the given node!" <<
	      endl;
	    return -1;
	  }

	if (!(subnetwork->valid (subnode)))
	  {
	    cerr << "The given node is not in the given network!" << endl;
	    return -1;
	  }

	int i = 0;
	//while in the array there is valid note that is not equvivalent with the one that would be noted increase i
	while ((i < edgenumber)
	       && (actuallayer->valid (assignments[i].actedge))
	       && (assignments[i].actedge != actedge))
	  i++;
	if (assignments[i].actedge == actedge)
	  {
	    cout << "Warning: Redefinement of assigment!!!" << endl;
	  }
	if (i == edgenumber)
	  {
	    cout <<
	      "This case can't be!!! (because there should be the guven edge in the array already and the cycle had to stop)"
	      << endl;
	  }
	//if(!(actuallayer->valid(assignments[i].actedge)))   //this condition is necessary if we do not obey redefinition
	{
	  assignments[i].actedge = actedge;
	  assignments[i].subnode = subnode;
	}

	/// If to all of the edges a subnode is assigned then the subnetwork is connectable (attachable?)
	/// We do not need to check for further attributes, because to notice an assignment we need
	/// all of them to be correctly initialised before.
	if (i == edgenumber - 1)
	  connectable = 1;

	return 0;
      }

      int setSubNetwork (Gsub * sn)
      {
	subnetwork = sn;
	return 0;
      }

      int setActualLayer (Gact * al)
      {
	actuallayer = al;
	return 0;
      }

      int setActualLayerNode (typename Gact::Node * aln)
      {
	typename Gact::InEdgeIt iei;
	typename Gact::OutEdgeIt oei;

	actuallayernode = aln;

	edgenumber = 0;

	if (actuallayer)
	  {
	    for (iei = actuallayer->first (iei, (*actuallayernode));
		 ((actuallayer->valid (iei))
		  && (actuallayer->target (iei) == (*actuallayernode)));
		 actuallayer->next (iei))
	      {
		cout << actuallayer->id (actuallayer->
					 source (iei)) << " " << actuallayer->
		  id (actuallayer->target (iei)) << endl;
		edgenumber++;
	      }
	    //cout << "Number of in-edges: " << edgenumber << endl;
	    for (oei = actuallayer->first (oei, (*actuallayernode));
		 ((actuallayer->valid (oei))
		  && (actuallayer->source (oei) == (*actuallayernode)));
		 actuallayer->next (oei))
	      {
		cout << actuallayer->id (actuallayer->
					 source (oei)) << " " << actuallayer->
		  id (actuallayer->target (oei)) << endl;
		edgenumber++;
	      }
	    //cout << "Number of in+out-edges: " << edgenumber << endl;
	    assignments = new actedgesubnodestruct[edgenumber];
	    for (int i = 0; i < edgenumber; i++)
	      {
		assignments[i].actedge = INVALID;
		assignments[i].subnode = INVALID;
	      }
	  }
	else
	  {
	    cerr << "There is no actual layer defined yet!" << endl;
	    return -1;
	  }

	return 0;
      }

    SubNetwork ():edgenumber (0), connectable (false), actuallayer (NULL),
	actuallayernode (NULL), subnetwork (NULL),
	assignments (NULL)
      {
      }

    };

    typename Gact::template NodeMap < SubNetwork > subnetworks;


    /// Defalult constructor.
    /// We don't need any extra lines, because the actuallayer
    /// variable has run its constructor, when we have created this class
    /// So only the two maps has to be initialised here.
  HierarchyGraph ():subnetworks (actuallayer)
    {
    }


    ///Copy consructor.
  HierarchyGraph (const HierarchyGraph < Gact, Gsub > &HG):actuallayer (HG.actuallayer),
      subnetworks
      (actuallayer)
    {
    }


    /// The base type of the node iterators.

    /// This is the base type of each node iterators,
    /// thus each kind of node iterator will convert to this.
    /// The Node type of the HierarchyGraph is the Node type of the actual layer.
    typedef typename Gact::Node Node;


    /// This iterator goes through each node.

    /// Its usage is quite simple, for example you can count the number
    /// of nodes in graph \c G of type \c Graph like this:
    /// \code
    ///int count=0;
    ///for(Graph::NodeIt n(G);G.valid(n);G.next(n)) count++;
    /// \endcode
    /// The NodeIt type of the HierarchyGraph is the NodeIt type of the actual layer.
    typedef typename Gact::NodeIt NodeIt;


    /// The base type of the edge iterators.
    /// The Edge type of the HierarchyGraph is the Edge type of the actual layer.
    typedef typename Gact::Edge Edge;


    /// This iterator goes trough the outgoing edges of a node.

    /// This iterator goes trough the \e outgoing edges of a certain node
    /// of a graph.
    /// Its usage is quite simple, for example you can count the number
    /// of outgoing edges of a node \c n
    /// in graph \c G of type \c Graph as follows.
    /// \code
    ///int count=0;
    ///for(Graph::OutEdgeIt e(G,n);G.valid(e);G.next(e)) count++;
    /// \endcode
    /// The OutEdgeIt type of the HierarchyGraph is the OutEdgeIt type of the actual layer.
    typedef typename Gact::OutEdgeIt OutEdgeIt;


    /// This iterator goes trough the incoming edges of a node.

    /// This iterator goes trough the \e incoming edges of a certain node
    /// of a graph.
    /// Its usage is quite simple, for example you can count the number
    /// of outgoing edges of a node \c n
    /// in graph \c G of type \c Graph as follows.
    /// \code
    ///int count=0;
    ///for(Graph::InEdgeIt e(G,n);G.valid(e);G.next(e)) count++;
    /// \endcode
    /// The InEdgeIt type of the HierarchyGraph is the InEdgeIt type of the actual layer.
    typedef typename Gact::InEdgeIt InEdgeIt;


    /// This iterator goes through each edge.

    /// This iterator goes through each edge of a graph.
    /// Its usage is quite simple, for example you can count the number
    /// of edges in a graph \c G of type \c Graph as follows:
    /// \code
    ///int count=0;
    ///for(Graph::EdgeIt e(G);G.valid(e);G.next(e)) count++;
    /// \endcode
    /// The EdgeIt type of the HierarchyGraph is the EdgeIt type of the actual layer.
    typedef typename Gact::EdgeIt EdgeIt;


    /// First node of the graph.

    /// \retval i the first node.
    /// \return the first node.
    typename Gact::NodeIt & first (typename Gact::NodeIt & i) const
    {
      return actuallayer.first (i);
    }


    /// The first incoming edge.
    typename Gact::InEdgeIt & first (typename Gact::InEdgeIt & i,
				     typename Gact::Node) const
    {
      return actuallayer.first (i);
    }


    /// The first outgoing edge.
    typename Gact::OutEdgeIt & first (typename Gact::OutEdgeIt & i,
				      typename Gact::Node) const
    {
      return actuallayer.first (i);
    }


    //  SymEdgeIt &first(SymEdgeIt &, Node) const { return i;}
    /// The first edge of the Graph.
    typename Gact::EdgeIt & first (typename Gact::EdgeIt & i) const
    {
      return actuallayer.first (i);
    }


//     Node getNext(Node) const {}
//     InEdgeIt getNext(InEdgeIt) const {}
//     OutEdgeIt getNext(OutEdgeIt) const {}
//     //SymEdgeIt getNext(SymEdgeIt) const {}
//     EdgeIt getNext(EdgeIt) const {}


    /// Go to the next node.
    typename Gact::NodeIt & next (typename Gact::NodeIt & i) const
    {
      return actuallayer.next (i);
    }
    /// Go to the next incoming edge.
    typename Gact::InEdgeIt & next (typename Gact::InEdgeIt & i) const
    {
      return actuallayer.next (i);
    }
    /// Go to the next outgoing edge.
    typename Gact::OutEdgeIt & next (typename Gact::OutEdgeIt & i) const
    {
      return actuallayer.next (i);
    }
    //SymEdgeIt &next(SymEdgeIt &) const {}
    /// Go to the next edge.
    typename Gact::EdgeIt & next (typename Gact::EdgeIt & i) const
    {
      return actuallayer.next (i);
    }

    ///Gives back the target node of an edge.
    typename Gact::Node target (typename Gact::Edge edge) const
    {
      return actuallayer.target (edge);
    }
    ///Gives back the source node of an edge.
    typename Gact::Node source (typename Gact::Edge edge) const
    {
      return actuallayer.source (edge);
    }

    //   Node aNode(InEdgeIt) const {}
    //   Node aNode(OutEdgeIt) const {}
    //   Node aNode(SymEdgeIt) const {}

    //   Node bNode(InEdgeIt) const {}
    //   Node bNode(OutEdgeIt) const {}
    //   Node bNode(SymEdgeIt) const {}

    /// Checks if a node iterator is valid

    ///\todo Maybe, it would be better if iterator converted to
    ///bool directly, as Jacint prefers.
    bool valid (const typename Gact::Node & node) const
    {
      return actuallayer.valid (node);
    }
    /// Checks if an edge iterator is valid

    ///\todo Maybe, it would be better if iterator converted to
    ///bool directly, as Jacint prefers.
    bool valid (const typename Gact::Edge & edge) const
    {
      return actuallayer.valid (edge);
    }

    ///Gives back the \e id of a node.

    ///\warning Not all graph structures provide this feature.
    ///
    int id (const typename Gact::Node & node) const
    {
      return actuallayer.id (node);
    }
    ///Gives back the \e id of an edge.

    ///\warning Not all graph structures provide this feature.
    ///
    int id (const typename Gact::Edge & edge) const
    {
      return actuallayer.id (edge);
    }

    //void setInvalid(Node &) const {};
    //void setInvalid(Edge &) const {};

    ///Add a new node to the graph.

    /// \return the new node.
    ///
    typename Gact::Node addNode ()
    {
      return actuallayer.addNode ();
    }
    ///Add a new edge to the graph.

    ///Add a new edge to the graph with source node \c source
    ///and target node \c target.
    ///\return the new edge.
    typename Gact::Edge addEdge (typename Gact::Node node1,
				 typename Gact::Node node2)
    {
      return actuallayer.addEdge (node1, node2);
    }

    /// Resets the graph.

    /// This function deletes all edges and nodes of the graph.
    /// It also frees the memory allocated to store them.
    void clear ()
    {
      actuallayer.clear ();
    }

    int nodeNum () const
    {
      return actuallayer.nodeNum ();
    }
    int edgeNum () const
    {
      return actuallayer.edgeNum ();
    }

    ///Read/write/reference map of the nodes to type \c T.

    ///Read/write/reference map of the nodes to type \c T.
    /// \sa MemoryMap
    /// \todo We may need copy constructor
    /// \todo We may need conversion from other nodetype
    /// \todo We may need operator=
    /// \warning Making maps that can handle bool type (NodeMap<bool>)
    /// needs extra attention!

    template < class T > class NodeMap
    {
    public:
      typedef T Value;
      typedef Node Key;

      NodeMap (const HierarchyGraph &)
      {
      }
      NodeMap (const HierarchyGraph &, T)
      {
      }

      template < typename TT > NodeMap (const NodeMap < TT > &)
      {
      }

      /// Sets the value of a node.

      /// Sets the value associated with node \c i to the value \c t.
      ///
      void set (Node, T)
      {
      }
      // Gets the value of a node.
      //T get(Node i) const {return *(T*)0;}  //FIXME: Is it necessary?
      T & operator[](Node)
      {
	return *(T *) 0;
      }
      const T & operator[] (Node) const
      {
	return *(T *) 0;
      }

      /// Updates the map if the graph has been changed

      /// \todo Do we need this?
      ///
      void update ()
      {
      }
      void update (T a)
      {
      }				//FIXME: Is it necessary
    };

    ///Read/write/reference map of the edges to type \c T.

    ///Read/write/reference map of the edges to type \c T.
    ///It behaves exactly in the same way as \ref NodeMap.
    /// \sa NodeMap
    /// \sa MemoryMap
    /// \todo We may need copy constructor
    /// \todo We may need conversion from other edgetype
    /// \todo We may need operator=
    template < class T > class EdgeMap
    {
    public:
      typedef T Value;
      typedef Edge Key;

      EdgeMap (const HierarchyGraph &)
      {
      }
      EdgeMap (const HierarchyGraph &, T)
      {
      }

      ///\todo It can copy between different types.
      ///
      template < typename TT > EdgeMap (const EdgeMap < TT > &)
      {
      }

      void set (Edge, T)
      {
      }
      //T get(Edge) const {return *(T*)0;}
      T & operator[](Edge)
      {
	return *(T *) 0;
      }
      const T & operator[] (Edge) const
      {
	return *(T *) 0;
      }

      void update ()
      {
      }
      void update (T a)
      {
      }				//FIXME: Is it necessary
    };
  };

  /// An empty erasable graph class.

  /// This class provides all the common features of an \e erasable graph
  /// structure,
  /// however completely without implementations and real data structures
  /// behind the interface.
  /// All graph algorithms should compile with this class, but it will not
  /// run properly, of course.
  ///
  /// \todo This blabla could be replaced by a sepatate description about
  /// s.
  ///
  /// It can be used for checking the interface compatibility,
  /// or it can serve as a skeleton of a new graph structure.
  ///
  /// Also, you will find here the full documentation of a certain graph
  /// feature, the documentation of a real graph imlementation
  /// like @ref ListGraph or
  /// @ref SmartGraph will just refer to this structure.
template < typename Gact, typename Gsub > class ErasableHierarchyGraph:public HierarchyGraph < Gact,
    Gsub
    >
  {
  public:
    /// Deletes a node.
    void erase (typename Gact::Node n)
    {
      actuallayer.erase (n);
    }
    /// Deletes an edge.
    void erase (typename Gact::Edge e)
    {
      actuallayer.erase (e);
    }

    /// Defalult constructor.
    ErasableHierarchyGraph ()
    {
    }
    ///Copy consructor.
    ErasableHierarchyGraph (const HierarchyGraph < Gact, Gsub > &EPG)
    {
    }
  };


  // @}

}				//namespace lemon


#endif // LEMON_SKELETON_GRAPH_H
