// -*- c++ -*-
#ifndef LEMON_LEDA_GRAPH_WRAPPER_H
#define LEMON_LEDA_GRAPH_WRAPPER_H

#include <LEDA/graph.h>
#include <LEDA/node_array.h>
#include <LEDA/edge_array.h>
#include <LEDA/node_map.h>
#include <LEDA/edge_map.h>
//#include <LEDA/graph_alg.h>
//#include <LEDA/dimacs.h>

//#if defined(LEDA_NAMESPACE)
//using namespace leda;
//#endif

#include <lemon/invalid.h>

namespace lemon {

  /// \brief A graph wrapper structure for wrapping LEDA graphs in LEMON.
  ///
  /// This graph wrapper class wraps LEDA graphs and LEDA parametrized graphs
  /// to satisfy LEMON graph concepts.
  /// Then the generic LEMON algorithms and wrappers can be used 
  /// with LEDA graphs. 
  /// \ingroup gwrapper
  template<typename Graph>
  class LedaGraphWrapper
  {
  protected:
    Graph* l_graph;
    LedaGraphWrapper() : l_graph(0) { }
    void setGraph(Graph& _l_graph) { l_graph=&_l_graph; }
  public:

    /// Constructor for wrapping LEDA graphs.
    LedaGraphWrapper(Graph& _l_graph) : l_graph(&_l_graph) { }
    /// Copy constructor
    LedaGraphWrapper(const LedaGraphWrapper &g) : l_graph(g.l_graph) { }

    template <typename T> class NodeMap;
    template <typename T> class EdgeMap;
    template <typename T> class NodeMapWrapper;
    template <typename T> class EdgeMapWrapper;

    class Node;
    class NodeIt;
    class Edge;
    class EdgeIt;
    class OutEdgeIt;
    class InEdgeIt;

    /// Trivial node-iterator
    class Node {
      friend class LedaGraphWrapper<Graph>;
      //friend class Edge;
      friend class EdgeIt;
      friend class InEdgeIt;
      friend class OutEdgeIt;
    protected:
      template <typename T> friend class NodeMap;
      leda_node l_n;
    public: //FIXME
      Node(leda_node _l_n) : l_n(_l_n) { } 
    public:
      /// @warning The default constructor sets the iterator
      /// to an undefined value.
      Node() { }   //FIXME
      /// Initialize the iterator to be invalid
      Node(Invalid) : l_n(0) { }
      //Node(const Node &) {} 
      bool operator==(Node n) const { return l_n==n.l_n; } //FIXME
      bool operator!=(Node n) const { return l_n!=n.l_n; } //FIXME
      operator leda_node () { return l_n; }
    };
    
    /// This iterator goes through each node.
    class NodeIt : public Node {
    public:
      /// @warning The default constructor sets the iterator
      /// to an undefined value.
      NodeIt() { } //FIXME
      /// Initialize the iterator to be invalid
      NodeIt(Invalid i) : Node(i) { }
      /// Sets the iterator to the first node of \c G.
      NodeIt(const LedaGraphWrapper &G) : Node(G.l_graph->first_node()) { }
      //NodeIt(const NodeIt &) {} //FIXME
    };
    
    /// Trivial edge-iterator.
    class Edge {
      friend class LedaGraphWrapper;
    protected:
      template <typename T> friend class EdgeMap;
      leda_edge l_e;
    public: //FIXME
      Edge(leda_edge _l_e) : l_e(_l_e) { } 
    public:
      /// @warning The default constructor sets the iterator
      /// to an undefined value.
      Edge() { }   //FIXME
      /// Initialize the iterator to be invalid
      Edge(Invalid) : l_e(0) { }
      //Edge(const Edge &) {} 
      bool operator==(Edge e) const { return l_e==e.l_e; } //FIXME
      bool operator!=(Edge e) const { return l_e!=e.l_e; } //FIXME 
      operator leda_edge () { return l_e; }
    };
    
    /// This iterator goes trought the outgoing edges of a certain node.
    class OutEdgeIt : public Edge {
    public:
      /// @warning The default constructor sets the iterator
      /// to an undefined value.
      OutEdgeIt() { }
      /// Initialize the iterator to be invalid
      OutEdgeIt(Invalid i) : Edge(i) { }
      /// This constructor sets the iterator to first outgoing edge.
    
      /// This constructor set the iterator to the first outgoing edge of
      /// node
      ///@param n the node
      ///@param G the graph
      OutEdgeIt(const LedaGraphWrapper & G, Node n) : Edge(G.l_graph->first_adj_edge(n.l_n)) { }
    };

    /// This iterator goes trought the incoming edges of a certain node.
    class InEdgeIt : public Edge {
    public:
      /// @warning The default constructor sets the iterator
      /// to an undefined value.
      InEdgeIt() { }
      /// Initialize the iterator to be invalid
      InEdgeIt(Invalid i) : Edge(i) { }
      InEdgeIt(const LedaGraphWrapper & G, Node n) : Edge(G.l_graph->first_in_edge(n.l_n)) { }
    };

    //  class SymEdgeIt : public Edge {};
    
    /// This iterator goes trought the edges of the graph.
    class EdgeIt : public Edge {
    public:
      /// @warning The default constructor sets the iterator
      /// to an undefined value.
      EdgeIt() { }
      /// Initialize the iterator to be invalid
      EdgeIt(Invalid i) : Edge(i) { }
      EdgeIt(const LedaGraphWrapper & G) : Edge(G.l_graph->first_edge()) { }
    };

    /// First node of the graph.
    ///
    /// \post \c i and the return value will be the first node.
    ///
    NodeIt &first(NodeIt &i) const { i=NodeIt(*this); return i; }

    /// The first outgoing edge.
    InEdgeIt &first(InEdgeIt &i, Node n) const { 
      i=InEdgeIt(*this, n); 
      return i;
    }
    /// The first incoming edge.
    OutEdgeIt &first(OutEdgeIt &i, Node n) const { 
      i=OutEdgeIt(*this, n); 
      return i;
    }
    //  SymEdgeIt &first(SymEdgeIt &, Node) const { return i;}
    /// The first edge of the graph.
    EdgeIt &first(EdgeIt &i) const {      
      i=EdgeIt(*this); 
      return i; }

//     Node getNext(Node) const {}
//     InEdgeIt getNext(InEdgeIt) const {}
//     OutEdgeIt getNext(OutEdgeIt) const {}
//     //SymEdgeIt getNext(SymEdgeIt) const {}
//     EdgeIt getNext(EdgeIt) const {}

    /// Go to the next node.
    NodeIt &next(NodeIt &i) const { 
      i.l_n=l_graph->succ_node(i.l_n); 
      return i; 
    }
    /// Go to the next incoming edge.
    InEdgeIt &next(InEdgeIt &i) const { 
      i.l_e=l_graph->in_succ(i.l_e); 
      return i;
    }
    /// Go to the next outgoing edge.
    OutEdgeIt &next(OutEdgeIt &i) const { 
      i.l_e=l_graph->adj_succ(i.l_e); 
      return i;
    }
    //SymEdgeIt &next(SymEdgeIt &) const {}
    /// Go to the next edge.
    EdgeIt &next(EdgeIt &i) const {      
      i.l_e=l_graph->succ_edge(i.l_e); 
      return i; 
    }

//     template< typename It >
//     It first() const { 
//       It e;
//       first(e);
//       return e; 
//     }

//     template< typename It >
//     It first(Node v) const { 
//       It e;
//       first(e, v);
//       return e; 
//     }

    ///Gives back the target node of an edge.
    Node target(Edge e) const { 
      return Node(l_graph->target(e.l_e)); 
    }
    ///Gives back the source node of an edge.
    Node source(Edge e) const { 
      return Node(l_graph->source(e.l_e)); 
    }
  
    Node aNode(InEdgeIt e) const { return target(e); }
    Node aNode(OutEdgeIt e) const { return source(e); }
    //   Node aNode(SymEdgeIt) const {}

    Node bNode(InEdgeIt e) const { return source(e); }
    Node bNode(OutEdgeIt e) const { return target(e); }
    //   Node bNode(SymEdgeIt) const {}

    /// Checks if a node iterator is valid
    bool valid(Node n) const { return n.l_n; }
    /// Checks if an edge iterator is valid
    bool valid(Edge e) const { return e.l_e; }

    ///Gives back the \e id of a node.
    int id(Node n) const { return n.l_n->id(); }
    ///Gives back the \e id of an edge.
    int id(Edge e) const { return e.l_e->id(); }

    //void setInvalid(Node &) const {};
    //void setInvalid(Edge &) const {};
  
    Node addNode() const { return Node(l_graph->new_node()); }
    Edge addEdge(Node source, Node target) const { 
      return Edge(l_graph->new_edge(source.l_n, target.l_n));
    }
    
    void erase(Node n) const { l_graph->del_node(n.l_n); }
    void erase(Edge e) const { l_graph->del_edge(e.l_e); }

    void clear() const { l_graph->clear(); }

    int nodeNum() const { return l_graph->number_of_nodes(); }
    int edgeNum() const { return l_graph->number_of_edges(); }

    /// Read/write map from the nodes to type \c T.
    template<typename T> class NodeMap
    {
      leda_node_map<T> leda_stuff;
    public:
      typedef T Value;
      typedef Node Key;

      NodeMap(const LedaGraphWrapper &G) : leda_stuff(*(G.l_graph)) {}
      NodeMap(const LedaGraphWrapper &G, T t) : leda_stuff(*(G.l_graph), t) {}

      void set(Node i, T t) { leda_stuff[i.l_n]=t; }
      T get(Node i) const { return leda_stuff[i.l_n]; }  //FIXME: Is it necessary
      T &operator[](Node i) { return leda_stuff[i.l_n]; }
      const T &operator[](Node i) const { return leda_stuff[i.l_n]; }

      void update() { /*leda_stuff.init(leda_stuff.get_graph());*/ }
      //void update(T a) { leda_stuff.init(leda_stuff.get_graph()/**(G.l_graph)*/, a); }   //FIXME: Is it necessary
    };

    /// Read/write map from the edges to type \c T.
    template<typename T> class EdgeMap
    {
      leda_edge_map<T> leda_stuff;
    public:
      typedef T Value;
      typedef Edge Key;

      EdgeMap(const LedaGraphWrapper &G) : leda_stuff(*(G.l_graph)) {}
      EdgeMap(const LedaGraphWrapper &G, T t) : leda_stuff(*(G.l_graph), t) {}

      void set(Edge i, T t) { leda_stuff[i.l_e]=t; }
      T get(Edge i) const { return leda_stuff[i.l_e]; }  //FIXME: Is it necessary
      T &operator[](Edge i) { return leda_stuff[i.l_e]; }
      const T &operator[](Edge i) const { return leda_stuff[i.l_e]; }

      void update() { /*leda_stuff.init(leda_stuff.get_graph());*/ }
      //void update(T a) { leda_stuff.init(leda_stuff.get_graph()/**(G.l_graph)*/, a); }   //FIXME: Is it necessary
    };


    /// This class is to wrap existing 
    /// LEDA node-maps to LEMON ones.
    template<typename T> class NodeMapWrapper
    {
      leda_node_array<T>* leda_stuff;
    public:
      typedef T Value;
      typedef Node Key;

      NodeMapWrapper(leda_node_array<T>& _leda_stuff) : 
	leda_stuff(&_leda_stuff) { }
      //NodeMap(leda_node_map& &G, T t) : leda_stuff(*(G.l_graph), t) {}

      void set(Node i, T t) { (*leda_stuff)[i.l_n]=t; }
      //T get(Node i) const { return (*leda_stuff)[i.l_n]; }  //FIXME: Is it necessary
      T &operator[](Node i) { return (*leda_stuff)[i.l_n]; }
      const T &operator[](Node i) const { return (*leda_stuff)[i.l_n]; }

      void update() { /*leda_stuff.init(leda_stuff.get_graph());*/ }
      //void update(T a) { leda_stuff.init(leda_stuff.get_graph()/**(G.l_graph)*/, a); }   //FIXME: Is it necessary
    };

    /// This class is to wrap existing 
    /// LEDA edge-maps to LEMON ones.
    template<typename T> class EdgeMapWrapper
    {
      leda_edge_array<T>* leda_stuff;
    public:
      typedef T Value;
      typedef Edge Key;

      EdgeMapWrapper(leda_edge_array<T>& _leda_stuff) : 
	leda_stuff(_leda_stuff) { }
      //EdgeMap(const LedaGraphWrapper &G, T t) : leda_stuff(*(G.l_graph), t) {}

      void set(Edge i, T t) { (*leda_stuff)[i.l_e]=t; }
      //T get(Edge i) const { return (*leda_stuff)[i.l_e]; }  //FIXME: Is it necessary
      T &operator[](Edge i) { return (*leda_stuff)[i.l_e]; }
      const T &operator[](Edge i) const { return (*leda_stuff)[i.l_e]; }

      void update() { /*leda_stuff.init(leda_stuff.get_graph());*/ }
      //void update(T a) { leda_stuff.init(leda_stuff.get_graph()/**(G.l_graph)*/, a); }   //FIXME: Is it necessary
    };

    /// This class is used for access node-data of 
    /// LEDA parametrized graphs.
    template<typename T> 
    class NodeDataMap : public NodeMapWrapper<T>
    {
    public:
      NodeDataMap(LedaGraphWrapper<Graph>& LGW) : 
	NodeMapWrapper<T>(*(LGW._g_graph).node_data()) { }
    };

    /// This class is used for access edge-data of 
    /// LEDA parametrized graphs.
    template<typename T> 
    class EdgeDataMap : public EdgeMapWrapper<T>
    {
    public:
      EdgeDataMap(LedaGraphWrapper<Graph>& LGW) : 
	EdgeMapWrapper<T>(*(LGW._g_graph).edge_data()) { }
    };

  };


  /// \brief LEDA graph template.
  ///
  /// This graph stucture uses LEDA graphs for physical storage.
  /// \ingroup graphs
  template<typename Graph>
  class LedaGraph : public LedaGraphWrapper<Graph> {
    typedef LedaGraphWrapper<Graph> Parent;
  protected:
    Graph gr;
  public:
    LedaGraph() { 
      Parent::setGraph(gr); 
    }
  };

} //namespace lemon

#endif // LEMON_LEDA_GRAPH_WRAPPER_H
