// -*- c++ -*-
#ifndef LEMON_NET_GRAPH_H
#define LEMON_NET_GRAPH_H

///\file
///\brief Declaration of EdgePathGraph.

#include <lemon/invalid.h>
#include <lemon/maps.h>

/// The namespace of LEMON
namespace lemon {

  // @defgroup empty_graph The EdgePathGraph class
  // @{

  /// A graph class in that a simple edge can represent a path.
  
  /// This class provides all the common features of a graph structure
  /// that represents a network. You can handle with it layers. This
  /// means that an edge in one layer can be a complete path in a nother
  /// layer.

  template <typename P, class Gact, class Gsub>
  class EdgePathGraph
  {

  public:

    /// The actual layer
    Gact actuallayer;


    /// The layer on which the edges in this layer can represent paths.
    Gsub * sublayer;


    /// Map of nodes that represent the nodes of this layer in the sublayer
    typename Gact::template NodeMap<typename Gsub::Node *> projection;


    /// Map of routes that are represented by some edges in this layer
    typename Gact::template EdgeMap<P *> edgepath;


    /// Defalult constructor.
    /// We don't need any extra lines, because the actuallayer
    /// variable has run its constructor, when we have created this class
    /// So only the two maps has to be initialised here.
    EdgePathGraph() : projection(actuallayer), edgepath(actuallayer)
    {
    }


    ///Copy consructor.
    EdgePathGraph(const EdgePathGraph<P, Gact, Gsub> & EPG ) : actuallayer(EPG.actuallayer) , edgepath(actuallayer), projection(actuallayer)
    {
    }


    /// Map adder

    /// This function gets two edgemaps. One belongs to the actual layer and the
    /// other belongs to the sublayer.
    /// The function iterates through all of the edges in the edgemap belonging to the actual layer.
    /// It gets the value that belongs to the actual edge, and adds it to the value of each edge in the
    /// path represented by itself in the edgemap that belongs to the sublayer.
    
    template <typename T1, typename T2> void addMap (typename Gact::EdgeMap<T1> & actmap, typename Gsub::EdgeMap<T2> & submap)
    {
      for(EdgeIt e(actuallayer);actuallayer.valid(e);actuallayer.next(e))
      {
	typedef typename P::EdgeIt PEdgeIt;
	PEdgeIt f;

	//dep//cout << "Edge " << id(source(e)) << " - " << id(target(e)) << " in actual layer is";
	T1 incr=actmap[e];
	//cout << incr << endl;

        if(edgepath[e])
	{
	  //dep//cout << endl << "Path";
	  for(edgepath[e]->first(f); edgepath[e]->valid(f); edgepath[e]->next(f))
	  {
	    //dep//cout << " " << sublayer->id(sublayer->source(f)) << "-" << sublayer->id(sublayer->target(f));
	    submap[f]+=incr;
	  }
	  //dep////cout << EPGr2.id(EPGr2.target(f)) << endl;
	  //dep//cout << endl;
	}
	else
	{
	  //dep//cout << " itself." <<endl;
	}
      }  

    };


    /// Describe
    /// This function walks thorugh the edges of the actual layer
    /// and displays the path represented by the actual edge.
    void describe ()
    {
      for(EdgeIt e(actuallayer);actuallayer.valid(e);actuallayer.next(e))
      {
	typedef typename P::EdgeIt PEdgeIt;
	PEdgeIt f;

	cout << "Edge " << id(source(e)) << " - " << id(target(e)) << " in actual layer is";
        if(edgepath[e])
	{
	  cout << endl << "Path";
	  for(edgepath[e]->first(f); edgepath[e]->valid(f); edgepath[e]->next(f))
	  {
	    cout << " " << sublayer->id(sublayer->source(f)) << "-" << sublayer->id(sublayer->target(f));
	  }
	  //cout << EPGr2.id(EPGr2.target(f)) << endl;
	  cout << endl;
	}
	else
	{
	  cout << " itself." <<endl;
	}
      }  

    };




    /// The base type of the node iterators.

    /// This is the base type of each node iterators,
    /// thus each kind of node iterator will convert to this.
    /// The Node type of the EdgePathGraph is the Node type of the actual layer.
    typedef typename Gact::Node Node;

    
    /// This iterator goes through each node.

    /// Its usage is quite simple, for example you can count the number
    /// of nodes in graph \c G of type \c Graph like this:
    /// \code
    ///int count=0;
    ///for(Graph::NodeIt n(G);G.valid(n);G.next(n)) count++;
    /// \endcode
    /// The NodeIt type of the EdgePathGraph is the NodeIt type of the actual layer.
    typedef typename Gact::NodeIt NodeIt;
    
    
    /// The base type of the edge iterators.
    /// The Edge type of the EdgePathGraph is the Edge type of the actual layer.
    typedef typename  Gact::Edge Edge;

    
    /// This iterator goes trough the outgoing edges of a node.

    /// This iterator goes trough the \e outgoing edges of a certain node
    /// of a graph.
    /// Its usage is quite simple, for example you can count the number
    /// of outgoing edges of a node \c n
    /// in graph \c G of type \c Graph as follows.
    /// \code
    ///int count=0;
    ///for(Graph::OutEdgeIt e(G,n);G.valid(e);G.next(e)) count++;
    /// \endcode
    /// The OutEdgeIt type of the EdgePathGraph is the OutEdgeIt type of the actual layer.
    typedef typename Gact::OutEdgeIt OutEdgeIt;


    /// This iterator goes trough the incoming edges of a node.

    /// This iterator goes trough the \e incoming edges of a certain node
    /// of a graph.
    /// Its usage is quite simple, for example you can count the number
    /// of outgoing edges of a node \c n
    /// in graph \c G of type \c Graph as follows.
    /// \code
    ///int count=0;
    ///for(Graph::InEdgeIt e(G,n);G.valid(e);G.next(e)) count++;
    /// \endcode
    /// The InEdgeIt type of the EdgePathGraph is the InEdgeIt type of the actual layer.
    typedef typename Gact::InEdgeIt InEdgeIt;


    /// This iterator goes through each edge.

    /// This iterator goes through each edge of a graph.
    /// Its usage is quite simple, for example you can count the number
    /// of edges in a graph \c G of type \c Graph as follows:
    /// \code
    ///int count=0;
    ///for(Graph::EdgeIt e(G);G.valid(e);G.next(e)) count++;
    /// \endcode
    /// The EdgeIt type of the EdgePathGraph is the EdgeIt type of the actual layer.
    typedef typename Gact::EdgeIt EdgeIt;


    /// First node of the graph.

    /// \retval i the first node.
    /// \return the first node.
    typename Gact::NodeIt &first(typename Gact::NodeIt &i) const { return actuallayer.first(i);}


    /// The first incoming edge.
    typename Gact::InEdgeIt &first(typename Gact::InEdgeIt &i, typename Gact::Node) const { return actuallayer.first(i);}


    /// The first outgoing edge.
    typename Gact::OutEdgeIt &first(typename Gact::OutEdgeIt &i, typename Gact::Node) const { return actuallayer.first(i);}


    //  SymEdgeIt &first(SymEdgeIt &, Node) const { return i;}
    /// The first edge of the Graph.
    typename Gact::EdgeIt &first(typename Gact::EdgeIt &i) const { return actuallayer.first(i);}


//     Node getNext(Node) const {}
//     InEdgeIt getNext(InEdgeIt) const {}
//     OutEdgeIt getNext(OutEdgeIt) const {}
//     //SymEdgeIt getNext(SymEdgeIt) const {}
//     EdgeIt getNext(EdgeIt) const {}


    /// Go to the next node.
    typename Gact::NodeIt &next(typename Gact::NodeIt &i) const { return actuallayer.next(i);}
    /// Go to the next incoming edge.
    typename Gact::InEdgeIt &next(typename Gact::InEdgeIt &i) const { return actuallayer.next(i);}
    /// Go to the next outgoing edge.
    typename Gact::OutEdgeIt &next(typename Gact::OutEdgeIt &i) const { return actuallayer.next(i);}
    //SymEdgeIt &next(SymEdgeIt &) const {}
    /// Go to the next edge.
    typename Gact::EdgeIt &next(typename Gact::EdgeIt &i) const { return actuallayer.next(i);}

    ///Gives back the target node of an edge.
    typename Gact::Node target(typename Gact::Edge edge) const { return actuallayer.target(edge); }
    ///Gives back the source node of an edge.
    typename Gact::Node source(typename Gact::Edge edge) const { return actuallayer.source(edge); }
  
    //   Node aNode(InEdgeIt) const {}
    //   Node aNode(OutEdgeIt) const {}
    //   Node aNode(SymEdgeIt) const {}

    //   Node bNode(InEdgeIt) const {}
    //   Node bNode(OutEdgeIt) const {}
    //   Node bNode(SymEdgeIt) const {}

    /// Checks if a node iterator is valid

    ///\todo Maybe, it would be better if iterator converted to
    ///bool directly, as Jacint prefers.
    bool valid(const typename Gact::Node& node) const { return actuallayer.valid(node);}
    /// Checks if an edge iterator is valid

    ///\todo Maybe, it would be better if iterator converted to
    ///bool directly, as Jacint prefers.
    bool valid(const typename Gact::Edge& edge) const { return actuallayer.valid(edge);}

    ///Gives back the \e id of a node.

    ///\warning Not all graph structures provide this feature.
    ///
    int id(const typename Gact::Node & node) const { return actuallayer.id(node);}
    ///Gives back the \e id of an edge.

    ///\warning Not all graph structures provide this feature.
    ///
    int id(const typename Gact::Edge & edge) const { return actuallayer.id(edge);}

    //void setInvalid(Node &) const {};
    //void setInvalid(Edge &) const {};
  
    ///Add a new node to the graph.

    /// \return the new node.
    ///
    typename Gact::Node addNode() { return actuallayer.addNode();}
    ///Add a new edge to the graph.

    ///Add a new edge to the graph with source node \c source
    ///and target node \c target.
    ///\return the new edge.
    typename Gact::Edge addEdge(typename Gact::Node node1, typename Gact::Node node2) { return actuallayer.addEdge(node1, node2);}
    
    /// Resets the graph.

    /// This function deletes all edges and nodes of the graph.
    /// It also frees the memory allocated to store them.
    void clear() {actuallayer.clear();}

    int nodeNum() const { return actuallayer.nodeNum();}
    int edgeNum() const { return actuallayer.edgeNum();}

    ///Read/write/reference map of the nodes to type \c T.

    ///Read/write/reference map of the nodes to type \c T.
    /// \sa MemoryMap
    /// \todo We may need copy constructor
    /// \todo We may need conversion from other nodetype
    /// \todo We may need operator=
    /// \warning Making maps that can handle bool type (NodeMap<bool>)
    /// needs extra attention!

    template<class T> class NodeMap
    {
    public:
      typedef T Value;
      typedef Node Key;

      NodeMap(const EdgePathGraph &) {}
      NodeMap(const EdgePathGraph &, T) {}

      template<typename TT> NodeMap(const NodeMap<TT> &) {}

      /// Sets the value of a node.

      /// Sets the value associated with node \c i to the value \c t.
      ///
      void set(Node, T) {}
      // Gets the value of a node.
      //T get(Node i) const {return *(T*)0;}  //FIXME: Is it necessary?
      T &operator[](Node) {return *(T*)0;}
      const T &operator[](Node) const {return *(T*)0;}

      /// Updates the map if the graph has been changed

      /// \todo Do we need this?
      ///
      void update() {}
      void update(T a) {}   //FIXME: Is it necessary
    };

    ///Read/write/reference map of the edges to type \c T.

    ///Read/write/reference map of the edges to type \c T.
    ///It behaves exactly in the same way as \ref NodeMap.
    /// \sa NodeMap
    /// \sa MemoryMap
    /// \todo We may need copy constructor
    /// \todo We may need conversion from other edgetype
    /// \todo We may need operator=
    template<class T> class EdgeMap
    {
    public:
      typedef T Value;
      typedef Edge Key;

      EdgeMap(const EdgePathGraph &) {}
      EdgeMap(const EdgePathGraph &, T ) {}
    
      ///\todo It can copy between different types.
      ///
      template<typename TT> EdgeMap(const EdgeMap<TT> &) {}

      void set(Edge, T) {}
      //T get(Edge) const {return *(T*)0;}
      T &operator[](Edge) {return *(T*)0;}
      const T &operator[](Edge) const {return *(T*)0;}
    
      void update() {}
      void update(T a) {}   //FIXME: Is it necessary
    };
  };

  /// An empty erasable graph class.
  
  /// This class provides all the common features of an \e erasable graph
  /// structure,
  /// however completely without implementations and real data structures
  /// behind the interface.
  /// All graph algorithms should compile with this class, but it will not
  /// run properly, of course.
  ///
  /// \todo This blabla could be replaced by a sepatate description about
  /// s.
  ///
  /// It can be used for checking the interface compatibility,
  /// or it can serve as a skeleton of a new graph structure.
  /// 
  /// Also, you will find here the full documentation of a certain graph
  /// feature, the documentation of a real graph imlementation
  /// like @ref ListGraph or
  /// @ref SmartGraph will just refer to this structure.
  template <typename P, typename Gact, typename Gsub>
  class ErasableEdgePathGraph : public EdgePathGraph<P, Gact, Gsub>
  {
  public:
    /// Deletes a node.
    void erase(typename Gact::Node n) {actuallayer.erase(n);}
    /// Deletes an edge.
    void erase(typename Gact::Edge e) {actuallayer.erase(e);}

    /// Defalult constructor.
    ErasableEdgePathGraph() {}
    ///Copy consructor.
    ErasableEdgePathGraph(const EdgePathGraph<P, Gact, Gsub> &EPG) {}
  };

  
  // @}

} //namespace lemon


#endif // LEMON_SKELETON_GRAPH_H
