// -*- c++ -*-
/**
2 dimensional vector (plainvector) implementation

*/
#ifndef HUGO_XY_H
#define HUGO_XY_H

#include <iostream>

namespace hugo {

  template<typename T>
    class xy {
    T _x,_y;

    public:
      
      ///Default constructor: both coordinates become 0
      xy() { _x=_y=0; }

      ///Constructing from coordinates
      xy(T a, T b) { _x=a; _y=b; }

      ///Gives back the x coordinate
      T x(){
	return _x;
      };

      ///Gives back the y coordinate
      T y(){
	return _y;
      };

      ///Gives back the square of the norm of the vector
      T normSquare(){
	return _x*_x+_y*_y;
      };
  
      ///Increments the left hand side by u
      xy<T>& operator +=(const xy<T>& u){
	_x += u._x;
	_y += u._y;
	return *this;
      };
  
      ///Decrements the left hand side by u
      xy<T>& operator -=(const xy<T>& u){
	_x -= u._x;
	_y -= u._y;
	return *this;
      };

      ///Multiplying the left hand side with a scalar
      xy<T>& operator *=(const T &u){
	_x *= u;
	_y *= u;
	return *this;
      };

      ///Dividing the left hand side by a scalar
      xy<T>& operator /=(const T &u){
	_x /= u;
	_y /= u;
	return *this;
      };
  
      ///Returns the scalar product of two vectors
      T operator *(const xy<T>& u){
	return _x*u._x+_y*u._y;
      };
  
      ///Returns the sum of two vectors
      xy<T> operator+(const xy<T> &u) const {
	xy<T> b=*this;
	return b+=u;
      };

      ///Returns the difference of two vectors
      xy<T> operator-(const xy<T> &u) const {
	xy<T> b=*this;
	return b-=u;
      };

      ///Returns a vector multiplied by a scalar
      xy<T> operator*(const T &u) const {
	xy<T> b=*this;
	return b*=u;
      };

      ///Returns a vector divided by a scalar
      xy<T> operator/(const T &u) const {
	xy<T> b=*this;
	return b/=u;
      };

      ///Testing equality
      bool operator==(const xy<T> &u){
	return (_x==u._x) && (_y==u._y);
      };

      ///Testing inequality
      bool operator!=(xy u){
	return  (_x!=u._x) || (_y!=u._y);
      };

    };

  ///Reading a plainvector from a stream
  template<typename T>
  inline
  std::istream& operator>>(std::istream &is, xy<T> &z)
  {
    ///This is not the best solution here: I didn't know how to solve this with friend functions
    T a,b;
    is >> a >> b;
    xy<T> buf(a,b);
    z=buf;
    return is;
  }

  ///Outputting a plainvector to a stream
  template<typename T>
  inline
  std::ostream& operator<<(std::ostream &os, xy<T> z)
  {
    os << "(" << z.x() << ", " << z.y() << ")";
    return os;
  }

} //namespace hugo

#endif //HUGO_XY_H
