/* -*- C++ -*-
 * src/lemon/map_utils.h - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2004 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

///\ingroup gutils
///\file
///\brief Map utilities.

#include <map>


namespace lemon {

  /// \addtogroup gutils
  /// @{


  /// \brief General inversable map type.

  /// This type provides simple inversable map functions. 
  /// The InversableMap wraps an arbitrary ReadWriteMap 
  /// and if a key is setted to a new value then store it
  /// in the inverse map.
  template <
    typename _Graph, 
    typename _Map, 
    template <typename, typename> class _InvMap = std::Map
  >
  class InversableMap : protected _Map {

  public:

    typename _Map Map;
    typename _InvMap<Map::Value, Map::Key> InverseMap;
    
    typename _Map::Key Key;
    typename _Map::Value Value;
    typename _Map::ConstReference ConstReference;

    /// Constructor.

    /// Construct a new InversableMap for the graph.
    ///
    InversableMap(const Graph& graph) : Map(graph) {} 
    
    /// The setter function of the map.

    /// It sets the map and the inverse map 
    void set(const Key& key, const Value& val) {
      Value oldval = Map::operator[](key);
      InverseMap::iterator it = invMap.find(oldval);
      if (it != invMap.end() && it->second == key) {
	invMap.erase(it);
      }      
      invMap.insert(make_pair(val, key));
      Map::set(key, val);
    }

    ConstReference operator[](const Key&) const {
      return Map::operator[](key);
    }

    virtual void add(const Key&) {
      Map::add(key);
    }

    virtual void erase(const Key&) {
      Value val = Map::operator[](key);
      InverseMap::iterator it = invMap.find(val);
      if (it != invMap.end() && it->second == key) {
	invMap.erase(it);
      }
      Map::erase(key);
    }

    const InverseMap& inverse() const {
      return invMap;
    } 


  private:
    InverseMap invMap;    
  };

}

