/* -*- C++ -*-
 * src/lemon/demo/graph_to_eps.cc - 
 * Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2005 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include<lemon/graph_to_eps.h>
#include<lemon/maps.h>
#include<lemon/list_graph.h>
#include<lemon/graph_utils.h>


using namespace std;
using namespace lemon;

int main()
{
  ColorSet colorSet;

  ListGraph g;
  typedef ListGraph::Node Node;
  typedef ListGraph::NodeIt NodeIt;
  typedef ListGraph::Edge Edge;
  typedef xy<int> Xy;
  
  Node n1=g.addNode();
  Node n2=g.addNode();
  Node n3=g.addNode();
  Node n4=g.addNode();
  Node n5=g.addNode();

  ListGraph::NodeMap<Xy> coords(g);
  ListGraph::NodeMap<double> sizes(g);
  ListGraph::NodeMap<int> colors(g);
  ListGraph::NodeMap<int> shapes(g);
  ListGraph::EdgeMap<int> ecolors(g);
  ListGraph::EdgeMap<int> widths(g);
  
  coords[n1]=Xy(50,50);  sizes[n1]=1; colors[n1]=1; shapes[n1]=0;
  coords[n2]=Xy(50,70);  sizes[n2]=2; colors[n2]=2; shapes[n2]=2;
  coords[n3]=Xy(70,70);  sizes[n3]=1; colors[n3]=3; shapes[n3]=0;
  coords[n4]=Xy(70,50);  sizes[n4]=2; colors[n4]=4; shapes[n4]=1;
  coords[n5]=Xy(85,60);  sizes[n5]=3; colors[n5]=5; shapes[n5]=2;
  
  Edge e;

  e=g.addEdge(n1,n2); ecolors[e]=0; widths[e]=1;
  e=g.addEdge(n2,n3); ecolors[e]=0; widths[e]=1;
  e=g.addEdge(n3,n5); ecolors[e]=0; widths[e]=3;
  e=g.addEdge(n5,n4); ecolors[e]=0; widths[e]=1;
  e=g.addEdge(n4,n1); ecolors[e]=0; widths[e]=1;
  e=g.addEdge(n2,n4); ecolors[e]=1; widths[e]=2;
  e=g.addEdge(n3,n4); ecolors[e]=2; widths[e]=1;
  
  IdMap<ListGraph,Node> id(g);

  graphToEps(g,"graph_to_eps_demo_out.eps").scale(10).coords(coords).
    title("Sample .eps figure").
    copyright("(C) 2005 LEMON Project").
    nodeScale(2).nodeSizes(sizes).
    nodeShapes(shapes).
    nodeColors(composeMap(colorSet,colors)).
    edgeColors(composeMap(colorSet,ecolors)).
    edgeWidthScale(.4).edgeWidths(widths).
    nodeTexts(id).nodeTextSize(3).
    run();

  graphToEps(g,"graph_to_eps_demo_out_arr.eps").scale(10).
    title("Sample .eps figure (with arrowheads)").
    copyright("(C) 2005 LEMON Project").
    nodeColors(composeMap(colorSet,colors)).
    coords(coords).
    nodeScale(2).nodeSizes(sizes).
    nodeShapes(shapes).
    edgeColors(composeMap(colorSet,ecolors)).
    edgeWidthScale(.4).edgeWidths(widths).
    nodeTexts(id).nodeTextSize(3).
    drawArrows().arrowWidth(1).arrowLength(1).
    run();

  e=g.addEdge(n1,n4); ecolors[e]=2; widths[e]=1;
  e=g.addEdge(n4,n1); ecolors[e]=1; widths[e]=2;

  e=g.addEdge(n1,n2); ecolors[e]=1; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=2; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=3; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=4; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=5; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=6; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=7; widths[e]=1;

  graphToEps(g,"graph_to_eps_demo_out_par.eps").scale(10).
    title("Sample .eps figure (parallel edges)").
    copyright("(C) 2005 LEMON Project").
    nodeShapes(shapes).
    coords(coords).
    nodeScale(2).nodeSizes(sizes).
    nodeColors(composeMap(colorSet,colors)).
    edgeColors(composeMap(colorSet,ecolors)).
    edgeWidthScale(.4).edgeWidths(widths).
    nodeTexts(id).nodeTextSize(3).
    enableParallel().parEdgeDist(1.5).
    run();
  
  graphToEps(g,"graph_to_eps_demo_out_par_arr.eps").scale(10).
    title("Sample .eps figure (parallel edges and arrowheads)").
    copyright("(C) 2005 LEMON Project").
    nodeScale(2).nodeSizes(sizes).
    coords(coords).
    nodeShapes(shapes).
    nodeColors(composeMap(colorSet,colors)).
    edgeColors(composeMap(colorSet,ecolors)).
    edgeWidthScale(.3).edgeWidths(widths).
    nodeTexts(id).nodeTextSize(3).
    enableParallel().parEdgeDist(1).
    drawArrows().arrowWidth(1).arrowLength(1).
    run();

  graphToEps(g,"graph_to_eps_demo_out_a4.eps").scaleToA4().
    title("Sample .eps figure (fits to A4)").
    copyright("(C) 2005 LEMON Project").
    nodeScale(2).nodeSizes(sizes).
    coords(coords).
    nodeShapes(shapes).
    nodeColors(composeMap(colorSet,colors)).
    edgeColors(composeMap(colorSet,ecolors)).
    edgeWidthScale(.3).edgeWidths(widths).
    nodeTexts(id).nodeTextSize(3).
    enableParallel().parEdgeDist(1).
    drawArrows().arrowWidth(1).arrowLength(1).
    run();

  ListGraph h;
  ListGraph::NodeMap<int> hcolors(h);
  ListGraph::NodeMap<Xy> hcoords(h);
  
  int cols=int(sqrt(double(colorSet.size())));
  for(int i=0;i<int(colorSet.size());i++) {
    Node n=h.addNode();
    hcoords[n]=Xy(i%cols,i/cols);
    hcolors[n]=i;
  }
  
  graphToEps(h,"graph_to_eps_demo_out_colors.eps").scale(60).
    title("Sample .eps figure (parallel edges and arrowheads)").
    copyright("(C) 2005 LEMON Project").
    coords(hcoords).
    nodeScale(.45).
    distantColorNodeTexts().
    //    distantBWNodeTexts().
    nodeTexts(hcolors).nodeTextSize(.6).
    nodeColors(composeMap(colorSet,hcolors)).
    run();


}
