// -*- C++ -*-
/*
preflow_push_max_flow_h
by jacint. 
Runs a preflow push algorithm with the modification, 
that we do not push on nodes with level at least n. 
Moreover, if a level gets empty, we set all nodes above that
level to level n. Hence, in the end, we arrive at a maximum preflow 
with value of a max flow value. An empty level gives a minimum cut.

Member functions:

void run() : runs the algorithm

  The following functions should be used after run() was already run.

T maxflow() : returns the value of a maximum flow

void mincut(CutMap& M) : sets M to the characteristic vector of a 
     minimum cut. M should be a map of bools initialized to false.

*/

#ifndef PREFLOW_PUSH_MAX_FLOW_H
#define PREFLOW_PUSH_MAX_FLOW_H

#define A 1

#include <algorithm>
#include <vector>
#include <stack>

#include <reverse_bfs.h>


namespace hugo {

  template <typename Graph, typename T,  
    typename FlowMap=typename Graph::EdgeMap<T>, typename CapMap=typename Graph::EdgeMap<T>, 
    typename IntMap=typename Graph::NodeMap<int>, typename TMap=typename Graph::NodeMap<T> >
  class preflow_push_max_flow {
    
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::EachNodeIt EachNodeIt;
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    typedef typename Graph::InEdgeIt InEdgeIt;
    
    Graph& G;
    NodeIt s;
    NodeIt t;
    IntMap level;
    CapMap& capacity;  
    int empty_level;    //an empty level in the end of run()
    T value; 
    
  public:
      
    preflow_push_max_flow(Graph& _G, NodeIt _s, NodeIt _t, CapMap& _capacity) :
      G(_G), s(_s), t(_t), level(_G), capacity(_capacity) { }


    /*
      The run() function runs a modified version of the 
      highest label preflow-push, which only 
      finds a maximum preflow, hence giving the value of a maximum flow.
    */
    void run() {
 
      int n=G.nodeNum(); 
      int b=n-2; 
      /*
	b is a bound on the highest level of an active node. 
      */

      IntMap level(G,n);      
      TMap excess(G); 
      FlowMap flow(G,0);

      std::vector<int> numb(n);    
      /*
	The number of nodes on level i < n. It is
	initialized to n+1, because of the reverse_bfs-part.
      */

      std::vector<std::stack<NodeIt> > stack(n);    
      //Stack of the active nodes in level i.


      /*Reverse_bfs from t, to find the starting level.*/
      level.set(t,0);
      std::queue<NodeIt> bfs_queue;
      bfs_queue.push(t);

      while (!bfs_queue.empty()) {

	NodeIt v=bfs_queue.front();	
	bfs_queue.pop();
	int l=level.get(v)+1;

	for(InEdgeIt e=G.template first<InEdgeIt>(v); e.valid(); ++e) {
	  NodeIt w=G.tail(e);
	  if ( level.get(w) == n ) {
	    bfs_queue.push(w);
	    ++numb[l];
	    level.set(w, l);
	  }
	}
      }
	
      level.set(s,n);



      /* Starting flow. It is everywhere 0 at the moment. */     
      for(OutEdgeIt e=G.template first<OutEdgeIt>(s); e.valid(); ++e) 
	{
	  if ( capacity.get(e) == 0 ) continue;
	  NodeIt w=G.head(e);
	  if ( level.get(w) < n ) {	  
	    if ( excess.get(w) == 0 && w!=t ) stack[level.get(w)].push(w); 
	    flow.set(e, capacity.get(e)); 
	    excess.set(w, excess.get(w)+capacity.get(e));
	  }
	}
      
      /* 
	 End of preprocessing 
      */


      /*
	Push/relabel on the highest level active nodes.
      */
      /*While there exists an active node.*/
      while (b) { 
	if ( stack[b].empty() ) { 
	  --b;
	  continue;
	} 
	
	NodeIt w=stack[b].top();        //w is a highest label active node.
	stack[b].pop();           
	int lev=level.get(w);
	int exc=excess.get(w);
	int newlevel=2*n-2;      //In newlevel we bound the next level of w.
	
	//  if ( level.get(w) < n ) { //Nem tudom ez mukodik-e
	  for(OutEdgeIt e=G.template first<OutEdgeIt>(w); e.valid(); ++e) {
	    
	    if ( flow.get(e) == capacity.get(e) ) continue; 
	    NodeIt v=G.head(e);            
	    //e=wv	    
	    
	    if( lev > level.get(v) ) {      
	      /*Push is allowed now*/
	      
	      if ( excess.get(v)==0 && v != s && v !=t ) 
		stack[level.get(v)].push(v); 
	      /*v becomes active.*/
	      
	      int cap=capacity.get(e);
	      int flo=flow.get(e);
	      int remcap=cap-flo;
	      
	      if ( remcap >= exc ) {       
		/*A nonsaturating push.*/
		
		flow.set(e, flo+exc);
		excess.set(v, excess.get(v)+exc);
		exc=0;
		break; 
		
	      } else { 
		/*A saturating push.*/
		
		flow.set(e, cap );
		excess.set(v, excess.get(v)+remcap);
		exc-=remcap;
	      }
	    } else if ( newlevel > level.get(v) ) newlevel = level.get(v);
	    
	  } //for out edges wv 
	
	
	if ( exc > 0 ) {	
	  for( InEdgeIt e=G.template first<InEdgeIt>(w); e.valid(); ++e) {
	    
	    if( flow.get(e) == 0 ) continue; 
	    NodeIt v=G.tail(e);  
	    //e=vw
	    
	    if( lev > level.get(v) ) {  
	      /*Push is allowed now*/
	      
	      if ( excess.get(v)==0 && v != s && v !=t) 
		stack[level.get(v)].push(v); 
	      /*v becomes active.*/
	      
	      int flo=flow.get(e);
	      
	      if ( flo >= exc ) { 
		/*A nonsaturating push.*/
		
		flow.set(e, flo-exc);
		excess.set(v, excess.get(v)+exc);
		exc=0;
		break; 
	      } else {                                               
		/*A saturating push.*/
		
		excess.set(v, excess.get(v)+flo);
		exc-=flo;
		flow.set(e,0);
	      }  
	    } else if ( newlevel > level.get(v) ) newlevel = level.get(v);
	    
	  } //for in edges vw
	  
	} // if w still has excess after the out edge for cycle
	
	excess.set(w, exc);
	
	
	/*
	  Relabel
	*/
	  
	if ( exc > 0 ) {
	  //now 'lev' is the old level of w
	  level.set(w,++newlevel);
	  --numb[lev];
	    
	  if ( !numb[lev] && lev < A*n ) {  //If the level of w gets empty. 
	      
	    for (EachNodeIt v=G.template first<EachNodeIt>(); v.valid() ; ++v) {
	      if (level.get(v) > lev ) level.set(v,n);  
	    }
	    for (int i=lev+1 ; i!=n ; ++i) numb[i]=0; 
	    if ( newlevel < n ) newlevel=n; 
	  } else if ( newlevel < n ) {
	    ++numb[newlevel]; 
	    stack[newlevel].push(w);
	    b=newlevel;
	  }
	}



      } //while(b)

      value=excess.get(t);
      /*Max flow value.*/
      

      /* 
	 We count empty_level. The nodes above this level is a mincut.
      */
      while(true) {
	if(numb[empty_level]) ++empty_level;
	else break;
      } 
      
    } // void run()



    /*
      Returns the maximum value of a flow.
     */

    T maxflow() {
      return value;
    }



    /*
      Returns a minimum cut.
    */    
    template<typename CutMap>
    void mincut(CutMap& M) {

      for (EachNodeIt v=G.template first<EachNodeIt>(); v.valid(); ++v) {
	if ( level.get(v) > empty_level ) M.set(v, true);
      }
    }


  };
}//namespace hugo
#endif 





