/* -*- C++ -*-
 * src/lemon/lp_base.h - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2005 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#ifndef LEMON_LP_BASE_H
#define LEMON_LP_BASE_H

#include<vector>
#include<limits>

#include<lemon/utility.h>
#include<lemon/error.h>
#include<lemon/invalid.h>

#include"lin_expr.h"
///\file
///\brief The interface of the LP solver interface.
namespace lemon {
  
  ///Internal data structure to convert floating id's to fix one's
    
  ///\todo This might by implemented to be usable in other places.
  class _FixId 
  {
    std::vector<int> index;
    std::vector<int> cross;
    int first_free;
  public:
    _FixId() : first_free(-1) {};
    ///Convert a floating id to a fix one

    ///\param n is a floating id
    ///\return the corresponding fix id
    int fixId(int n) {return cross[n];}
    ///Convert a fix id to a floating one

    ///\param n is a fix id
    ///\return the corresponding floating id
    int floatingId(int n) { return index[n];}
    ///Add a new floating id.

    ///\param n is a floating id
    ///\return the fix id of the new value
    ///\todo Multiple additions should also be handled.
    int insert(int n)
    {
      if(n>=int(cross.size())) {
	cross.resize(n+1);
	if(first_free==-1) {
	  cross[n]=index.size();
	  index.push_back(n);
	}
	else {
	  cross[n]=first_free;
	  int next=index[first_free];
	  index[first_free]=n;
	  first_free=next;
	}
	return cross[n];
      }
      else throw LogicError(); //floatingId-s must form a continuous range;
    }
    ///Remove a fix id.

    ///\param n is a fix id
    ///
    void erase(int n) 
    {
      int fl=index[n];
      index[n]=first_free;
      first_free=n;
      for(int i=fl+1;i<int(cross.size());++i) {
	cross[i-1]=cross[i];
	index[cross[i]]--;
      }
      cross.pop_back();
    }
    ///An upper bound on the largest fix id.

    ///\todo Do we need this?
    ///
    std::size_t maxFixId() { return cross.size()-1; }
  
  };
    
  ///Common base class for LP solvers
  class LpSolverBase {
    
  public:

    ///\e
    enum SolutionType {
      ///\e
      INFEASIBLE = 0,
      ///\e
      UNBOUNDED = 1,
      ///\e
      OPTIMAL = 2,
      ///\e
      FEASIBLE = 3,
    };
      
    ///The floating point type used by the solver
    typedef double Value;
    ///The infinity constant
    static const Value INF;
    
    ///Refer to a column of the LP.

    ///This type is used to refer to a column of the LP.
    ///
    ///Its value remains valid and correct even after the addition or erase of
    ///new column (unless the referred column itself was also deleted,
    ///of course).
    ///
    ///\todo Document what can one do with a Col (INVALID, comparing,
    ///it is similar to Node/Edge)
    class Col {
    protected:
      int id;
      friend class LpSolverBase;
    public:
      typedef Value ExprValue;
      typedef True LpSolverCol;
      Col() {}
      Col(const Invalid&) : id(-1) {}
      bool operator<(Col c) const  {return id<c.id;}
      bool operator==(Col c) const  {return id==c.id;}
      bool operator!=(Col c) const  {return id==c.id;}
    };

    ///Refer to a row of the LP.

    ///This type is used to refer to a row of the LP.
    ///
    ///Its value remains valid and correct even after the addition or erase of
    ///new rows (unless the referred row itself was also deleted, of course).
    ///
    ///\todo Document what can one do with a Row (INVALID, comparing,
    ///it is similar to Node/Edge)
    class Row {
    protected:
      int id;
      friend class LpSolverBase;
    public:
      typedef Value ExprValue;
      typedef True LpSolverRow;
      Row() {}
      Row(const Invalid&) : id(-1) {}
      typedef True LpSolverRow;
      bool operator<(Row c) const  {return id<c.id;}
      bool operator==(Row c) const  {return id==c.id;}
      bool operator!=(Row c) const  {return id==c.id;} 
   };
    
    ///Linear expression
    typedef SparseLinExpr<Col> Expr;

  protected:
    _FixId rows;
    _FixId cols;

    /// \e
    virtual int _addCol() = 0;
    /// \e
    virtual int _addRow() = 0;
    /// \e

    /// \warning Arrays are indexed from 1 (datum at index 0 is ignored)
    ///
    virtual void _setRowCoeffs(int i, 
			       int length,
                               int  const * indices, 
                               Value  const * values ) = 0;
    /// \e

    /// \warning Arrays are indexed from 1 (datum at index 0 is ignored)
    ///
    virtual void _setColCoeffs(int i, 
			       int length,
                               int  const * indices, 
                               Value  const * values ) = 0;
    
    /// \e

    /// The lower bound of a variable (column) have to be given by an 
    /// extended number of type Value, i.e. a finite number of type 
    /// Value or -\ref INF.
    virtual void _setColLowerBound(int i, Value value) = 0;
    /// \e

    /// The upper bound of a variable (column) have to be given by an 
    /// extended number of type Value, i.e. a finite number of type 
    /// Value or \ref INF.
    virtual void _setColUpperBound(int i, Value value) = 0;
    /// \e

    /// The lower bound of a linear expression (row) have to be given by an 
    /// extended number of type Value, i.e. a finite number of type 
    /// Value or -\ref INF.
    virtual void _setRowLowerBound(int i, Value value) = 0;
    /// \e

    /// The upper bound of a linear expression (row) have to be given by an 
    /// extended number of type Value, i.e. a finite number of type 
    /// Value or \ref INF.
    virtual void _setRowUpperBound(int i, Value value) = 0;

    /// \e
    virtual void _setObjCoeff(int i, Value obj_coef) = 0;

    ///\e
    
    ///\bug Wrong interface
    ///
    virtual SolutionType _solve() = 0;

    ///\e

    ///\bug Wrong interface
    ///
    virtual Value _getSolution(int i) = 0;
    ///\e

    ///\bug unimplemented!!!!
    void clearObj() {}
  public:


    ///\e
    virtual ~LpSolverBase() {}

    ///\name Building up and modification of the LP

    ///@{

    ///Add a new empty column (i.e a new variable) to the LP
    Col addCol() { Col c; c.id=cols.insert(_addCol()); return c;}

    ///\brief Fill the elements of a container with newly created columns
    ///(i.e a new variables)
    ///
    ///This magic function takes container as its argument
    ///and fills its elements
    ///with new columns (i.e. variables)
    ///\param t can be either any standard STL iterable container with
    ///\ref Col \c values_type or \c mapped_type
    ///like <tt>std::vector<LpSolverBase::Col></tt>,
    /// <tt>std::list<LpSolverBase::Col></tt> or
    /// <tt>std::map<AnyType,LpSolverBase::Col></tt> or
    ///it can be an iterable lemon map like 
    /// <tt>ListGraph::NodeMap<LpSolverBase::Col></tt>.
    ///\return The number of the created column.
    ///\bug Iterable nodemap hasn't been implemented yet.
#ifdef DOXYGEN
    template<class T>
    int addColSet(T &t) { return 0;} 
#else
    template<class T>
    typename enable_if<typename T::value_type::LpSolverCol,int>::type
    addColSet(T &t,dummy<0> = 0) {
      int s=0;
      for(typename T::iterator i=t.begin();i!=t.end();++i) {*i=addCol();s++;}
      return s;
    }
    template<class T>
    typename enable_if<typename T::value_type::second_type::LpSolverCol,
		       int>::type
    addColSet(T &t,dummy<1> = 1) { 
      int s=0;
      for(typename T::iterator i=t.begin();i!=t.end();++i) {
	i->second=addCol();
	s++;
      }
      return s;
    }
#endif

    ///Add a new empty row (i.e a new constaint) to the LP

    ///This function adds a new empty row (i.e a new constaint) to the LP.
    ///\return The created row
    Row addRow() { Row r; r.id=rows.insert(_addRow()); return r;}

    ///Set a row (i.e a constaint) of the LP

    ///\param r is the row to be modified
    ///\param l is lower bound (-\ref INF means no bound)
    ///\param e is a linear expression (see \ref Expr)
    ///\param u is the upper bound (\ref INF means no bound)
    ///\bug This is a temportary function. The interface will change to
    ///a better one.
    void setRow(Row r, Value l,const Expr &e, Value u) {
      std::vector<int> indices;
      std::vector<Value> values;
      indices.push_back(0);
      values.push_back(0);
      for(Expr::const_iterator i=e.begin(); i!=e.end(); ++i)
	if((*i).second!=0) { ///\bug EPSILON would be necessary here!!!
	  indices.push_back(cols.floatingId((*i).first.id));
	  values.push_back((*i).second);
	}
      _setRowCoeffs(rows.floatingId(r.id),indices.size()-1,
		    &indices[0],&values[0]);
      _setRowLowerBound(rows.floatingId(r.id),l-e.constComp());
      _setRowUpperBound(rows.floatingId(r.id),u-e.constComp());
    }

    ///Add a new row (i.e a new constaint) to the LP

    ///\param l is the lower bound (-\ref INF means no bound)
    ///\param e is a linear expression (see \ref Expr)
    ///\param u is the upper bound (\ref INF means no bound)
    ///\return The created row.
    ///\bug This is a temportary function. The interface will change to
    ///a better one.
    Row addRow(Value l,const Expr &e, Value u) {
      Row r=addRow();
      setRow(r,l,e,u);
      return r;
    }

    /// Set the lower bound of a column (i.e a variable)

    /// The upper bound of a variable (column) have to be given by an 
    /// extended number of type Value, i.e. a finite number of type 
    /// Value or -\ref INF.
    virtual void setColLowerBound(Col c, Value value) {
      _setColLowerBound(cols.floatingId(c.id),value);
    }
    /// Set the upper bound of a column (i.e a variable)

    /// The upper bound of a variable (column) have to be given by an 
    /// extended number of type Value, i.e. a finite number of type 
    /// Value or \ref INF.
    virtual void setColUpperBound(Col c, Value value) {
      _setColUpperBound(cols.floatingId(c.id),value);
    };
    /// Set the lower bound of a row (i.e a constraint)

    /// The lower bound of a linear expression (row) have to be given by an 
    /// extended number of type Value, i.e. a finite number of type 
    /// Value or -\ref INF.
    virtual void setRowLowerBound(Row r, Value value) {
      _setRowLowerBound(rows.floatingId(r.id),value);
    };
    /// Set the upper bound of a row (i.e a constraint)

    /// The upper bound of a linear expression (row) have to be given by an 
    /// extended number of type Value, i.e. a finite number of type 
    /// Value or \ref INF.
    virtual void setRowUpperBound(Row r, Value value) {
      _setRowUpperBound(rows.floatingId(r.id),value);
    };
    ///Set an element of the objective function
    void setObjCoeff(Col c, Value v) {_setObjCoeff(cols.floatingId(c.id),v); };
    ///Set the objective function
    
    ///\param e is a linear expression of type \ref Expr.
    ///\todo What to do with the constant component?
    void setObj(Expr e) {
      clearObj();
      for (Expr::iterator i=e.begin(); i!=e.end(); ++i)
	setObjCoeff((*i).first,(*i).second);
    }

    ///@}


    ///\name Solving the LP

    ///@{

    ///\e
    SolutionType solve() { return _solve(); }
    
    ///@}
    
    ///\name Obtaining the solution LP

    ///@{

    ///\e
    Value solution(Col c) { return _getSolution(cols.floatingId(c.id)); }

    ///@}
    
  };  

} //namespace lemon

#endif //LEMON_LP_BASE_H
