// -*- c++ -*-
#ifndef DEFAULT_MAP_FACTORY_H
#define DEFAULT_MAP_FACTORY_H


#include <hugo/array_map_factory.h>
#include <hugo/vector_map_factory.h>

///\ingroup graphmapfactory
///\file
///\brief Graph maps that construates and destruates
///their elements dynamically.

namespace hugo {

/// \addtogroup graphmapfactory
/// @{

#define DEFAULT_MAP_BODY(Factory, Val) \
  { \
    typedef typename Factory<MapRegistry>::template Map<Val> MapImpl; \
  \
  public: \
  \
    typedef typename MapRegistry::Graph Graph; \
    typedef typename MapRegistry::KeyType KeyType; \
    typedef typename MapRegistry::KeyIt KeyIt; \
    typedef Val Value; \
  \
    typedef typename MapRegistry::MapBase MapBase; \
  \
    DefaultMap() : MapImpl() {} \
  \
    DefaultMap(const Graph& g, MapRegistry& r) : MapImpl(g, r) {} \
  \
    DefaultMap(const Graph& g, MapRegistry& r, const Value& v) \
      : MapImpl(g, r, v) {} \
  \
    DefaultMap(const DefaultMap& copy) \
      : MapImpl(static_cast<const MapImpl&>(copy)) {} \
  \
    template <typename CMap> DefaultMap(const CMap& copy) : MapImpl(copy) {} \
  \
    DefaultMap& operator=(const DefaultMap& copy) { \
      MapImpl::operator=(static_cast<const MapImpl&>(copy)); \
      return *this; \
    } \
  \
    template <typename CMap> DefaultMap& operator=(const CMap& copy) { \
      MapImpl::operator=(copy); \
      return *this; \
    } \
  \
  };


  template <typename MapRegistry, typename Type>
  class DefaultMap : public ArrayMapFactory<MapRegistry>::template Map<Type> 
  DEFAULT_MAP_BODY(ArrayMapFactory, Type);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, bool> 
    : public VectorMapFactory<MapRegistry>::template Map<bool> 
  DEFAULT_MAP_BODY(VectorMapFactory, bool);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, char> 
    : public VectorMapFactory<MapRegistry>::template Map<char> 
  DEFAULT_MAP_BODY(VectorMapFactory, char);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, int> 
    : public VectorMapFactory<MapRegistry>::template Map<int> 
  DEFAULT_MAP_BODY(VectorMapFactory, int);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, short> 
    : public VectorMapFactory<MapRegistry>::template Map<short> 
  DEFAULT_MAP_BODY(VectorMapFactory, short);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, long> 
    : public VectorMapFactory<MapRegistry>::template Map<long> 
  DEFAULT_MAP_BODY(VectorMapFactory, long);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, float> 
    : public VectorMapFactory<MapRegistry>::template Map<float> 
  DEFAULT_MAP_BODY(VectorMapFactory, float);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, double> 
    : public VectorMapFactory<MapRegistry>::template Map<double> 
  DEFAULT_MAP_BODY(VectorMapFactory, double);

  template <typename MapRegistry>
  class DefaultMap<MapRegistry, long double> 
    : public VectorMapFactory<MapRegistry>::template Map<long double> 
  DEFAULT_MAP_BODY(VectorMapFactory, long double);

  template <typename MapRegistry, typename Type>
  class DefaultMap<MapRegistry, Type*>
    : public VectorMapFactory<MapRegistry>::template Map<Type*> 
  DEFAULT_MAP_BODY(VectorMapFactory, Type*);


  /** The DefaultMapFactory template class is a factory class
   *  to create maps for the edge and nodes. This map factory
   *  uses the VectorMapFactory if the ValueType is a primitive
   *  type and the ArrayMapFactory for the other cases.
   *
   *  The template parameter is the MapRegistry that the maps
   *  will belong to.
   */

  template <typename MapRegistry>
  class DefaultMapFactory {
		
  public:
    /// The graph type of the maps. 
    typedef typename MapRegistry::Graph Graph;
    /// The key type of the maps.
    typedef typename MapRegistry::KeyType KeyType;
    /// The iterator to iterate on the keys.
    typedef typename MapRegistry::KeyIt KeyIt;

    /// The MapBase of the Map which imlements the core regisitry function.
    typedef typename MapRegistry::MapBase MapBase;
		

    /** The template Map type.
     */
    template <typename V> 
    class Map : public DefaultMap<MapRegistry, V> {

      typedef DefaultMap<MapRegistry, V> MapImpl;

    public:
      
      typedef V Value;

      /** Default constructor for the map.
       */
      Map() : MapImpl() {}

      /** Graph and Registry initialized map constructor.
       */
      Map(const Graph& g, MapRegistry& r) : MapImpl(g, r) {}

      /** Constructor to use default value to initialize the map. 
       */
      Map(const Graph& g, MapRegistry& r, const Value& v) : MapImpl(g, r, v) {}

      /** Constructor to copy a map of the same map type.
       */
      Map(const Map& copy) : MapImpl(static_cast<const MapImpl&>(copy)) {}

      /** Constructor to copy a map of an other map type.
       */
      template <typename CMap> Map(const CMap& copy) : MapImpl(copy) {}

      /** Assign operator to copy a map of the same map type.
       */
      Map& operator=(const Map& copy) {
	MapImpl::operator=(static_cast<const MapImpl&>(copy));
	return *this;
      }

      /** Assign operator to copy a map an other map type.
       */
      template <typename CMap> Map& operator=(const CMap& copy) {
	MapImpl::operator=(copy);
	return *this;
      }

    };

  };
}

#endif
