// -*- c++ -*-

// Use a DIMACS max flow file as stdin.
// sub_graph_wrapper_demo < dimacs_max_flow_file
// This program computes a maximum number of edge-disjoint shortest paths
// between s and t.

#include <iostream>
#include <fstream>

#include <hugo/smart_graph.h>
#include <hugo/dijkstra.h>
#include <hugo/maps.h>
#include <hugo/graph_wrapper.h>
#include <hugo/dimacs.h>
#include <hugo/preflow.h>
#include <tight_edge_filter_map.h>

using namespace hugo;

using std::cout;
using std::endl;

int main()
{    
  typedef SmartGraph Graph;

  typedef Graph::Edge Edge;
  typedef Graph::Node Node;
  typedef Graph::EdgeIt EdgeIt;
  typedef Graph::NodeIt NodeIt;
  typedef Graph::EdgeMap<int> LengthMap;

  Graph g;
  Node s, t;
  LengthMap length(g);

  readDimacs(std::cin, g, length, s, t);

  cout << "edges with lengths (of form tail--length->head): " << endl;
  for(EdgeIt e(g); e!=INVALID; ++e) 
    cout << " " << g.id(g.tail(e)) << "--" 
	 << length[e] << "->" << g.id(g.head(e)) << endl;

  cout << "s: " << g.id(s) << " t: " << g.id(t) << endl;

  typedef Dijkstra<Graph, LengthMap> Dijkstra;
  Dijkstra dijkstra(g, length);
  dijkstra.run(s);

  // This map returns true exactly for those edges which are 
  // tight w.r.t the length funcion and the potential 
  // given by the dijkstra algorithm.
  typedef TightEdgeFilterMap<Graph, const Dijkstra::DistMap, LengthMap> 
    TightEdgeFilter;
  TightEdgeFilter tight_edge_filter(g, dijkstra.distMap(), length);

  ConstMap<Node, bool> const_true_map(true);
  // This graph contains exaclty the tight edges.
  typedef SubGraphWrapper<Graph, ConstMap<Node, bool>, TightEdgeFilter> SubGW;
  SubGW gw(g, const_true_map, tight_edge_filter);

  ConstMap<Edge, int> const_1_map(1);
  Graph::EdgeMap<int> flow(g, 0);
  // Max flow between s and t in the graph of tight edges.
  Preflow<SubGW, int, ConstMap<Edge, int>, Graph::EdgeMap<int> > 
    preflow(gw, s, t, const_1_map, flow);
  preflow.run();

  cout << "edges of the maximum number of edge-disjoint shortest s-t paths: " 
       << endl;
  for(EdgeIt e(g); e!=INVALID; ++e) 
    if (flow[e])
      cout << " " << g.id(g.tail(e)) << "--" 
	   << length[e] << "->" << g.id(g.head(e)) << endl;

  return 0;
}
