// -*- c++ -*- //

/**
 *
 * Class for representing paths in graphs.
 *
 */

#ifndef HUGO_PATH_H
#define HUGO_PATH_H

#include <deque>
#include <algorithm>

#include <invalid.h>

namespace hugo {

  /* Ennek az allocatorosdinak sokkal jobban utana kene nezni a hasznalata
     elott. Eleg bonyinak nez ki, ahogyan azokat az STL-ben hasznaljak. */

  template<typename Graph>
  class Path {

  public:
    typedef typename Graph::Edge GraphEdge;
    typedef typename Graph::Node GraphNode;
    class NodeIt;
    class EdgeIt;

  protected:
    Graph& G;
    // FIXME: ehelyett eleg lenne tarolni ket boolt: a ket szelso el
    // iranyitasat:
    GraphNode _first, _last;
    typedef std::deque<GraphEdge> Container;
    Container edges;

  public:

    Path(Graph &_G) : G(_G), _first(INVALID), _last(INVALID) {}

    /// Subpath defined by two nodes.
    /// Nodes may be in reversed order, then
    /// we contstruct the reversed path.
    Path(const Path &P, const NodeIt &a, const NodeIt &b);
    /// Subpath defined by two edges. Contains edges in [a,b)
    /// It is an error if the two edges are not in order!
    Path(const Path &P, const EdgeIt &a, const EdgeIt &b);
    
    size_t length() const { return edges.size(); }
    GraphNode from() const { return _first; }
    GraphNode to() const { return _last; }

    NodeIt& first(NodeIt &n) const { return nth(n, 0); }
    EdgeIt& first(EdgeIt &e) const { return nth(e, 0); }
    template<typename It>
    It first() const { 
      It e;
      first(e);
      return e; 
    }

    NodeIt& nth(NodeIt &, size_t) const;
    EdgeIt& nth(EdgeIt &, size_t) const;
    template<typename It>
    It nth(size_t n) const { 
      It e;
      nth(e, n);
      return e; 
    }

    bool valid(const NodeIt &n) const { return n.idx <= length(); }
    bool valid(const EdgeIt &e) const { return e.it < edges.end(); }

    bool isForward(const EdgeIt &e) const { return e.forw; }

    /// index of a node on the path. Returns length+2 for the invalid NodeIt
    int index(const NodeIt &n) const { return n.idx; }
    /// index of an edge on the path. Returns length+1 for the invalid EdgeIt
    int index(const EdgeIt &e) const { return e.it - edges.begin(); }

    EdgeIt& next(EdgeIt &e) const;
    NodeIt& next(NodeIt &n) const;
    template <typename It>
    It getNext(It it) const {
      It tmp(it); return next(tmp);
    }

    // A path is constructed using the following four functions.
    // They return false if the requested operation is inconsistent
    // with the path constructed so far.
    // If your path has only one edge you MUST set either "from" or "to"!
    // So you probably SHOULD call it in any case to be safe (and check the
    // returned value to check if your path is consistent with your idea).
    bool pushFront(const GraphEdge &e);
    bool pushBack(const GraphEdge &e);
    bool setFrom(const GraphNode &n);
    bool setTo(const GraphNode &n);

    // WARNING: these two functions return the head/tail of an edge with
    // respect to the direction of the path!
    // So G.head(P.graphEdge(e)) == P.graphNode(P.head(e)) holds only if 
    // P.forward(e) is true (or the edge is a loop)!
    NodeIt head(const EdgeIt& e) const;
    NodeIt tail(const EdgeIt& e) const;

    // FIXME: ezeknek valami jobb nev kellene!!!
    GraphEdge graphEdge(const EdgeIt& e) const;
    GraphNode graphNode(const NodeIt& n) const;


    /*** Iterator classes ***/
    class EdgeIt {
      friend class Path;

      typename Container::const_iterator it;
      bool forw;
    public:
      // FIXME: jarna neki ilyen is...
      // EdgeIt(Invalid);

      bool forward() const { return forw; }

      bool operator==(const EdgeIt& e) const { return it==e.it; }
      bool operator!=(const EdgeIt& e) const { return it!=e.it; }
      bool operator<(const EdgeIt& e) const { return it<e.it; }
    };

    class NodeIt {
      friend class Path;
      friend class EdgeIt;

      size_t idx;
      bool tail;  // Is this node the tail of the edge with same idx?

    public:
      // FIXME: jarna neki ilyen is...
      // NodeIt(Invalid);

      bool operator==(const NodeIt& n) const { return idx==n.idx; }
      bool operator!=(const NodeIt& n) const { return idx!=n.idx; }
      bool operator<(const NodeIt& n) const { return idx<n.idx; }
    };

  private:
    bool edgeIncident(const GraphEdge &e, const GraphNode &a,
		      GraphNode &b);
    bool connectTwoEdges(const GraphEdge &e, const GraphEdge &f);
  };

  template<typename Gr>
  typename Path<Gr>::EdgeIt&
  Path<Gr>::next(Path::EdgeIt &e) const {
    if( e.it == edges.end() ) 
      return e;

    GraphNode common_node = ( e.forw ? G.head(*e.it) : G.tail(*e.it) );
    ++e.it;

    // Invalid edgeit is always forward :)
    if( e.it == edges.end() ) {
      e.forw = true;
      return e;
    }

    e.forw = ( G.tail(*e.it) == common_node );
    return e;
  }

  template<typename Gr>
  typename Path<Gr>::NodeIt& Path<Gr>::next(NodeIt &n) const {
    if( n.idx >= length() ) {
      // FIXME: invalid
      n.idx = length()+1;
      return n;
    }

    
    GraphNode next_node = ( n.tail ? G.head(edges[n.idx]) :
			      G.tail(edges[n.idx]) );
    ++n.idx;
    if( n.idx < length() ) {
      n.tail = ( next_node == G.tail(edges[n.idx]) );
    }
    else {
      n.tail = true;
    }

    return n;
  }

  template<typename Gr>
  bool Path<Gr>::edgeIncident(const GraphEdge &e, const GraphNode &a,
			  GraphNode &b) {
    if( G.tail(e) == a ) {
      b=G.head(e);
      return true;
    }
    if( G.head(e) == a ) {
      b=G.tail(e);
      return true;
    }
    return false;
  }

  template<typename Gr>
  bool Path<Gr>::connectTwoEdges(const GraphEdge &e,
			     const GraphEdge &f) {
    if( edgeIncident(f, G.tail(e), _last) ) {
      _first = G.head(e);
      return true;
    }
    if( edgeIncident(f, G.head(e), _last) ) {
      _first = G.tail(e);
      return true;
    }
    return false;
  }

  template<typename Gr>
  bool Path<Gr>::pushFront(const GraphEdge &e) {
    if( G.valid(_first) ) {
	if( edgeIncident(e, _first, _first) ) {
	  edges.push_front(e);
	  return true;
	}
	else
	  return false;
    }
    else if( length() < 1 || connectTwoEdges(e, edges[0]) ) {
      edges.push_front(e);
      return true;
    }
    else
      return false;
  }

  template<typename Gr>
  bool Path<Gr>::pushBack(const GraphEdge &e) {
    if( G.valid(_last) ) {
	if( edgeIncident(e, _last, _last) ) {
	  edges.push_back(e);
	  return true;
	}
	else
	  return false;
    }
    else if( length() < 1 || connectTwoEdges(edges[0], e) ) {
      edges.push_back(e);
      return true;
    }
    else
      return false;
  }


  template<typename Gr>
  bool Path<Gr>::setFrom(const GraphNode &n) {
    if( G.valid(_first) ) {
      return _first == n;
    }
    else {
      if( length() > 0) {
	if( edgeIncident(edges[0], n, _last) ) {
	  _first = n;
	  return true;
	}
	else return false;
      }
      else {
	_first = _last = n;
	return true;
      }
    }
  }

  template<typename Gr>
  bool Path<Gr>::setTo(const GraphNode &n) {
    if( G.valid(_last) ) {
      return _last == n;
    }
    else {
      if( length() > 0) {
	if( edgeIncident(edges[0], n, _first) ) {
	  _last = n;
	  return true;
	}
	else return false;
      }
      else {
	_first = _last = n;
	return true;
      }
    }
  }


  template<typename Gr>
  typename Path<Gr>::NodeIt
  Path<Gr>::tail(const EdgeIt& e) const {
    NodeIt n;

    if( e.it == edges.end() ) {
      // FIXME: invalid-> invalid
      n.idx = length() + 1;
      n.tail = true;
      return n;
    }

    n.idx = e.it-edges.begin();
    n.tail = e.forw;
    return n;
  }

  template<typename Gr>
  typename Path<Gr>::NodeIt
  Path<Gr>::head(const EdgeIt& e) const {
    if( e.it == edges.end()-1 ) {
      return _last;
    }

    EdgeIt next_edge = e;
    next(next_edge);
    return tail(next_edge);
  }
      
  template<typename Gr>
  typename Path<Gr>::GraphEdge
  Path<Gr>::graphEdge(const EdgeIt& e) const {
    if( e.it != edges.end() ) {
      return *e.it;
    }
    else {
      return INVALID;
    }
  }
  
  template<typename Gr>
  typename Path<Gr>::GraphNode
  Path<Gr>::graphNode(const NodeIt& n) const {
    if( n.idx < length() ) {
      return n.tail ? G.tail(edges[n.idx]) : G.head(edges[n.idx]);
    }
    else if( n.idx == length() ) {
      return _last;
    }
    else {
      return INVALID;
    }
  }

  template<typename Gr>
  typename Path<Gr>::EdgeIt& Path<Gr>::nth(EdgeIt &e, size_t k) const {
    if( k<0 || k>=length() ) {
      // FIXME: invalid EdgeIt
      e.it = edges.end();
      e.forw = true;
      return e;
    }

    e.it = edges.begin()+k;
    if(k==0) {
      e.forw = ( G.tail(*e.it) == _first );
    }
    else {
      e.forw = ( G.tail(*e.it) == G.tail(edges[k-1]) ||
		 G.tail(*e.it) == G.head(edges[k-1]) );
    }
    return e;
  }
    
  template<typename Gr>
  typename Path<Gr>::NodeIt& Path<Gr>::nth(NodeIt &n, size_t k) const {
    if( k<0 || k>length() ) {
      // FIXME: invalid NodeIt
      n.idx = length()+1;
      n.tail = true;
      return n;
    }
    if( k==length() ) {
      n.idx = length();
      n.tail = true;
      return n;
    }
    n = tail(nth<EdgeIt>(k));
    return n;
  }

  // Reszut konstruktorok:


  template<typename Gr>
  Path<Gr>::Path(const Path &P, const EdgeIt &a, const EdgeIt &b) :
    G(P.G), edges(a.it, b.it)    // WARNING: if b.it < a.it this will blow up! 
  {
    if( G.valid(P._first) && a.it < P.edges.end() ) {
      _first = ( a.forw ? G.tail(*a.it) : G.head(*a.it) );
      if( b.it < P.edges.end() ) {
	_last = ( b.forw ? G.tail(*b.it) : G.head(*b.it) );
      }
      else {
	_last = P._last;
      }
    }
  }

  template<typename Gr>
  Path<Gr>::Path(const Path &P, const NodeIt &a, const NodeIt &b) :
    G(P.G)
  {
    if( !P.valid(a) || !P.valid(b) )
      return;

    int ai = a.idx, bi = b.idx;
    if( bi<ai )
      swap(ai,bi);
    
    edges.resize(bi-ai);
    copy(P.edges.begin()+ai, P.edges.begin()+bi, edges.begin());

    _first = P.graphNode(a);
    _last = P.graphNode(b);
  }


} // namespace hugo

#endif // HUGO_PATH_H
