/* -*- C++ -*-
 *
 * This file is a part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2003-2006
 * Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Research Group on Combinatorial Optimization, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all cop	ies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include "i18n.h"
#include <limits>
#include <cmath>
#include <iostream>
#include <fstream>
#include <string>
#include <algorithm>
#include <gtkmm.h>
#include "file_import_dialog.h"
#include <mapstorage.h>
#include <gui_writer.h>
#include <gui_reader.h>
#include <lemon/graph_to_eps.h>
#include <lemon/lgf_reader.h>
#include <lemon/lgf_writer.h>

const int i_d=20;
const double a_d=0.05;
const double p_d=40000;

MapStorage::MapStorage() :
  gui_sect_save_dest(LGF_FILE),
  node_coords_save_dest(SpecMapSaveOpts::GUI_SECT),
  arrow_coords_save_dest(SpecMapSaveOpts::GUI_SECT),
  modified(false),
  file_name(""),
  max_node_label(0),
  max_arc_label(0),
  node_coords_one_map_name("coord"),
  node_coords_two_maps_1_name("coord_x"),
  node_coords_two_maps_2_name("coord_y"),
  arrow_coords_one_map_name("arrow"),
  arrow_coords_two_maps_1_name("arrow_x"),
  arrow_coords_two_maps_2_name("arrow_y"),
  iterations(i_d),
  attraction(a_d),
  propulsation(p_d),
  node_coords_x(digraph),
  node_coords_y(digraph),
  arrow_coords_x(digraph),
  arrow_coords_y(digraph),
  node_label(digraph),
  arc_label(digraph),
  background_set(false)
{
  node_coords.setXMap(node_coords_x);
  node_coords.setYMap(node_coords_y);
  arrow_coords.setXMap(arrow_coords_x);
  arrow_coords.setYMap(arrow_coords_y);

  active_nodemaps.resize(NODE_PROPERTY_NUM);
  for(int i=0;i<NODE_PROPERTY_NUM;i++)
    {
      active_nodemaps[i]="";
    }

  active_arcmaps.resize(EDGE_PROPERTY_NUM);
  for(int i=0;i<EDGE_PROPERTY_NUM;i++)
    {
      active_arcmaps[i]="";
    }
}

MapStorage::~MapStorage()
{
  clear();
}

void MapStorage::createNodeMap(const std::string& name, MapValue::Type type,
    MapValue def_val)
{
  NodeMapStore::const_iterator it = nodemaps.find(name);
  if (it != nodemaps.end())
    throw Error("Node map " + name + " already exists.");

  switch (type)
  {
    case MapValue::NUMERIC:
      nodemaps[name] = new NumericNodeMapData(digraph, def_val);
      break;
    case MapValue::STRING:
      nodemaps[name] = new StringNodeMapData(digraph, def_val);
      break;
  }

  nodemaps[name]->default_value = def_val;

  signal_node_map.emit(name, type);
}

void MapStorage::createArcMap(const std::string& name, MapValue::Type type,
    MapValue def_val)
{
  ArcMapStore::const_iterator it = arcmaps.find(name);
  if (it != arcmaps.end())
    throw Error("Arc map " + name + " already exists.");

  switch (type)
  {
    case MapValue::NUMERIC:
      arcmaps[name] = new NumericArcMapData(digraph, def_val);
      break;
    case MapValue::STRING:
      arcmaps[name] = new StringArcMapData(digraph, def_val);
      break;
  }

  arcmaps[name]->default_value = def_val;

  signal_arc_map.emit(name, type);
}

void MapStorage::changeActiveMap(bool itisarc, int prop, std::string mapname)
{
  if(itisarc)
    {
      active_arcmaps[prop]=mapname;
    }
  else
    {
      active_nodemaps[prop]=mapname;
    }
  signal_prop.emit(itisarc, prop);
}

void MapStorage::broadcastActiveMaps()
{
  for(int i=0;i<NODE_PROPERTY_NUM;i++)
    {
      signal_map_win.emit(false, i, active_nodemaps[i]);
    }
  
  for(int i=0;i<EDGE_PROPERTY_NUM;i++)
    {
      signal_map_win.emit(true, i, active_arcmaps[i]);
    }
}

std::string MapStorage::getActiveArcMap(int prop)
{
  return active_arcmaps[prop];
}

std::string MapStorage::getActiveNodeMap(int prop)
{
  return active_nodemaps[prop];
}

std::vector<std::string> MapStorage::getArcMapList(MapType type)
{
  if (type == ALL)
  {
    std::vector<std::string> ret;
    for (ArcMapStore::const_iterator it = arcmaps.begin();
        it != arcmaps.end(); ++it)
    {
      ret.push_back(it->first);
    }
    return ret;
  }
  else
  {
    std::vector<std::string> ret;
    for (ArcMapStore::const_iterator it = arcmaps.begin();
        it != arcmaps.end(); ++it)
    {
      ArcMapData* data = getArcMapData(it->first);
      MapValue::Type t = data->type();
      if ((t == MapValue::NUMERIC && (type & NUM)) ||
          (t == MapValue::STRING && (type & STR)))
      {
        ret.push_back(it->first);
      }
    }
    return ret;
  }
}

std::vector<std::string> MapStorage::getNodeMapList(MapType type)
{
  if (type == ALL)
  {
    std::vector<std::string> ret;
    for (NodeMapStore::const_iterator it = nodemaps.begin();
        it != nodemaps.end(); ++it)
    {
      ret.push_back(it->first);
    }
    return ret;
  }
  else
  {
    std::vector<std::string> ret;
    for (NodeMapStore::const_iterator it = nodemaps.begin();
        it != nodemaps.end(); ++it)
    {
      NodeMapData* data = getNodeMapData(it->first);
      MapValue::Type t = data->type();
      if ((t == MapValue::NUMERIC && (type & NUM)) ||
          (t == MapValue::STRING && (type & STR)))
      {
        ret.push_back(it->first);
      }
    }
    return ret;
  }
}

sigc::signal<void, bool, int> MapStorage::signal_prop_ch()
{
  return signal_prop;
}

int MapStorage::readFromFile(const std::string &filename)
{
  using std::vector;
  using std::map;
  using std::string;

  //reading content of file
  LgfContents content(filename);
  try
    {
      content.run();
    }
  catch (Exception& error)
    {
      Gtk::MessageDialog mdialog(error.what());
      mdialog.run();
      clear();
      return 1;
    }

  // check whether the .conf file exists
  bool gui_data_in_conf = g_file_test((filename + ".conf").c_str(),
      (GFileTest)(G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR));

  // check whether the .lgf file contains a gui section
  bool gui_data_in_lgf = false;
  {
    for(int i=0;i<content.extraSectionNum();i++)
      {
	if(content.extraSection(i)=="gui")
	  {
	    gui_data_in_lgf = true;
	  }
      }
  }

  bool gui_data_found = gui_data_in_lgf || gui_data_in_conf;

  // ask for user input if both exist
  bool use_gui_data_in_lgf = false;
  if (gui_data_in_conf && gui_data_in_lgf)
  {
    Gtk::MessageDialog mdialog(_("<b>Found both ") + filename +
        _(".conf and a gui section in ") + filename + _(".</b>"), true,
        Gtk::MESSAGE_QUESTION, Gtk::BUTTONS_NONE);
	  mdialog.add_button(_("Use the ._conf file"), 1);
	  mdialog.add_button(_("Use the _gui section"), 2);
    switch (mdialog.run())
    {
      case 1:
        use_gui_data_in_lgf = false;
        break;
      case 2:
        use_gui_data_in_lgf = true;
        break;
      case Gtk::RESPONSE_NONE:
        return 1;
    }
  }
  else
  {
    use_gui_data_in_lgf = gui_data_in_lgf;
  }

  if (gui_data_found)
  {
    GUISectData gui_data;
    if (use_gui_data_in_lgf)
    {
      // read the gui section from the .lgf file
      try
      {
	sectionReader(filename).sectionStream("gui", GuiReader(this, gui_data)).run();
        gui_sect_save_dest = LGF_FILE;
      }
      catch (Exception& error)
      {
        clear();
        return 1;
      }
    }
    else
    {
      // read the gui section from the .conf file
      try
      {
	sectionReader(filename + ".conf").sectionStream("gui", GuiReader(this, gui_data)).run();
        gui_sect_save_dest = CONF_FILE;
      }
      catch (Exception& error)
      {
        clear();
        return 1;
      }
    }

    // read the digraph and maps form the .lgf file
    try
    {
      std::string node_coord_xmap_name, node_coord_ymap_name;
      std::string arrow_coord_xmap_name, arrow_coord_ymap_name;

      if (gui_data.node_coords_save_dest ==
          MapStorage::SpecMapSaveOpts::NESET_SECT)
      {
        switch (gui_data.node_coords_save_map_num)
        {
          case SpecMapSaveOpts::ONE_MAP:
            node_coord_xmap_name = gui_data.node_coords_one_map_name + ":x";
            node_coord_ymap_name = gui_data.node_coords_one_map_name + ":y";
            node_coords_one_map_name = gui_data.node_coords_one_map_name;
            break;
          case SpecMapSaveOpts::TWO_MAPS:
            node_coord_xmap_name = gui_data.node_coords_two_maps_1_name;
            node_coord_ymap_name = gui_data.node_coords_two_maps_2_name;
            node_coords_two_maps_1_name = gui_data.node_coords_two_maps_1_name;
            node_coords_two_maps_2_name = gui_data.node_coords_two_maps_2_name;
            break;
        }
        node_coords_save_dest = gui_data.node_coords_save_dest;
        node_coords_save_map_num = gui_data.node_coords_save_map_num;
      }

      if (gui_data.arrow_coords_save_dest ==
          MapStorage::SpecMapSaveOpts::NESET_SECT)
      {
        switch (gui_data.arrow_coords_save_map_num)
        {
          case SpecMapSaveOpts::ONE_MAP:
            arrow_coord_xmap_name = gui_data.arrow_coords_one_map_name + ":x";
            arrow_coord_ymap_name = gui_data.arrow_coords_one_map_name + ":y";
            arrow_coords_one_map_name = gui_data.arrow_coords_one_map_name;
            break;
          case SpecMapSaveOpts::TWO_MAPS:
            arrow_coord_xmap_name = gui_data.arrow_coords_two_maps_1_name;
            arrow_coord_ymap_name = gui_data.arrow_coords_two_maps_2_name;
            arrow_coords_two_maps_1_name =
              gui_data.arrow_coords_two_maps_1_name;
            arrow_coords_two_maps_2_name =
              gui_data.arrow_coords_two_maps_2_name;
            break;
        }
        arrow_coords_save_dest = gui_data.arrow_coords_save_dest;
        arrow_coords_save_map_num = gui_data.arrow_coords_save_map_num;
      }
      readLGF(filename, true,
          gui_data.main_node_map_names, gui_data.main_arc_map_names,
          gui_data.node_map_types, gui_data.arc_map_types,
          node_coord_xmap_name, node_coord_ymap_name,
          arrow_coord_xmap_name, arrow_coord_ymap_name);
    }
    catch (Exception& error)
    {
      clear();
      return 1;
    }

    // add the maps from the gui section
    for (vector<string>::const_iterator
        it = gui_data.gui_node_map_names.begin();
        it != gui_data.gui_node_map_names.end(); ++it)
    {
      string map_name = *it;
      switch (gui_data.node_map_types[map_name])
      {
        case MapValue::NUMERIC:
          {
            createNodeMap(map_name, MapValue::NUMERIC, double());
            NumericNodeMap& dmap = getNumericNodeMap(map_name);
            map<int, double>& smap = *gui_data.numeric_node_maps[map_name];
            for (NodeIt n(digraph); n != INVALID; ++n)
            {
              dmap[n] = smap[node_label[n]];
            }
            break;
          }
        case MapValue::STRING:
          {
            createNodeMap(map_name, MapValue::STRING, string());
            StringNodeMap& dmap = getStringNodeMap(map_name);
            map<int, string>& smap = *gui_data.string_node_maps[map_name];
            for (NodeIt n(digraph); n != INVALID; ++n)
            {
              dmap[n] = smap[node_label[n]];
            }
            break;
          }
      }
      getNodeMapData(map_name)->save_dest = GUI_SECT;
    }
    for (vector<string>::const_iterator
        it = gui_data.gui_arc_map_names.begin();
        it != gui_data.gui_arc_map_names.end(); ++it)
    {
      string map_name = *it;
      switch (gui_data.arc_map_types[map_name])
      {
        case MapValue::NUMERIC:
          {
            createArcMap(map_name, MapValue::NUMERIC, double());
            NumericArcMap& dmap = getNumericArcMap(map_name);
            map<int, double>& smap = *gui_data.numeric_arc_maps[map_name];
            for (ArcIt e(digraph); e != INVALID; ++e)
            {
              dmap[e] = smap[arc_label[e]];
            }
            break;
          }
        case MapValue::STRING:
          {
            createArcMap(map_name, MapValue::STRING, string());
            StringArcMap& dmap = getStringArcMap(map_name);
            map<int, string>& smap = *gui_data.string_arc_maps[map_name];
            for (ArcIt e(digraph); e != INVALID; ++e)
            {
              dmap[e] = smap[arc_label[e]];
            }
            break;
          }
      }
      getArcMapData(map_name)->save_dest = GUI_SECT;
    }

    // restore the node coordinate maps
    if (gui_data.node_coords_save_dest ==
        MapStorage::SpecMapSaveOpts::GUI_SECT)
    {
      for (NodeIt n(digraph); n != INVALID; ++n)
      {
        node_coords.set(n, gui_data.node_coord_map[node_label[n]]);
      }
      node_coords_save_dest = gui_data.node_coords_save_dest;
    }
    // restore the arrow coordinate maps
    if (gui_data.arrow_coords_save_dest ==
        MapStorage::SpecMapSaveOpts::GUI_SECT)
    {
      for (ArcIt e(digraph); e != INVALID; ++e)
      {
        arrow_coords.set(e, gui_data.arrow_coord_map[arc_label[e]]);
      }
      arrow_coords_save_dest = gui_data.arrow_coords_save_dest;
    }
  }
  else
  {
    // there is no gui section neither in the .lgf file nor in the .conf file
    {
      if (content.nodeSectionNum() < 1)
      {
        Gtk::MessageDialog mdialog("No nodeset found in file.");
        mdialog.run();
        clear();
        return 1;
      }

      if (content.arcSectionNum() < 1)
      {
        Gtk::MessageDialog mdialog("No arcset found in file.");
        mdialog.run();
        clear();
        return 1;
      }

      std::vector<std::string> nodeMapNames = content.nodeMapNames(0);
      std::vector<std::string> arcMapNames = content.arcMapNames(0);
      
      bool read_arc_label = true;
      if (std::find(arcMapNames.begin(), arcMapNames.end(), "label") ==
          arcMapNames.end())
      {
        read_arc_label = false;
      }

      nodeMapNames.erase(
          std::remove(nodeMapNames.begin(), nodeMapNames.end(), "label"),
          nodeMapNames.end());

      arcMapNames.erase(
          std::remove(arcMapNames.begin(), arcMapNames.end(), "label"),
          arcMapNames.end());

      FileImportDialog::ImportData data(nodeMapNames, arcMapNames);
      FileImportDialog fidialog(&data);
      int response = fidialog.run();
      if (response == Gtk::RESPONSE_OK)
      {
        try
        {
          std::string node_coord_xmap_name, node_coord_ymap_name;
          std::string arrow_coord_xmap_name, arrow_coord_ymap_name;
          bool gen_node_coords = false;
          bool gen_arrow_coords = false;

          switch (data.node_coord_load_from)
          {
            case FileImportDialog::ImportData::ONE_MAP:
              node_coord_xmap_name = data.node_coord_one_map_name + ":x";
              node_coord_ymap_name = data.node_coord_one_map_name + ":y";
              node_coords_one_map_name = data.node_coord_one_map_name;

              node_coords_save_dest = SpecMapSaveOpts::NESET_SECT;
              node_coords_save_map_num = SpecMapSaveOpts::ONE_MAP;
              break;
            case FileImportDialog::ImportData::TWO_MAPS:
              node_coord_xmap_name = data.node_coord_two_maps_1_name;
              node_coord_ymap_name = data.node_coord_two_maps_2_name;
              node_coords_two_maps_1_name = data.node_coord_two_maps_1_name;
              node_coords_two_maps_2_name = data.node_coord_two_maps_2_name;

              node_coords_save_dest = SpecMapSaveOpts::NESET_SECT;
              node_coords_save_map_num = SpecMapSaveOpts::TWO_MAPS;
              break;
            case FileImportDialog::ImportData::DONT_READ:
              node_coord_xmap_name = "";
              node_coord_ymap_name = "";

              node_coords_save_dest = SpecMapSaveOpts::GUI_SECT;
              gen_node_coords = true;
              break;
          }

          switch (data.arrow_coord_load_from)
          {
            case FileImportDialog::ImportData::ONE_MAP:
              arrow_coord_xmap_name = data.arrow_coord_one_map_name + ":x";
              arrow_coord_ymap_name = data.arrow_coord_one_map_name + ":y";
              arrow_coords_one_map_name = data.arrow_coord_one_map_name;

              arrow_coords_save_dest = SpecMapSaveOpts::NESET_SECT;
              arrow_coords_save_map_num = SpecMapSaveOpts::ONE_MAP;
              break;
            case FileImportDialog::ImportData::TWO_MAPS:
              arrow_coord_xmap_name = data.arrow_coord_two_maps_1_name;
              arrow_coord_ymap_name = data.arrow_coord_two_maps_2_name;
              arrow_coords_two_maps_1_name = data.arrow_coord_two_maps_1_name;
              arrow_coords_two_maps_2_name = data.arrow_coord_two_maps_2_name;

              arrow_coords_save_dest = SpecMapSaveOpts::NESET_SECT;
              arrow_coords_save_map_num = SpecMapSaveOpts::TWO_MAPS;
              break;
            case FileImportDialog::ImportData::DONT_READ:
              arrow_coord_xmap_name = "";
              arrow_coord_ymap_name = "";

              arrow_coords_save_dest = SpecMapSaveOpts::GUI_SECT;
              gen_arrow_coords = true;
              break;
          }

          // read arc and node maps
          std::vector<std::string> node_map_names;
          std::vector<std::string> arc_map_names;
          std::map<std::string, MapValue::Type> node_map_types;
          std::map<std::string, MapValue::Type> arc_map_types;
          for (std::vector<std::string>::const_iterator it =
              data.numeric_node_map_names.begin();
              it != data.numeric_node_map_names.end(); ++it)
          {
            node_map_names.push_back(*it);
            node_map_types[*it] = MapValue::NUMERIC;
          }
          for (std::vector<std::string>::const_iterator it =
              data.string_node_map_names.begin();
              it != data.string_node_map_names.end(); ++it)
          {
            node_map_names.push_back(*it);
            node_map_types[*it] = MapValue::STRING;
          }
          for (std::vector<std::string>::const_iterator it =
              data.numeric_arc_map_names.begin();
              it != data.numeric_arc_map_names.end(); ++it)
          {
            arc_map_names.push_back(*it);
            arc_map_types[*it] = MapValue::NUMERIC;
          }
          for (std::vector<std::string>::const_iterator it =
              data.string_arc_map_names.begin();
              it != data.string_arc_map_names.end(); ++it)
          {
            arc_map_names.push_back(*it);
            arc_map_types[*it] = MapValue::STRING;
          }

          readLGF(filename, read_arc_label,
              node_map_names, arc_map_names,
              node_map_types, arc_map_types,
              node_coord_xmap_name, node_coord_ymap_name,
              arrow_coord_xmap_name, arrow_coord_ymap_name);

          // generate arc labels
          if (!read_arc_label)
          {
            int l = 0;
            for (ArcIt e(digraph); e != INVALID; ++e)
            {
              arc_label[e] = l++;
            }
          }

          if (gen_node_coords)
          {
            // generate node coordinates
            int node_num = 0;
            for (NodeIt n(digraph); n != INVALID; ++n) { node_num++; }
            const double pi = 3.142;
            double step = 2 * pi / (double) node_num;
            int i = 0;
            for (NodeIt n(digraph); n != INVALID; ++n)
            {
              setNodeCoords(n,
                  XY(250.0 * std::cos(i * step),
                    250.0 * std::sin(i * step)));
              i++;
            }
          }
          if (gen_arrow_coords)
          {
            // generate arrow coordinates
            for (ArcIt e(digraph); e != INVALID; ++e)
            {
              if (digraph.source(e) == digraph.target(e))
              {
                setArrowCoords(e,
                    getNodeCoords(digraph.source(e)) + XY(0.0, 80.0));
              }
              else
              {
                setArrowCoords(e,
                    (getNodeCoords(digraph.source(e)) +
                     getNodeCoords(digraph.target(e))) / 2.0);
              }
            }
          }
        }
        catch (Exception& error)
        {
          clear();
          return 1;
        }
      }
      else
      {
        clear();
        return 1;
      }
    }
  }

  // set max_node_label
  {
    max_node_label = std::numeric_limits<int>::min();
    for (NodeIt n(digraph); n != INVALID; ++n)
    {
      if (node_label[n] > max_node_label)
      {
        max_node_label = node_label[n];
      }
    }
  }
  // set max_arc_label
  {
    max_arc_label = std::numeric_limits<int>::min();
    for (ArcIt e(digraph); e != INVALID; ++e)
    {
      if (arc_label[e] > max_arc_label)
      {
        max_arc_label = arc_label[e];
      }
    }
  }

  return 0;
}

void MapStorage::writeToFile(const std::string &filename)
{
  // relabel nodes and arcs
  int i = 0;
  for (NodeIt n(digraph); n != INVALID; ++n)
  {
    node_label[n] = i++;
  }
  max_node_label = i-1;
  i = 0;
  for (ArcIt e(digraph); e != INVALID; ++e)
  {
    arc_label[e] = i++;
  }
  max_arc_label = i-1;

  // write .lgf file
  {
    DigraphWriter<Digraph> gwriter(digraph, filename);

    gwriter.nodeMap("label", node_label);
    gwriter.arcMap("label", arc_label);

    // write node maps
    for (NodeMapStore::const_iterator it = nodemaps.begin();
        it != nodemaps.end(); ++it)
    {
      if (it->second->save_dest == NESET_SECT)
      {
        switch (it->second->type())
        {
          case MapValue::NUMERIC:
            gwriter.nodeMap(it->first, getNumericNodeMap(it->first));
            break;
          case MapValue::STRING:
            gwriter.nodeMap(it->first, getStringNodeMap(it->first));
            break;
        }
      }
    }

    // write arc maps
    for (ArcMapStore::const_iterator it = arcmaps.begin();
        it != arcmaps.end(); ++it)
    {
      if (it->second->save_dest == NESET_SECT)
      {
        switch (it->second->type())
        {
          case MapValue::NUMERIC:
            gwriter.arcMap(it->first, getNumericArcMap(it->first));
            break;
          case MapValue::STRING:
            gwriter.arcMap(it->first, getStringArcMap(it->first));
            break;
        }
      }
    }

    // write node coordinates
    switch (getNodeCoordsSaveDest())
    {
      case MapStorage::SpecMapSaveOpts::GUI_SECT:
        break;
      case MapStorage::SpecMapSaveOpts::NESET_SECT:
        switch (getNodeCoordsSaveMapNum())
        {
          case MapStorage::SpecMapSaveOpts::ONE_MAP:
            gwriter.nodeMap(node_coords_one_map_name + ":x",
                node_coords_x);
            gwriter.nodeMap(node_coords_one_map_name + ":y",
                node_coords_y);
            break;
          case MapStorage::SpecMapSaveOpts::TWO_MAPS:
            gwriter.nodeMap(node_coords_two_maps_1_name,
                node_coords_x);
            gwriter.nodeMap(node_coords_two_maps_2_name,
                node_coords_y);
            break;
        }
        break;
    }

    // write arrow coordinates
    switch (getArrowCoordsSaveDest())
    {
      case MapStorage::SpecMapSaveOpts::GUI_SECT:
        break;
      case MapStorage::SpecMapSaveOpts::NESET_SECT:
        switch (getArrowCoordsSaveMapNum())
        {
          case MapStorage::SpecMapSaveOpts::ONE_MAP:
            gwriter.arcMap(arrow_coords_one_map_name + ":x",
                arrow_coords_x);
            gwriter.arcMap(arrow_coords_one_map_name + ":y",
                arrow_coords_y);
            break;
          case MapStorage::SpecMapSaveOpts::TWO_MAPS:
            gwriter.arcMap(arrow_coords_two_maps_1_name,
                arrow_coords_x);
            gwriter.arcMap(arrow_coords_two_maps_2_name,
                arrow_coords_y);
            break;
        }
        break;
    }

    if (gui_sect_save_dest == LGF_FILE)
    {
      GuiWriter gui_writer(this);
      gui_writer.write(gwriter.ostream());
      gwriter.run();
    }
    else
    {
      gwriter.run();
    }
  }

  // write .conf file
  if (gui_sect_save_dest == CONF_FILE)
  {
    DigraphWriter<Digraph> lwriter(digraph, filename + ".conf");
    GuiWriter gui_writer(this);
    gui_writer.write(lwriter.ostream());
    lwriter.run();
  }
}

void MapStorage::clear()
{
  for (NodeMapStore::iterator it = nodemaps.begin(); it != nodemaps.end(); ++it)
  {
    delete it->second;
    nodemaps.erase(it);
  }
  for (ArcMapStore::iterator it = arcmaps.begin(); it != arcmaps.end(); ++it)
  {
    delete it->second;
    arcmaps.erase(it);
  }
  digraph.clear();
  file_name = "";
  modified = false;
  max_node_label = 0;
  max_arc_label = 0;
  background_set = false;

  gui_sect_save_dest = LGF_FILE;
  node_coords_save_dest = SpecMapSaveOpts::GUI_SECT;
  arrow_coords_save_dest = SpecMapSaveOpts::GUI_SECT;
  node_coords_one_map_name = "coord";
  node_coords_two_maps_1_name = "coord_x";
  node_coords_two_maps_2_name = "coord_y";
  arrow_coords_one_map_name = "arrow";
  arrow_coords_two_maps_1_name = "arrow_x";
  arrow_coords_two_maps_2_name = "arrow_y";

  for(int i=0;i<NODE_PROPERTY_NUM;i++)
    {
      changeActiveMap(false, i, "");
      signal_map_win.emit(false, i, "");
    }
  
  for(int i=0;i<EDGE_PROPERTY_NUM;i++)
    {
      changeActiveMap(true, i, "");
      signal_map_win.emit(true, i, "");
    }

  attraction=a_d;
  propulsation=p_d;
  iterations=i_d;

  signal_design_win.emit(attraction, propulsation, iterations);
}

void MapStorage::mapChanged(bool itisarc, std::string mapname)
{
  if(itisarc)
  {
    for(int i=0;i<EDGE_PROPERTY_NUM;i++)
    {
      if(active_arcmaps[i]==mapname)
      {
        signal_prop.emit(itisarc, i);
      }
    }
  }
  else
  {
    for(int i=0;i<NODE_PROPERTY_NUM;i++)
    {
      if(active_nodemaps[i]==mapname)
      {
        signal_prop.emit(itisarc, i);
      }
    }
  }
}

void MapStorage::get_design_data(double & attraction_p, double & propulsation_p, int & iterations_p)
{
  attraction_p=attraction;
  propulsation_p=propulsation;
  iterations_p=iterations;
}

void MapStorage::set_attraction(double attraction_p)
{
  attraction=attraction_p;
}

void MapStorage::set_propulsation(double propulsation_p)
{
  propulsation=propulsation_p;
}

void MapStorage::set_iteration(int iterations_p)
{
  iterations=iterations_p;
}

void MapStorage::redesign_data_changed()
{
  signal_design_win.emit(attraction, propulsation, iterations);
}

XY MapStorage::getNodeCoords(Node n) const
{
  return node_coords[n];
}

void MapStorage::setNodeCoords(Node n, XY c)
{
  node_coords.set(n, c);
}

XY MapStorage::getArrowCoords(Arc e) const
{
  return arrow_coords[e];
}

void MapStorage::setArrowCoords(Arc e, XY c)
{
  arrow_coords.set(e, c);
}

MapValue MapStorage::get(const std::string& name, Node node) const
{
  NodeMapData* data = getNodeMapData(name);
  return data->get(node);
}

void MapStorage::set(const std::string& name, Node node, MapValue val)
{
  NodeMapData* data = getNodeMapData(name);
  data->set(node, val);
}

MapValue MapStorage::get(const std::string& name, Arc arc) const
{
  ArcMapData* data = getArcMapData(name);
  return data->get(arc);
}

void MapStorage::set(const std::string& name, Arc arc, MapValue val)
{
  ArcMapData* data = getArcMapData(name);
  data->set(arc, val);
}

const std::string& MapStorage::getFileName() const
{
  return file_name;
}

void MapStorage::setFileName(const std::string& fn)
{
  file_name = fn;
}

bool MapStorage::getModified() const
{
  return modified;
}

void MapStorage::setModified(bool m)
{
  modified = m;
}

Node MapStorage::addNode(XY coords)
{
  Node node = digraph.addNode();

  setNodeCoords(node, coords);

  max_node_label++;

  node_label[node] = max_node_label;

  std::vector<std::string> node_maps = getNodeMapList();
  for (std::vector<std::string>::const_iterator it = node_maps.begin();
      it != node_maps.end(); ++it)
  {
    NodeMapData* data = getNodeMapData(*it);
    set(*it, node, data->default_value);
  }

  return node;
}

Arc MapStorage::addArc(Node from, Node to)
{
  Arc arc = digraph.addArc(from, to);

  if (from == to)
  {
    setArrowCoords(arc, getNodeCoords(from) + XY(0.0, 80.0));
  }
  else
  {
    setArrowCoords(arc, (getNodeCoords(from) + getNodeCoords(to)) / 2.0);
  }

  max_arc_label++;

  arc_label[arc] = max_arc_label;

  std::vector<std::string> arc_maps = getArcMapList();
  for (std::vector<std::string>::const_iterator it = arc_maps.begin();
      it != arc_maps.end(); ++it)
  {
    ArcMapData* data = getArcMapData(*it);
    set(*it, arc, data->default_value);
  }
  return arc;
}

MapStorage::NumericNodeMap& MapStorage::getNumericNodeMap(const std::string& name)
{
  NodeMapData* data = getNodeMapData(name);
  if (data->type() != MapValue::NUMERIC)
    throw Error("Numeric node map " + name + " does not exists.");
  return static_cast<NumericNodeMapData*>(data)->map;
}

MapStorage::StringNodeMap& MapStorage::getStringNodeMap(const std::string& name)
{
  NodeMapData* data = getNodeMapData(name);
  if (data->type() != MapValue::STRING)
    throw Error("String node map " + name + " does not exists.");
  return static_cast<StringNodeMapData*>(data)->map;
}

MapStorage::NumericArcMap& MapStorage::getNumericArcMap(const std::string& name)
{
  ArcMapData* data = getArcMapData(name);
  if (data->type() != MapValue::NUMERIC)
    throw Error("Numeric arc map " + name + " does not exists.");
  return static_cast<NumericArcMapData*>(data)->map;
}

MapStorage::StringArcMap& MapStorage::getStringArcMap(const std::string& name)
{
  ArcMapData* data = getArcMapData(name);
  if (data->type() != MapValue::STRING)
    throw Error("String arc map " + name + " does not exists.");
  return static_cast<StringArcMapData*>(data)->map;
}

MapValueArcMap MapStorage::getArcMap(const std::string& name)
{
  return MapValueArcMap(name, this);
}

MapValueNodeMap MapStorage::getNodeMap(const std::string& name)
{
  return MapValueNodeMap(name, this);
}

int MapStorage::getLabel(Node n) const
{
  return node_label[n];
}

int MapStorage::getLabel(Arc e) const
{
  return arc_label[e];
}

MapStorage::GuiSectSaveDest MapStorage::getGUIDataSaveLocation()
{
  return gui_sect_save_dest;
}

void MapStorage::setGUIDataSaveLocation(MapStorage::GuiSectSaveDest dest)
{
  gui_sect_save_dest = dest;
}

MapStorage::MapSaveDest MapStorage::getNodeMapSaveDest(std::string name) const
{
  NodeMapData *data = getNodeMapData(name);
  return data->save_dest;
}

MapStorage::MapSaveDest MapStorage::getArcMapSaveDest(std::string name) const
{
  ArcMapData *data = getArcMapData(name);
  return data->save_dest;
}

void MapStorage::setNodeMapSaveDest(std::string name, MapStorage::MapSaveDest dest)
{
  NodeMapData *data = getNodeMapData(name);
  data->save_dest = dest;
}

void MapStorage::setArcMapSaveDest(std::string name, MapStorage::MapSaveDest dest)
{
  ArcMapData *data = getArcMapData(name);
  data->save_dest = dest;
}

MapStorage::ArcMapData* MapStorage::getArcMapData(std::string name) const
{
  ArcMapStore::const_iterator it = arcmaps.find(name);
  if (it != arcmaps.end())
    return it->second;
  else
    throw Error("Arc map " + name + " does not exists.");
}

MapStorage::NodeMapData* MapStorage::getNodeMapData(std::string name) const
{
  NodeMapStore::const_iterator it = nodemaps.find(name);
  if (it != nodemaps.end())
    return it->second;
  else
    throw Error("Node map " + name + " does not exists.");
}

MapValue::Type MapStorage::getNodeMapElementType(std::string name) const
{
  NodeMapData *data = getNodeMapData(name);
  return data->type();
}

MapValue::Type MapStorage::getArcMapElementType(std::string name) const
{
  ArcMapData *data = getArcMapData(name);
  return data->type();
}

const MapStorage::NodeLabelMap& MapStorage::getNodeLabelMap()
{
  return node_label;
}

const MapStorage::ArcLabelMap& MapStorage::getArcLabelMap()
{
  return arc_label;
}

const Digraph& MapStorage::getDigraph()
{
  return digraph;
}

bool MapStorage::nodeMapExists(std::string name)
{
  NodeMapStore::const_iterator it = nodemaps.find(name);
  if (it == nodemaps.end())
    return false;
  else
    return true;
}

bool MapStorage::arcMapExists(std::string name)
{
  ArcMapStore::const_iterator it = arcmaps.find(name);
  if (it == arcmaps.end())
    return false;
  else
    return true;
}

std::vector<std::string> MapStorage::getArcMaps(MapType type)
{
  std::vector<std::string> maps;
  for (ArcMapStore::const_iterator it = arcmaps.begin(); it != arcmaps.end(); ++it)
  {
    if (it->second->type() & type)
    {
      maps.push_back(it->first);
    }
  }
  return maps;
}

std::vector<std::string> MapStorage::getNodeMaps(MapType type)
{
  std::vector<std::string> maps;
  for (NodeMapStore::const_iterator it = nodemaps.begin(); it != nodemaps.end(); ++it)
  {
    if (it->second->type() & type)
    {
      maps.push_back(it->first);
    }
  }
  return maps;
}

MapStorage::NodeCoordMap& MapStorage::getNodeCoordMap()
{
  return node_coords;
}

MapStorage::ArrowCoordMap& MapStorage::getArrowCoordMap()
{
  return arrow_coords;
}

MapStorage::SpecMapSaveOpts::Dest MapStorage::getNodeCoordsSaveDest()
{
  return node_coords_save_dest;
}

MapStorage::SpecMapSaveOpts::Dest MapStorage::getArrowCoordsSaveDest()
{
  return arrow_coords_save_dest;
}

void MapStorage::setNodeCoordsSaveDest(MapStorage::SpecMapSaveOpts::Dest dest)
{
  node_coords_save_dest = dest;
}

void MapStorage::setArrowCoordsSaveDest(MapStorage::SpecMapSaveOpts::Dest dest)
{
  arrow_coords_save_dest = dest;
}

MapStorage::SpecMapSaveOpts::MapNum MapStorage::getNodeCoordsSaveMapNum()
{
  return node_coords_save_map_num;
}

MapStorage::SpecMapSaveOpts::MapNum MapStorage::getArrowCoordsSaveMapNum()
{
  return arrow_coords_save_map_num;
}

void MapStorage::setNodeCoordsSaveMapNum(MapStorage::SpecMapSaveOpts::MapNum num)
{
  node_coords_save_map_num = num;
}

void MapStorage::setArrowCoordsSaveMapNum(MapStorage::SpecMapSaveOpts::MapNum num)
{
  arrow_coords_save_map_num = num;
}

const std::string& MapStorage::getNodeCoordsOneMapName()
{
  return node_coords_one_map_name;
}
const std::string& MapStorage::getNodeCoordsTwoMaps1Name()
{
  return node_coords_two_maps_1_name;
}
const std::string& MapStorage::getNodeCoordsTwoMaps2Name()
{
  return node_coords_two_maps_2_name;
}

void MapStorage::setNodeCoordsOneMapName(const std::string& name)
{
  node_coords_one_map_name = name;
}
void MapStorage::setNodeCoordsTwoMaps1Name(const std::string& name)
{
  node_coords_two_maps_1_name = name;
}
void MapStorage::setNodeCoordsTwoMaps2Name(const std::string& name)
{
  node_coords_two_maps_2_name = name;
}

const std::string& MapStorage::getArrowCoordsOneMapName()
{
  return arrow_coords_one_map_name;
}
const std::string& MapStorage::getArrowCoordsTwoMaps1Name()
{
  return arrow_coords_two_maps_1_name;
}
const std::string& MapStorage::getArrowCoordsTwoMaps2Name()
{
  return arrow_coords_two_maps_2_name;
}

void MapStorage::setArrowCoordsOneMapName(const std::string& name)
{
  arrow_coords_one_map_name = name;
}
void MapStorage::setArrowCoordsTwoMaps1Name(const std::string& name)
{
  arrow_coords_two_maps_1_name = name;
}
void MapStorage::setArrowCoordsTwoMaps2Name(const std::string& name)
{
  arrow_coords_two_maps_2_name = name;
}

void MapStorage::readLGF(
    const std::string& filename,
    bool read_arc_label,
    const std::vector<std::string>& node_map_names,
    const std::vector<std::string>& arc_map_names,
    const std::map<std::string, MapValue::Type>& node_map_types,
    const std::map<std::string, MapValue::Type>& arc_map_types,
    const std::string& node_coord_xmap_name,
    const std::string& node_coord_ymap_name,
    const std::string& arrow_coord_xmap_name,
    const std::string& arrow_coord_ymap_name)
{
  using std::vector;
  using std::map;
  using std::string;

  DigraphReader<Digraph> greader(digraph, filename);

  // read the label maps
  greader.nodeMap("label", node_label);
  if (read_arc_label)
    greader.arcMap("label", arc_label);

  // read the node maps
  for (vector<string>::const_iterator
      it = node_map_names.begin();
      it != node_map_names.end(); ++it)
  {
    switch (node_map_types.find(*it)->second)
    {
      case MapValue::NUMERIC:
        {
          createNodeMap(*it, MapValue::NUMERIC, double());
          greader.nodeMap(*it, getNumericNodeMap(*it));
          break;
        }
      case MapValue::STRING:
        {
          createNodeMap(*it, MapValue::STRING, string());
          greader.nodeMap(*it, getStringNodeMap(*it));
          break;
        }
    }
    getNodeMapData(*it)->save_dest = NESET_SECT;
  }

  // read the arc maps
  for (vector<string>::const_iterator
      it = arc_map_names.begin();
      it != arc_map_names.end(); ++it)
  {
    switch (arc_map_types.find(*it)->second)
    {
      case MapValue::NUMERIC:
        {
          createArcMap(*it, MapValue::NUMERIC, double());
          greader.arcMap(*it, getNumericArcMap(*it));
          break;
        }
      case MapValue::STRING:
        {
          createArcMap(*it, MapValue::STRING, string());
          greader.arcMap(*it, getStringArcMap(*it));
          break;
        }
    }
    getArcMapData(*it)->save_dest = NESET_SECT;
  }

  // read the node coordinate maps
  if (node_coord_xmap_name != "")
    greader.nodeMap(node_coord_xmap_name, node_coords_x);
  if (node_coord_ymap_name != "")
    greader.nodeMap(node_coord_ymap_name, node_coords_y);

  // read the arrow coordinate maps
  if (arrow_coord_xmap_name != "")
    greader.arcMap(arrow_coord_xmap_name, arrow_coords_x);
  if (arrow_coord_ymap_name != "")
    greader.arcMap(arrow_coord_ymap_name, arrow_coords_y);

  greader.run();
}

void MapStorage::setBackground(const std::string& file_name)
{
  if (file_name == background_file_name) return;
  if (file_name == "")
  {
    background_file_name = "";
    background_set = false;
  }
  else
  {
    background_file_name = file_name;
    background_set = true;
  }
  signal_background.emit();
}

const std::string& MapStorage::getBackgroundFilename()
{
  return background_file_name;
}

bool MapStorage::isBackgroundSet()
{
  return background_set;
}

double MapStorage::getBackgroundScaling()
{
  return background_scaling;
}

void MapStorage::setBackgroundScaling(double scaling)
{
  background_scaling = scaling;
}

void MapStorage::exportDigraphToEPS(std::vector<bool> options, std::string filename, std::string shapemap)
{
  Digraph::NodeMap<int> _shapes(digraph, 0);
  Digraph::NodeMap<int> _nodeColors(digraph, 0);
  Digraph::ArcMap<int> _arcColors(digraph, 0);
  Digraph::NodeMap<double> _nodeSizes(digraph, 6.0);
  Digraph::ArcMap<double> _arcWidths(digraph, 1.0);
  bool _drawArrows=options[ARROWS];
  bool _enableParallel=options[PAR];

  std::string emptyString="";
  Digraph::NodeMap<std::string> _nodeTextMap(digraph,emptyString);

  //_nodeTextMap=(Digraph::NodeMap<void> *)&emptyStringMap;

  if(options[N_MAPS])
    {
      if(active_nodemaps[N_RADIUS]!="")
	{
	  const  Digraph::NodeMap<double> &temp = 
	    getNumericNodeMap(active_nodemaps[N_RADIUS]);
	  for(NodeIt n(digraph);n!=INVALID;++n)
	    _nodeSizes[n]= temp[n];
	}
      if(active_nodemaps[N_COLOR]!="")
	{
	  for(NodeIt ni(digraph);ni!=INVALID;++ni)
	    {
	      _nodeColors[ni]=(int)get(active_nodemaps[N_COLOR], ni);
	    }
	}
      if(active_nodemaps[N_TEXT]!="")
	{
	  for(NodeIt ni(digraph);ni!=INVALID;++ni)
	    {
	      std::ostringstream o;
	      o << get(active_nodemaps[N_TEXT], ni);
	      _nodeTextMap[ni]=o.str();	      
	    }
	}
    }
  if(options[E_MAPS])
    {
      if(active_arcmaps[E_WIDTH]!="")
	{
	  const  Digraph::ArcMap<double> &temp = 
	    getNumericArcMap(active_arcmaps[E_WIDTH]);
	  for(ArcIt a(digraph);a!=INVALID;++a)
	    _arcWidths[a]=temp[a];
	}
      if(active_arcmaps[E_COLOR]!="")
	{
	  for(ArcIt ei(digraph);ei!=INVALID;++ei)
	    {
	      _arcColors[ei]=(int)get(active_arcmaps[E_COLOR], ei);
	    }
	}
    }
  if(shapemap!="Default values")
    {
      double min = std::numeric_limits<double>::max();
      double max = std::numeric_limits<double>::min();
      for (NodeIt n(digraph); n != INVALID; ++n)
      {
        double v = static_cast<double>(get(shapemap, n));
        if (v < min) min = v;
        if (v > max) max = v;
      }
      if((min>=0)&&(max<=4))
	{
          NumericNodeMap& map = static_cast<NumericNodeMapData*>(getNodeMapData(shapemap))->map;
          for (NodeIt n(digraph); n != INVALID; ++n)
          {
            _shapes[n] = static_cast<int>(map[n]);
          }
	}
    }

  Palette palette;
  Palette paletteW(true);

  graphToEps(digraph,filename).
    title("Sample .eps figure (fits to A4)").
    copyright("(C) 2006 LEMON Project").
    absoluteNodeSizes().absoluteArcWidths().
    nodeScale(2).nodeSizes(_nodeSizes).
    coords(node_coords).
    nodeShapes(_shapes).
    nodeColors(composeMap(paletteW,_nodeColors)).
    arcColors(composeMap(palette,_arcColors)).
    arcWidthScale(0.3).arcWidths(_arcWidths).
    nodeTexts(_nodeTextMap).nodeTextSize(7).
    enableParallel(_enableParallel).parArcDist(5).
    drawArrows(_drawArrows).arrowWidth(7).arrowLength(7).
    run();

}
