// -*- c++ -*-
#ifndef MAP_DEFINES_H
#define MAP_DEFINES_H

/** Creates the EdgeMapRegistry type an declare a mutable instance 
 *  named edge_maps.
 */
#define CREATE_EDGE_MAP_REGISTRY \
typedef MapRegistry<Graph, Edge, EdgeIt> EdgeMapRegistry; \
mutable EdgeMapRegistry edge_maps;

/** Creates the NodeMapRegistry type an declare a mutable instance 
 *  named node_maps.
 */
#define CREATE_NODE_MAP_REGISTRY \
typedef MapRegistry<Graph, Node, NodeIt> NodeMapRegistry; \
mutable NodeMapRegistry node_maps;

/** Creates both map registries.
 */
#define CREATE_MAP_REGISTRIES \
CREATE_NODE_MAP_REGISTRY \
CREATE_EDGE_MAP_REGISTRY

/** Creates a concrete factory type from a template map
 *  factory to use as node map factory.
 */
#define CREATE_NODE_MAP_FACTORY(TemplateFactory) \
typedef TemplateFactory<NodeMapRegistry> NodeMapFactory;

/** Creates a concrete factory type from a template map
 *  factory to use as edge map factory.
 */
#define CREATE_EDGE_MAP_FACTORY(TemplateFactory) \
typedef TemplateFactory<EdgeMapRegistry> EdgeMapFactory;

/** Creates both map factories.
 */
#define CREATE_MAP_FACTORIES(TemplateFactory) \
CREATE_NODE_MAP_FACTORY(TemplateFactory) \
CREATE_EDGE_MAP_FACTORY(TemplateFactory) 

/** Import a map from a concrete map factory. The import method is
 *  an overloading of the map type.
 *  The reason to use these macro is that the c++ does not support
 *  the template typedefs. If a future release of the c++ 
 *  supports this feature it should be fixed.
 */
#define IMPORT_NODE_MAP(Factory) \
template <typename V> \
class NodeMap : public Factory::template Map<V> { \
typedef typename Factory::template Map<V> MapImpl; \
public: \
NodeMap() {} \
NodeMap(const Graph& g) : MapImpl(g, g.node_maps) {} \
NodeMap(const Graph& g, const V& v) : MapImpl(g, g.node_maps, v) {} \
NodeMap(const NodeMap& copy) : MapImpl(static_cast<const MapImpl&>(copy)) {} \
template <typename CMap> NodeMap(const CMap& copy) : MapImpl(copy) {} \
NodeMap& operator=(const NodeMap& copy) { \
  MapImpl::operator=(static_cast<const MapImpl&>(copy));\
  return *this; \
} \
template <typename CMap> NodeMap& operator=(const CMap& copy) { \
  MapImpl::operator=(copy);\
  return *this; \
} \
};

/** Import a map from a concrete map factory. The import method is
 *  an overloading of the map type.
 *  The reason to use these macro is that the c++ does not support
 *  the template typedefs. If a future release of the c++ 
 *  supports this feature it should be fixed.
 */
#define IMPORT_EDGE_MAP(Factory) \
template <typename V> \
class EdgeMap : public Factory::template Map<V> { \
typedef typename Factory::template Map<V> MapImpl; \
public: \
EdgeMap() {} \
EdgeMap(const Graph& g) : MapImpl(g, g.edge_maps) {} \
EdgeMap(const Graph& g, const V& v) : MapImpl(g, g.edge_maps, v) {} \
EdgeMap(const EdgeMap& copy) : MapImpl(static_cast<const MapImpl&>(copy)) {} \
template <typename CMap> EdgeMap(const CMap& copy) : MapImpl(copy) {} \
EdgeMap& operator=(const EdgeMap& copy) { \
  MapImpl::operator=(static_cast<const MapImpl&>(copy));\
  return *this; \
} \
template <typename CMap> EdgeMap& operator=(const CMap& copy) { \
  MapImpl::operator=(copy);\
  return *this; \
} \
};

/** This macro creates both map factories and imports both maps.
 */
#define CREATE_MAPS(TemplateFactory) \
CREATE_MAP_FACTORIES(TemplateFactory) \
IMPORT_NODE_MAP(NodeMapFactory) \
IMPORT_EDGE_MAP(EdgeMapFactory)

/** This macro creates MapRegistry for Symmetric Edge Maps.
 */
#define CREATE_SYM_EDGE_MAP_REGISTRY \
typedef SymEdgeIt<Graph, Edge, EdgeIt> SymEdgeIt; \
typedef MapRegistry<Graph, Edge, SymEdgeIt> SymEdgeMapRegistry; \
mutable EdgeMapRegistry sym_edge_maps;

/** Creates a concrete factory type from a template map
 *  factory to use as edge map factory.
 */
#define CREATE_SYM_EDGE_MAP_FACTORY(TemplateFactory) \
typedef SymMapFactory<SymEdgeMapRegistry, TemplateFactory > \
SymEdgeMapFactory;

/** Import a map from a concrete map factory. The import method is
 *  an overloading of the map type.
 *  The reason to use these macro is that the c++ does not support
 *  the template typedefs. If a future release of the c++ 
 *  supports this feature it should be fixed.
 */
#define IMPORT_SYM_EDGE_MAP(Factory) \
template <typename V> \
class SymEdgeMap : public Factory::template Map<V> { \
typedef typename Factory::template Map<V> MapImpl; \
public: \
SymEdgeMap() {} \
SymEdgeMap(const Graph& g) : MapImpl(g, g.sym_edge_maps) {} \
SymEdgeMap(const Graph& g, const V& v) : MapImpl(g, g.sym_edge_maps, v) {} \
SymEdgeMap(const SymEdgeMap& copy) \
  : MapImpl(static_cast<const MapImpl&>(copy)) {} \
template <typename CMap> SymEdgeMap(const CMap& copy) : MapImpl(copy) {} \
SymEdgeMap& operator=(const SymEdgeMap& copy) { \
  MapImpl::operator=(static_cast<const MapImpl&>(copy));\
  return *this; \
} \
template <typename CMap> SymEdgeMap& operator=(const CMap& copy) { \
  MapImpl::operator=(copy);\
  return *this; \
} \
};


#define KEEP_NODE_MAP(GraphBase) \
    template <typename V> class NodeMap \
      : public GraphBase::template NodeMap<V> \
    { \
      typedef typename GraphBase::template NodeMap<V> MapImpl; \
      typedef V Value; \
    public: \
      NodeMap() : MapImpl() {} \
\
      NodeMap(const Graph& graph) \
	: MapImpl(static_cast<const GraphBase&>(graph)) { } \
\
      NodeMap(const Graph& graph, const Value& value) \
	: MapImpl(static_cast<const GraphBase&>(graph), value) { } \
\
      NodeMap(const NodeMap& copy) \
	: MapImpl(static_cast<const MapImpl&>(copy)) {} \
\
      template<typename CMap> \
      NodeMap(const CMap& copy) \
	: MapImpl(copy) {} \
\
      NodeMap& operator=(const NodeMap& copy) { \
	MapImpl::operator=(static_cast<const MapImpl&>(copy)); \
	return *this; \
      } \
\
      template <typename CMap> \
      NodeMap& operator=(const CMap& copy) { \
	MapImpl::operator=(copy); \
	return *this; \
      } \
    };

#define KEEP_EDGE_MAP(GraphBase) \
    template <typename V> class EdgeMap \
      : public GraphBase::template EdgeMap<V> \
    { \
      typedef typename GraphBase::template EdgeMap<V> MapImpl; \
      typedef V Value; \
    public: \
      EdgeMap() : MapImpl() {} \
\
      EdgeMap(const Graph& graph) \
	: MapImpl(static_cast<const GraphBase&>(graph)) { } \
\
      EdgeMap(const Graph& graph, const Value& value) \
	: MapImpl(static_cast<const GraphBase&>(graph), value) { } \
\
      EdgeMap(const EdgeMap& copy) \
	: MapImpl(static_cast<const MapImpl&>(copy)) {} \
\
      template<typename CMap> \
      EdgeMap(const CMap& copy) \
	: MapImpl(copy) {} \
\
      EdgeMap& operator=(const EdgeMap& copy) { \
	MapImpl::operator=(static_cast<const MapImpl&>(copy)); \
	return *this; \
      } \
\
      template <typename CMap> \
      EdgeMap& operator=(const CMap& copy) { \
	MapImpl::operator=(copy); \
	return *this; \
      } \
    };

#endif
