// -*- c++ -*-
#ifndef HUGO_TIME_MEASURE_H
#define HUGO_TIME_MEASURE_H

#include <sys/time.h>
#include <sys/times.h>
#include <fstream>
#include <iostream>
#include <unistd.h>

namespace hugo {

//   double currTime() {
//     timeval tv;
//     //timezone tz;
//     gettimeofday(&tv, 0);
//     return double(tv.tv_sec)+double(tv.tv_usec)/1000000.0;
//   }

  /// Class to store (cpu)time instances.

  /// This class stores five time values.
  /// - a real time
  /// - a user cpu time
  /// - a system cpu time
  /// - a user cpu time of children
  /// - a system cpu time of children
  ///
  /// TimeStamp's can be added to or substracted from each other and
  /// they can be push to a stream.

  class TimeStamp
  {
    tms ts;
    double real_time;
  
  public:

    tms &getTms() {return ts;}
    const tms &getTms() const {return ts;}
    double getRealTime() const {return real_time;}
    ///Read the current time values of the process.
    void stamp()
    {
      timeval tv;
      times(&ts);
      gettimeofday(&tv, 0);real_time=tv.tv_sec+double(tv.tv_usec)/1e6;
    }
  
    /// Constructor initializing with zero.
    TimeStamp()
    { ts.tms_utime=ts.tms_stime=ts.tms_cutime=ts.tms_cstime=0; real_time=0;}
    ///Constructor initializing with the current time values of the process.
    TimeStamp(void *) { stamp();}
  
    TimeStamp &operator+=(const TimeStamp &b)
    {
      ts.tms_utime+=b.ts.tms_utime;
      ts.tms_stime+=b.ts.tms_stime;
      ts.tms_cutime+=b.ts.tms_cutime;
      ts.tms_cstime+=b.ts.tms_cstime;
      real_time+=b.real_time;
      return *this;
    }
    TimeStamp operator+(const TimeStamp &b) const
    {
      TimeStamp t(*this);
      return t+=b;
    }
    TimeStamp &operator-=(const TimeStamp &b)
    {
      ts.tms_utime-=b.ts.tms_utime;
      ts.tms_stime-=b.ts.tms_stime;
      ts.tms_cutime-=b.ts.tms_cutime;
      ts.tms_cstime-=b.ts.tms_cstime;
      real_time-=b.real_time;
      return *this;
    }
    TimeStamp operator-(const TimeStamp &b) const
    {
      TimeStamp t(*this);
      return t-=b;
    }

    ///The time ellapsed since the last call of stamp()
    TimeStamp ellapsed() const
    {
      TimeStamp t(NULL);
      return t-*this;
    }
  
    friend std::ostream& operator<<(std::ostream& os,const TimeStamp &t);
  
    double getUserTime() const
    {
      return double(ts.tms_utime)/sysconf(_SC_CLK_TCK);
    }
    double getSystemTime() const
    {
      return double(ts.tms_stime)/sysconf(_SC_CLK_TCK);
    }
    double getCUserTime() const
    {
      return double(ts.tms_cutime)/sysconf(_SC_CLK_TCK);
    }
    double getCSystemTime() const
    {
      return double(ts.tms_cstime)/sysconf(_SC_CLK_TCK);
    }
  };

  ///Class measuring the cpu time and real time usage of the process.
  class Timer
  {
    TimeStamp start_time;

    void _reset() {start_time.stamp();}
  
  public: 
    ///Constructor. It starts with zero time counters.
    Timer() {_reset();}

    ///Computes the ellapsed time.

    ///This conversion computes the ellapsed time
    ///since the construction of \c t or since
    ///the last \c t.reset().
    operator TimeStamp ()
    {
      TimeStamp t;
      t.stamp();
      return t-start_time;
    }

    ///Resets the time counters.
    TimeStamp reset()
    {
      TimeStamp t(start_time);
      _reset();
      return start_time-t;
    }
  };

  ///Prints the time counters.
  inline std::ostream& operator<<(std::ostream& os,const TimeStamp &t)
  {
    long cls = sysconf(_SC_CLK_TCK);
    os << "u: " << double(t.getTms().tms_utime)/cls <<
      "s, s: " << double(t.getTms().tms_stime)/cls <<
      "s, cu: " << double(t.getTms().tms_cutime)/cls <<
      "s, cs: " << double(t.getTms().tms_cstime)/cls <<
      "s, real: " << t.getRealTime() << "s";
    return os;
  }

} //namespace hugo

#endif //HUGO_TIME_MEASURE_H
