// -*- c++ -*-
#ifndef LEMON_EDMONDS_KARP_H
#define LEMON_EDMONDS_KARP_H

#include <algorithm>
#include <list>
#include <iterator>

#include <bfs_iterator.h>
#include <invalid.h>

namespace lemon {

  template<typename Graph, typename Number, typename FlowMap, typename CapacityMap>
  class ResGraph {
  public:
    typedef typename Graph::Node Node;
    typedef typename Graph::NodeIt NodeIt;
  private:
    typedef typename Graph::SymEdgeIt OldSymEdgeIt;
    const Graph& G;
    FlowMap& flow;
    const CapacityMap& capacity;
  public:
    ResGraph(const Graph& _G, FlowMap& _flow, 
	     const CapacityMap& _capacity) : 
      G(_G), flow(_flow), capacity(_capacity) { }

    class Edge; 
    class OutEdgeIt; 
    friend class Edge; 
    friend class OutEdgeIt; 

    class Edge {
      friend class ResGraph<Graph, Number, FlowMap, CapacityMap>;
    protected:
      const ResGraph<Graph, Number, FlowMap, CapacityMap>* resG;
      OldSymEdgeIt sym;
    public:
      Edge() { } 
      //Edge(const Edge& e) : resG(e.resG), sym(e.sym) { }
      Number free() const { 
	if (resG->G.aNode(sym)==resG->G.source(sym)) { 
	  return (resG->capacity.get(sym)-resG->flow.get(sym)); 
	} else { 
	  return (resG->flow.get(sym)); 
	}
      }
      bool valid() const { return sym.valid(); }
      void augment(Number a) const {
	if (resG->G.aNode(sym)==resG->G.source(sym)) { 
	  resG->flow.set(sym, resG->flow.get(sym)+a);
	  //resG->flow[sym]+=a;
	} else { 
	  resG->flow.set(sym, resG->flow.get(sym)-a);
	  //resG->flow[sym]-=a;
	}
      }
    };

    class OutEdgeIt : public Edge {
      friend class ResGraph<Graph, Number, FlowMap, CapacityMap>;
    public:
      OutEdgeIt() { }
      //OutEdgeIt(const OutEdgeIt& e) { resG=e.resG; sym=e.sym; }
    private:
      OutEdgeIt(const ResGraph<Graph, Number, FlowMap, CapacityMap>& _resG, Node v) { 
      	resG=&_resG;
	sym=resG->G.template first<OldSymEdgeIt>(v);
	while( sym.valid() && !(free()>0) ) { ++sym; }
      }
    public:
      OutEdgeIt& operator++() { 
	++sym; 
	while( sym.valid() && !(free()>0) ) { ++sym; }
	return *this; 
      }
    };

    void /*getF*/first(OutEdgeIt& e, Node v) const { 
      e=OutEdgeIt(*this, v); 
    }
    void /*getF*/first(NodeIt& v) const { G./*getF*/first(v); }
    
    template< typename It >
    It first() const { 
      It e;      
      /*getF*/first(e);
      return e; 
    }

    template< typename It >
    It first(Node v) const { 
      It e;
      /*getF*/first(e, v);
      return e; 
    }

    Node source(Edge e) const { return G.aNode(e.sym); }
    Node target(Edge e) const { return G.bNode(e.sym); }

    Node aNode(OutEdgeIt e) const { return G.aNode(e.sym); }
    Node bNode(OutEdgeIt e) const { return G.bNode(e.sym); }

    int id(Node v) const { return G.id(v); }

    template <typename S>
    class NodeMap {
      typename Graph::NodeMap<S> node_map; 
    public:
      NodeMap(const ResGraph<Graph, Number, FlowMap, CapacityMap>& _G) : node_map(_G.G) { }
      NodeMap(const ResGraph<Graph, Number, FlowMap, CapacityMap>& _G, S a) : node_map(_G.G, a) { }
      void set(Node nit, S a) { node_map.set(nit, a); }
      S get(Node nit) const { return node_map.get(nit); }
      S& operator[](Node nit) { return node_map[nit]; } 
      const S& operator[](Node nit) const { return node_map[nit]; } 
    };

  };


  template<typename Graph, typename Number, typename FlowMap, typename CapacityMap>
  class ResGraph2 {
  public:
    typedef typename Graph::Node Node;
    typedef typename Graph::NodeIt NodeIt;
  private:
    //typedef typename Graph::SymEdgeIt OldSymEdgeIt;
    typedef typename Graph::OutEdgeIt OldOutEdgeIt;
    typedef typename Graph::InEdgeIt OldInEdgeIt;
    
    const Graph& G;
    FlowMap& flow;
    const CapacityMap& capacity;
  public:
    ResGraph2(const Graph& _G, FlowMap& _flow, 
	     const CapacityMap& _capacity) : 
      G(_G), flow(_flow), capacity(_capacity) { }

    class Edge; 
    class OutEdgeIt; 
    friend class Edge; 
    friend class OutEdgeIt; 

    class Edge {
      friend class ResGraph2<Graph, Number, FlowMap, CapacityMap>;
    protected:
      const ResGraph2<Graph, Number, FlowMap, CapacityMap>* resG;
      //OldSymEdgeIt sym;
      OldOutEdgeIt out;
      OldInEdgeIt in;
      bool out_or_in; //true, iff out
    public:
      Edge() : out_or_in(true) { } 
      Number free() const { 
	if (out_or_in) { 
	  return (resG->capacity.get(out)-resG->flow.get(out)); 
	} else { 
	  return (resG->flow.get(in)); 
	}
      }
      bool valid() const { 
	return out_or_in && out.valid() || in.valid(); }
      void augment(Number a) const {
	if (out_or_in) { 
	  resG->flow.set(out, resG->flow.get(out)+a);
	} else { 
	  resG->flow.set(in, resG->flow.get(in)-a);
	}
      }
    };

    class OutEdgeIt : public Edge {
      friend class ResGraph2<Graph, Number, FlowMap, CapacityMap>;
    public:
      OutEdgeIt() { }
    private:
      OutEdgeIt(const ResGraph2<Graph, Number, FlowMap, CapacityMap>& _resG, Node v) { 
      	resG=&_resG;
	out=resG->G.template first<OldOutEdgeIt>(v);
	while( out.valid() && !(free()>0) ) { ++out; }
	if (!out.valid()) {
	  out_or_in=0;
	  in=resG->G.template first<OldInEdgeIt>(v);
	  while( in.valid() && !(free()>0) ) { ++in; }
	}
      }
    public:
      OutEdgeIt& operator++() { 
	if (out_or_in) {
	  Node v=resG->G.aNode(out);
	  ++out;
	  while( out.valid() && !(free()>0) ) { ++out; }
	  if (!out.valid()) {
	    out_or_in=0;
	    in=resG->G.template first<OldInEdgeIt>(v);
	    while( in.valid() && !(free()>0) ) { ++in; }
	  }
	} else {
	  ++in;
	  while( in.valid() && !(free()>0) ) { ++in; } 
	}
	return *this; 
      }
    };

    void /*getF*/first(OutEdgeIt& e, Node v) const { 
      e=OutEdgeIt(*this, v); 
    }
    void /*getF*/first(NodeIt& v) const { G./*getF*/first(v); }
    
    template< typename It >
    It first() const { 
      It e;
      /*getF*/first(e);
      return e; 
    }

    template< typename It >
    It first(Node v) const { 
      It e;
      /*getF*/first(e, v);
      return e; 
    }

    Node source(Edge e) const { 
      return ((e.out_or_in) ? G.aNode(e.out) : G.aNode(e.in)); }
    Node target(Edge e) const { 
      return ((e.out_or_in) ? G.bNode(e.out) : G.bNode(e.in)); }

    Node aNode(OutEdgeIt e) const { 
      return ((e.out_or_in) ? G.aNode(e.out) : G.aNode(e.in)); }
    Node bNode(OutEdgeIt e) const { 
      return ((e.out_or_in) ? G.bNode(e.out) : G.bNode(e.in)); }

    int id(Node v) const { return G.id(v); }

    template <typename S>
    class NodeMap {
      typename Graph::NodeMap<S> node_map; 
    public:
      NodeMap(const ResGraph2<Graph, Number, FlowMap, CapacityMap>& _G) : node_map(_G.G) { }
      NodeMap(const ResGraph2<Graph, Number, FlowMap, CapacityMap>& _G, S a) : node_map(_G.G, a) { }
      void set(Node nit, S a) { node_map.set(nit, a); }
      S get(Node nit) const { return node_map.get(nit); }
    };
  };


  template <typename GraphWrapper, typename Number, typename FlowMap, typename CapacityMap>
  class MaxFlow {
  protected:
    typedef GraphWrapper GW;
    typedef typename GW::Node Node;
    typedef typename GW::Edge Edge;
    typedef typename GW::EdgeIt EdgeIt;
    typedef typename GW::OutEdgeIt OutEdgeIt;
    typedef typename GW::InEdgeIt InEdgeIt;
    //const Graph* G;
    GW gw;
    Node s;
    Node t;
    FlowMap* flow;
    const CapacityMap* capacity;
    typedef ResGraphWrapper<GW, Number, FlowMap, CapacityMap > ResGW;
    typedef typename ResGW::OutEdgeIt ResGWOutEdgeIt;
    typedef typename ResGW::Edge ResGWEdge;
  public:

    MaxFlow(const GW& _gw, Node _s, Node _t, FlowMap& _flow, const CapacityMap& _capacity) : 
      gw(_gw), s(_s), t(_t), flow(&_flow), capacity(&_capacity) { }

    bool augmentOnShortestPath() {
      ResGW res_graph(gw, *flow, *capacity);
      bool _augment=false;
      
      typedef typename ResGW::NodeMap<bool> ReachedMap;
      BfsIterator5< ResGW, ReachedMap > bfs(res_graph);
      bfs.pushAndSetReached(s);
	
      typename ResGW::NodeMap<ResGWEdge> pred(res_graph); 
      pred.set(s, INVALID);
      
      typename ResGW::NodeMap<Number> free(res_graph);
	
      //searching for augmenting path
      while ( !bfs.finished() ) { 
	ResGWOutEdgeIt e=bfs;
	if (res_graph.valid(e) && bfs.isBNodeNewlyReached()) {
	  Node v=res_graph.source(e);
	  Node w=res_graph.target(e);
	  pred.set(w, e);
	  if (res_graph.valid(pred.get(v))) {
	    free.set(w, std::min(free.get(v), res_graph.resCap(e)));
	  } else {
	    free.set(w, res_graph.resCap(e)); 
	  }
	  if (res_graph.target(e)==t) { _augment=true; break; }
	}
	
	++bfs;
      } //end of searching augmenting path

      if (_augment) {
	Node n=t;
	Number augment_value=free.get(t);
	while (res_graph.valid(pred.get(n))) { 
	  ResGWEdge e=pred.get(n);
	  res_graph.augment(e, augment_value); 
	  n=res_graph.source(e);
	}
      }

      return _augment;
    }

    template<typename MapGraphWrapper> 
    class DistanceMap {
    protected:
      MapGraphWrapper gw;
      typename MapGraphWrapper::NodeMap<int> dist; 
    public:
      DistanceMap(MapGraphWrapper& _gw) : gw(_gw), dist(_gw, _gw.nodeNum()) { }
      void set(const typename MapGraphWrapper::Node& n, int a) { dist[n]=a; }
      int get(const typename MapGraphWrapper::Node& n) const { return dist[n]; }
      bool get(const typename MapGraphWrapper::Edge& e) const { 
	return (dist.get(gw.source(e))<dist.get(gw.target(e))); 
      }
    };

    template<typename MutableGraph> bool augmentOnBlockingFlow() {      
      typedef MutableGraph MG;
      bool _augment=false;

      ResGW res_graph(gw, *flow, *capacity);

      typedef typename ResGW::NodeMap<bool> ReachedMap;
      BfsIterator5< ResGW, ReachedMap > bfs(res_graph);

      bfs.pushAndSetReached(s);
      //typename ResGW::NodeMap<int> dist(res_graph); //filled up with 0's
      DistanceMap<ResGW> dist(res_graph);
      while ( !bfs.finished() ) { 
	ResGWOutEdgeIt e=bfs;
	if (res_graph.valid(e) && bfs.isBNodeNewlyReached()) {
	  dist.set(res_graph.target(e), dist.get(res_graph.source(e))+1);
	}
	++bfs;
      } //computing distances from s in the residual graph

      MG F;
      typedef SubGraphWrapper<ResGW, DistanceMap<ResGW> > FilterResGW;
      FilterResGW filter_res_graph(res_graph, dist);
      typename ResGW::NodeMap<typename MG::Node> res_graph_to_F(res_graph);
      {
	typename ResGW::NodeIt n;
	for(res_graph.first(n); res_graph.valid(n); res_graph.next(n)) {
	  res_graph_to_F.set(n, F.addNode());
	}
      }

      typename MG::Node sF=res_graph_to_F.get(s);
      typename MG::Node tF=res_graph_to_F.get(t);
      typename MG::EdgeMap<ResGWEdge> original_edge(F);
      typename MG::EdgeMap<Number> residual_capacity(F);

      //Making F to the graph containing the edges of the residual graph 
      //which are in some shortest paths
      {
	typename FilterResGW::EdgeIt e;
	for(filter_res_graph.first(e); filter_res_graph.valid(e); filter_res_graph.next(e)) {
	  //if (dist.get(res_graph.target(e))==dist.get(res_graph.source(e))+1) {
	  typename MG::Edge f=F.addEdge(res_graph_to_F.get(res_graph.source(e)), res_graph_to_F.get(res_graph.target(e)));
	  original_edge.update();
	  original_edge.set(f, e);
	  residual_capacity.update();
	  residual_capacity.set(f, res_graph.resCap(e));
	  //} 
	}
      }

      bool __augment=true;

      while (__augment) {
	__augment=false;
	//computing blocking flow with dfs
	typedef typename TrivGraphWrapper<MG>::NodeMap<bool> BlockingReachedMap;
	DfsIterator5< TrivGraphWrapper<MG>, BlockingReachedMap > dfs(F);
	typename MG::NodeMap<typename MG::Edge> pred(F);
	pred.set(sF, INVALID);
	//invalid iterators for sources

	typename MG::NodeMap<Number> free(F);

	dfs.pushAndSetReached(sF);      
	while (!dfs.finished()) {
	  ++dfs;
	  if (F.valid(/*typename MG::OutEdgeIt*/(dfs))) {
	    if (dfs.isBNodeNewlyReached()) {
	      typename MG::Node v=F.aNode(dfs);
	      typename MG::Node w=F.bNode(dfs);
	      pred.set(w, dfs);
	      if (F.valid(pred.get(v))) {
		free.set(w, std::min(free.get(v), residual_capacity.get(dfs)));
	      } else {
		free.set(w, residual_capacity.get(dfs)); 
	      }
	      if (w==tF) { 
		__augment=true; 
		_augment=true;
		break; 
	      }
	      
	    } else {
	      F.erase(/*typename MG::OutEdgeIt*/(dfs));
	    }
	  } 
	}

	if (__augment) {
	  typename MG::Node n=tF;
	  Number augment_value=free.get(tF);
	  while (F.valid(pred.get(n))) { 
	    typename MG::Edge e=pred.get(n);
	    res_graph.augment(original_edge.get(e), augment_value); 
	    n=F.source(e);
	    if (residual_capacity.get(e)==augment_value) 
	      F.erase(e); 
	    else 
	      residual_capacity.set(e, residual_capacity.get(e)-augment_value);
	  }
	}
	
      }
            
      return _augment;
    }

    template<typename MutableGraph> bool augmentOnBlockingFlow1() {      
      typedef MutableGraph MG;
      bool _augment=false;

      ResGW res_graph(gw, *flow, *capacity);

      //bfs for distances on the residual graph
      typedef typename ResGW::NodeMap<bool> ReachedMap;
      BfsIterator5< ResGW, ReachedMap > bfs(res_graph);
      bfs.pushAndSetReached(s);
      typename ResGW::NodeMap<int> dist(res_graph); //filled up with 0's

      //F will contain the physical copy of the residual graph
      //with the set of edges which are on shortest paths
      MG F;
      typename ResGW::NodeMap<typename MG::Node> res_graph_to_F(res_graph);
      {
	typename ResGW::NodeIt n;
	for(res_graph.first(n); res_graph.valid(n); res_graph.next(n)) {
	  res_graph_to_F.set(n, F.addNode());
	}
      }

      typename MG::Node sF=res_graph_to_F.get(s);
      typename MG::Node tF=res_graph_to_F.get(t);
      typename MG::EdgeMap<ResGWEdge> original_edge(F);
      typename MG::EdgeMap<Number> residual_capacity(F);

      while ( !bfs.finished() ) { 
	ResGWOutEdgeIt e=bfs;
	if (res_graph.valid(e)) {
	  if (bfs.isBNodeNewlyReached()) {
	    dist.set(res_graph.target(e), dist.get(res_graph.source(e))+1);
	    typename MG::Edge f=F.addEdge(res_graph_to_F.get(res_graph.source(e)), res_graph_to_F.get(res_graph.target(e)));
	    original_edge.update();
	    original_edge.set(f, e);
	    residual_capacity.update();
	    residual_capacity.set(f, res_graph.resCap(e));
	  } else {
	    if (dist.get(res_graph.target(e))==(dist.get(res_graph.source(e))+1)) {
	      typename MG::Edge f=F.addEdge(res_graph_to_F.get(res_graph.source(e)), res_graph_to_F.get(res_graph.target(e)));
	      original_edge.update();
	      original_edge.set(f, e);
	      residual_capacity.update();
	      residual_capacity.set(f, res_graph.resCap(e));
	    }
	  }
	}
	++bfs;
      } //computing distances from s in the residual graph

      bool __augment=true;

      while (__augment) {
	__augment=false;
	//computing blocking flow with dfs
	typedef typename TrivGraphWrapper<MG>::NodeMap<bool> BlockingReachedMap;
	DfsIterator5< TrivGraphWrapper<MG>, BlockingReachedMap > dfs(F);
	typename MG::NodeMap<typename MG::Edge> pred(F);
	pred.set(sF, INVALID);
	//invalid iterators for sources

	typename MG::NodeMap<Number> free(F);

	dfs.pushAndSetReached(sF);      
	while (!dfs.finished()) {
	  ++dfs;
	  if (F.valid(/*typename MG::OutEdgeIt*/(dfs))) {
	    if (dfs.isBNodeNewlyReached()) {
	      typename MG::Node v=F.aNode(dfs);
	      typename MG::Node w=F.bNode(dfs);
	      pred.set(w, dfs);
	      if (F.valid(pred.get(v))) {
		free.set(w, std::min(free.get(v), residual_capacity.get(dfs)));
	      } else {
		free.set(w, residual_capacity.get(dfs)); 
	      }
	      if (w==tF) { 
		__augment=true; 
		_augment=true;
		break; 
	      }
	      
	    } else {
	      F.erase(/*typename MG::OutEdgeIt*/(dfs));
	    }
	  } 
	}

	if (__augment) {
	  typename MG::Node n=tF;
	  Number augment_value=free.get(tF);
	  while (F.valid(pred.get(n))) { 
	    typename MG::Edge e=pred.get(n);
	    res_graph.augment(original_edge.get(e), augment_value); 
	    n=F.source(e);
	    if (residual_capacity.get(e)==augment_value) 
	      F.erase(e); 
	    else 
	      residual_capacity.set(e, residual_capacity.get(e)-augment_value);
	  }
	}
	
      }
            
      return _augment;
    }

    bool augmentOnBlockingFlow2() {
      bool _augment=false;

      ResGW res_graph(gw, *flow, *capacity);

      typedef typename ResGW::NodeMap<bool> ReachedMap;
      BfsIterator5< ResGW, ReachedMap > bfs(res_graph);

      bfs.pushAndSetReached(s);
      DistanceMap<ResGW> dist(res_graph);
      while ( !bfs.finished() ) { 
 	ResGWOutEdgeIt e=bfs;
 	if (res_graph.valid(e) && bfs.isBNodeNewlyReached()) {
 	  dist.set(res_graph.target(e), dist.get(res_graph.source(e))+1);
 	}
	++bfs;
      } //computing distances from s in the residual graph

      //Subgraph containing the edges on some shortest paths
      typedef SubGraphWrapper<ResGW, DistanceMap<ResGW> > FilterResGW;
      FilterResGW filter_res_graph(res_graph, dist);

      //Subgraph, which is able to delete edges which are already 
      //met by the dfs
      typename FilterResGW::NodeMap<typename FilterResGW::OutEdgeIt> 
 	first_out_edges(filter_res_graph);
      typename FilterResGW::NodeIt v;
      for(filter_res_graph.first(v); filter_res_graph.valid(v); 
 	  filter_res_graph.next(v)) 
      {
 	typename FilterResGW::OutEdgeIt e;
 	filter_res_graph.first(e, v);
 	first_out_edges.set(v, e);
      }
      typedef ErasingFirstGraphWrapper<FilterResGW, typename FilterResGW::
	NodeMap<typename FilterResGW::OutEdgeIt> > ErasingResGW;
      ErasingResGW erasing_res_graph(filter_res_graph, first_out_edges);

      bool __augment=true;

      while (__augment) {

 	__augment=false;
 	//computing blocking flow with dfs
	typedef typename ErasingResGW::NodeMap<bool> BlockingReachedMap;
 	DfsIterator5< ErasingResGW, BlockingReachedMap > 
 	  dfs(erasing_res_graph);
 	typename ErasingResGW::NodeMap<typename ErasingResGW::OutEdgeIt> 
 	  pred(erasing_res_graph); 
 	pred.set(s, INVALID);
 	//invalid iterators for sources

 	typename ErasingResGW::NodeMap<Number> free(erasing_res_graph);

 	dfs.pushAndSetReached(s);
 	while (!dfs.finished()) {
 	  ++dfs;
 	  if (erasing_res_graph.valid(
 		/*typename ErasingResGW::OutEdgeIt*/(dfs))) 
 	  { 
 	    if (dfs.isBNodeNewlyReached()) {
	  
 	      typename ErasingResGW::Node v=erasing_res_graph.aNode(dfs);
 	      typename ErasingResGW::Node w=erasing_res_graph.bNode(dfs);

 	      pred.set(w, /*typename ErasingResGW::OutEdgeIt*/(dfs));
 	      if (erasing_res_graph.valid(pred.get(v))) {
 		free.set(w, std::min(free.get(v), res_graph.resCap(dfs)));
 	      } else {
 		free.set(w, res_graph.resCap(dfs)); 
 	      }
	      
 	      if (w==t) { 
 		__augment=true; 
 		_augment=true;
 		break; 
 	      }
	    } else {
	      erasing_res_graph.erase(dfs);
	    }
	  }
	}	

 	if (__augment) {
 	  typename ErasingResGW::Node n=t;
 	  Number augment_value=free.get(n);
 	  while (erasing_res_graph.valid(pred.get(n))) { 
 	    typename ErasingResGW::OutEdgeIt e=pred.get(n);
 	    res_graph.augment(e, augment_value);
 	    n=erasing_res_graph.source(e);
 	    if (res_graph.resCap(e)==0)
 	      erasing_res_graph.erase(e);
 	  }
 	}
      
      } //while (__augment) 
            
      return _augment;
    }

//     bool augmentOnBlockingFlow2() {
//       bool _augment=false;

//       //typedef ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap> EAugGraph;
//       typedef FilterGraphWrapper< ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap> > EAugGraph;
//       typedef typename EAugGraph::OutEdgeIt EAugOutEdgeIt;
//       typedef typename EAugGraph::Edge EAugEdge;

//       EAugGraph res_graph(*G, *flow, *capacity);

//       //typedef typename EAugGraph::NodeMap<bool> ReachedMap;
//       BfsIterator5< 
// 	ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>, 
// 	/*typename ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>::OutEdgeIt,*/ 
// 	ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>::NodeMap<bool> > bfs(res_graph);
      
//       bfs.pushAndSetReached(s);

//       typename ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>::
// 	NodeMap<int>& dist=res_graph.dist;

//       while ( !bfs.finished() ) {
// 	typename ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>::OutEdgeIt e=bfs;
// 	if (res_graph.valid(e) && bfs.isBNodeNewlyReached()) {
// 	  dist.set(res_graph.target(e), dist.get(res_graph.source(e))+1);
// 	}
// 	++bfs;	
//       } //computing distances from s in the residual graph

//       bool __augment=true;

//       while (__augment) {

// 	__augment=false;
// 	//computing blocking flow with dfs
// 	typedef typename EAugGraph::NodeMap<bool> BlockingReachedMap;
// 	DfsIterator5< EAugGraph/*, EAugOutEdgeIt*/, BlockingReachedMap > 
// 	  dfs(res_graph);
// 	typename EAugGraph::NodeMap<EAugEdge> pred(res_graph); 
// 	pred.set(s, EAugEdge(INVALID));
// 	//invalid iterators for sources

// 	typename EAugGraph::NodeMap<Number> free(res_graph);

// 	dfs.pushAndSetReached(s);
// 	while (!dfs.finished()) {
// 	  ++dfs;
// 	  if (res_graph.valid(EAugOutEdgeIt(dfs))) { 
// 	    if (dfs.isBNodeNewlyReached()) {
	  
// 	      typename EAugGraph::Node v=res_graph.aNode(dfs);
// 	      typename EAugGraph::Node w=res_graph.bNode(dfs);

// 	      pred.set(w, EAugOutEdgeIt(dfs));
// 	      if (res_graph.valid(pred.get(v))) {
// 		free.set(w, std::min(free.get(v), res_graph.free(dfs)));
// 	      } else {
// 		free.set(w, res_graph.free(dfs)); 
// 	      }
	      
// 	      if (w==t) { 
// 		__augment=true; 
// 		_augment=true;
// 		break; 
// 	      }
// 	    } else {
// 	      res_graph.erase(dfs);
// 	    }
// 	  } 

// 	}

// 	if (__augment) {
// 	  typename EAugGraph::Node n=t;
// 	  Number augment_value=free.get(t);
// 	  while (res_graph.valid(pred.get(n))) { 
// 	    EAugEdge e=pred.get(n);
// 	    res_graph.augment(e, augment_value);
// 	    n=res_graph.source(e);
// 	    if (res_graph.free(e)==0)
// 	      res_graph.erase(e);
// 	  }
// 	}
      
//       }
            
//       return _augment;
//     }

    void run() {
      //int num_of_augmentations=0;
      while (augmentOnShortestPath()) { 
	//while (augmentOnBlockingFlow<MutableGraph>()) { 
	//std::cout << ++num_of_augmentations << " ";
	//std::cout<<std::endl;
      } 
    }

    template<typename MutableGraph> void run() {
      //int num_of_augmentations=0;
      //while (augmentOnShortestPath()) { 
	while (augmentOnBlockingFlow<MutableGraph>()) { 
	//std::cout << ++num_of_augmentations << " ";
	//std::cout<<std::endl;
      } 
    }

    Number flowValue() { 
      Number a=0;
      OutEdgeIt e;
      for(gw.first(e, s); gw.valid(e); gw.next(e)) {
	a+=flow->get(e);
      }
      return a;
    }

  };


//   template <typename Graph, typename Number, typename FlowMap, typename CapacityMap>
//   class MaxMatching {
//   public:
//     typedef typename Graph::Node Node;
//     typedef typename Graph::NodeIt NodeIt;
//     typedef typename Graph::Edge Edge;
//     typedef typename Graph::EdgeIt EdgeIt;
//     typedef typename Graph::OutEdgeIt OutEdgeIt;
//     typedef typename Graph::InEdgeIt InEdgeIt;

//     typedef typename Graph::NodeMap<bool> SMap;
//     typedef typename Graph::NodeMap<bool> TMap;
//   private:
//     const Graph* G;
//     SMap* S;
//     TMap* T;
//     //Node s;
//     //Node t;
//     FlowMap* flow;
//     const CapacityMap* capacity;
//     typedef ResGraphWrapper<Graph, Number, FlowMap, CapacityMap > AugGraph;
//     typedef typename AugGraph::OutEdgeIt AugOutEdgeIt;
//     typedef typename AugGraph::Edge AugEdge;
//     typename Graph::NodeMap<int> used; //0

//   public:
//     MaxMatching(const Graph& _G, SMap& _S, TMap& _T, FlowMap& _flow, const CapacityMap& _capacity) : 
//       G(&_G), S(&_S), T(&_T), flow(&_flow), capacity(&_capacity), used(_G) { }
//     bool augmentOnShortestPath() {
//       AugGraph res_graph(*G, *flow, *capacity);
//       bool _augment=false;
      
//       typedef typename AugGraph::NodeMap<bool> ReachedMap;
//       BfsIterator5< AugGraph, /*AugOutEdgeIt,*/ ReachedMap > bfs(res_graph);
//       typename AugGraph::NodeMap<AugEdge> pred(res_graph); 
//       for(NodeIt s=G->template first<NodeIt>(); G->valid(s); G->next(s)) {
// 	if ((S->get(s)) && (used.get(s)<1) ) {
// 	  //Number u=0;
// 	  //for(OutEdgeIt e=G->template first<OutEdgeIt>(s); G->valid(e); G->next(e))
// 	  //u+=flow->get(e);
// 	  //if (u<1) {
// 	    bfs.pushAndSetReached(s);
// 	    pred.set(s, AugEdge(INVALID));
// 	    //}
// 	}
//       }
      
//       typename AugGraph::NodeMap<Number> free(res_graph);
	
//       Node n;
//       //searching for augmenting path
//       while ( !bfs.finished() ) { 
// 	AugOutEdgeIt e=bfs;
// 	if (res_graph.valid(e) && bfs.isBNodeNewlyReached()) {
// 	  Node v=res_graph.source(e);
// 	  Node w=res_graph.target(e);
// 	  pred.set(w, e);
// 	  if (res_graph.valid(pred.get(v))) {
// 	    free.set(w, std::min(free.get(v), res_graph.free(e)));
// 	  } else {
// 	    free.set(w, res_graph.free(e)); 
// 	  }
// 	  n=res_graph.target(e);
// 	  if (T->get(n) && (used.get(n)<1) ) { 
// 	    //Number u=0;
// 	    //for(InEdgeIt f=G->template first<InEdgeIt>(n); G->valid(f); G->next(f))
// 	    //u+=flow->get(f);
// 	    //if (u<1) {
// 	      _augment=true; 
// 	      break; 
// 	      //}
// 	  }
// 	}
	
// 	++bfs;
//       } //end of searching augmenting path

//       if (_augment) {
// 	//Node n=t;
// 	used.set(n, 1); //mind2 vegen jav
// 	Number augment_value=free.get(n);
// 	while (res_graph.valid(pred.get(n))) { 
// 	  AugEdge e=pred.get(n);
// 	  res_graph.augment(e, augment_value); 
// 	  n=res_graph.source(e);
// 	}
// 	used.set(n, 1); //mind2 vegen jav
//       }

//       return _augment;
//     }

// //     template<typename MutableGraph> bool augmentOnBlockingFlow() {      
// //       bool _augment=false;

// //       AugGraph res_graph(*G, *flow, *capacity);

// //       typedef typename AugGraph::NodeMap<bool> ReachedMap;
// //       BfsIterator4< AugGraph, AugOutEdgeIt, ReachedMap > bfs(res_graph);





// //       //typename AugGraph::NodeMap<AugEdge> pred(res_graph); 
// //       for(NodeIt s=G->template first<NodeIt>(); G->valid(s); G->next(s)) {
// // 	if (S->get(s)) {
// // 	  Number u=0;
// // 	  for(OutEdgeIt e=G->template first<OutEdgeIt>(s); G->valid(e); G->next(e))
// // 	    u+=flow->get(e);
// // 	  if (u<1) {
// // 	    bfs.pushAndSetReached(s);
// // 	    //pred.set(s, AugEdge(INVALID));
// // 	  }
// // 	}
// //       }




// //       //bfs.pushAndSetReached(s);
// //       typename AugGraph::NodeMap<int> dist(res_graph); //filled up with 0's
// //       while ( !bfs.finished() ) { 
// // 	AugOutEdgeIt e=bfs;
// // 	if (res_graph.valid(e) && bfs.isBNodeNewlyReached()) {
// // 	  dist.set(res_graph.target(e), dist.get(res_graph.source(e))+1);
// // 	}
	
// // 	++bfs;
// //       } //computing distances from s in the residual graph

// //       MutableGraph F;
// //       typename AugGraph::NodeMap<typename MutableGraph::Node> 
// // 	res_graph_to_F(res_graph);
// //       for(typename AugGraph::NodeIt n=res_graph.template first<typename AugGraph::NodeIt>(); res_graph.valid(n); res_graph.next(n)) {
// // 	res_graph_to_F.set(n, F.addNode());
// //       }
      
// //       typename MutableGraph::Node sF=res_graph_to_F.get(s);
// //       typename MutableGraph::Node tF=res_graph_to_F.get(t);

// //       typename MutableGraph::EdgeMap<AugEdge> original_edge(F);
// //       typename MutableGraph::EdgeMap<Number> residual_capacity(F);

// //       //Making F to the graph containing the edges of the residual graph 
// //       //which are in some shortest paths
// //       for(typename AugGraph::EdgeIt e=res_graph.template first<typename AugGraph::EdgeIt>(); res_graph.valid(e); res_graph.next(e)) {
// // 	if (dist.get(res_graph.target(e))==dist.get(res_graph.source(e))+1) {
// // 	  typename MutableGraph::Edge f=F.addEdge(res_graph_to_F.get(res_graph.source(e)), res_graph_to_F.get(res_graph.target(e)));
// // 	  original_edge.update();
// // 	  original_edge.set(f, e);
// // 	  residual_capacity.update();
// // 	  residual_capacity.set(f, res_graph.free(e));
// // 	} 
// //       }

// //       bool __augment=true;

// //       while (__augment) {
// // 	__augment=false;
// // 	//computing blocking flow with dfs
// // 	typedef typename MutableGraph::NodeMap<bool> BlockingReachedMap;
// // 	DfsIterator4< MutableGraph, typename MutableGraph::OutEdgeIt, BlockingReachedMap > dfs(F);
// // 	typename MutableGraph::NodeMap<typename MutableGraph::Edge> pred(F);
// // 	pred.set(sF, typename MutableGraph::Edge(INVALID));
// // 	//invalid iterators for sources

// // 	typename MutableGraph::NodeMap<Number> free(F);

// // 	dfs.pushAndSetReached(sF);      
// // 	while (!dfs.finished()) {
// // 	  ++dfs;
// // 	  if (F.valid(typename MutableGraph::OutEdgeIt(dfs))) {
// // 	    if (dfs.isBNodeNewlyReached()) {
// // 	      typename MutableGraph::Node v=F.aNode(dfs);
// // 	      typename MutableGraph::Node w=F.bNode(dfs);
// // 	      pred.set(w, dfs);
// // 	      if (F.valid(pred.get(v))) {
// // 		free.set(w, std::min(free.get(v), residual_capacity.get(dfs)));
// // 	      } else {
// // 		free.set(w, residual_capacity.get(dfs)); 
// // 	      }
// // 	      if (w==tF) { 
// // 		__augment=true; 
// // 		_augment=true;
// // 		break; 
// // 	      }
	      
// // 	    } else {
// // 	      F.erase(typename MutableGraph::OutEdgeIt(dfs));
// // 	    }
// // 	  } 
// // 	}

// // 	if (__augment) {
// // 	  typename MutableGraph::Node n=tF;
// // 	  Number augment_value=free.get(tF);
// // 	  while (F.valid(pred.get(n))) { 
// // 	    typename MutableGraph::Edge e=pred.get(n);
// // 	    res_graph.augment(original_edge.get(e), augment_value); 
// // 	    n=F.source(e);
// // 	    if (residual_capacity.get(e)==augment_value) 
// // 	      F.erase(e); 
// // 	    else 
// // 	      residual_capacity.set(e, residual_capacity.get(e)-augment_value);
// // 	  }
// // 	}
	
// //       }
            
// //       return _augment;
// //     }
//     bool augmentOnBlockingFlow2() {
//       bool _augment=false;

//       //typedef ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap> EAugGraph;
//       typedef FilterGraphWrapper< ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap> > EAugGraph;
//       typedef typename EAugGraph::OutEdgeIt EAugOutEdgeIt;
//       typedef typename EAugGraph::Edge EAugEdge;

//       EAugGraph res_graph(*G, *flow, *capacity);

//       //typedef typename EAugGraph::NodeMap<bool> ReachedMap;
//       BfsIterator5< 
// 	ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>, 
// 	/*typename ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>::OutEdgeIt,*/ 
// 	ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>::NodeMap<bool> > bfs(res_graph);


//       //typename AugGraph::NodeMap<AugEdge> pred(res_graph); 
//       for(NodeIt s=G->template first<NodeIt>(); G->valid(s); G->next(s)) {
// 	if (S->get(s)) {
// 	  Number u=0;
// 	  for(OutEdgeIt e=G->template first<OutEdgeIt>(s); G->valid(e); G->next(e))
// 	    u+=flow->get(e);
// 	  if (u<1) {
// 	    bfs.pushAndSetReached(s);
// 	    //pred.set(s, AugEdge(INVALID));
// 	  }
// 	}
//       }

      
//       //bfs.pushAndSetReached(s);

//       typename ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>::
// 	NodeMap<int>& dist=res_graph.dist;

//       while ( !bfs.finished() ) {
// 	typename ErasingResGraphWrapper<Graph, Number, FlowMap, CapacityMap>::OutEdgeIt e=bfs;
// 	if (res_graph.valid(e) && bfs.isBNodeNewlyReached()) {
// 	  dist.set(res_graph.target(e), dist.get(res_graph.source(e))+1);
// 	}
// 	++bfs;	
//       } //computing distances from s in the residual graph

//       bool __augment=true;

//       while (__augment) {

// 	__augment=false;
// 	//computing blocking flow with dfs
// 	typedef typename EAugGraph::NodeMap<bool> BlockingReachedMap;
// 	DfsIterator5< EAugGraph/*, EAugOutEdgeIt*/, BlockingReachedMap > 
// 	  dfs(res_graph);
// 	typename EAugGraph::NodeMap<EAugEdge> pred(res_graph, INVALID); 
// 	//pred.set(s, EAugEdge(INVALID));
// 	//invalid iterators for sources

// 	typename EAugGraph::NodeMap<Number> free(res_graph);


// 	//typename AugGraph::NodeMap<AugEdge> pred(res_graph); 
//       for(NodeIt s=G->template first<NodeIt>(); G->valid(s); G->next(s)) {
// 	if (S->get(s)) {
// 	  Number u=0;
// 	  for(OutEdgeIt e=G->template first<OutEdgeIt>(s); G->valid(e); G->next(e))
// 	    u+=flow->get(e);
// 	  if (u<1) {
// 	    dfs.pushAndSetReached(s);
// 	    //pred.set(s, AugEdge(INVALID));
// 	  }
// 	}
//       }



//       //dfs.pushAndSetReached(s);
//       typename EAugGraph::Node n;
// 	while (!dfs.finished()) {
// 	  ++dfs;
// 	  if (res_graph.valid(EAugOutEdgeIt(dfs))) { 
// 	    if (dfs.isBNodeNewlyReached()) {
	  
// 	      typename EAugGraph::Node v=res_graph.aNode(dfs);
// 	      typename EAugGraph::Node w=res_graph.bNode(dfs);

// 	      pred.set(w, EAugOutEdgeIt(dfs));
// 	      if (res_graph.valid(pred.get(v))) {
// 		free.set(w, std::min(free.get(v), res_graph.free(dfs)));
// 	      } else {
// 		free.set(w, res_graph.free(dfs)); 
// 	      }
	     
// 	      n=w;
// 	      if (T->get(w)) {
// 		Number u=0;
// 		for(InEdgeIt f=G->template first<InEdgeIt>(n); G->valid(f); G->next(f))
// 		  u+=flow->get(f);
// 		if (u<1) {
// 		  __augment=true; 
// 		  _augment=true;
// 		  break; 
// 		}
// 	      }
// 	    } else {
// 	      res_graph.erase(dfs);
// 	    }
// 	  } 

// 	}

// 	if (__augment) {
// 	  // typename EAugGraph::Node n=t;
// 	  Number augment_value=free.get(n);
// 	  while (res_graph.valid(pred.get(n))) { 
// 	    EAugEdge e=pred.get(n);
// 	    res_graph.augment(e, augment_value);
// 	    n=res_graph.source(e);
// 	    if (res_graph.free(e)==0)
// 	      res_graph.erase(e);
// 	  }
// 	}
      
//       }
            
//       return _augment;
//     }
//     void run() {
//       //int num_of_augmentations=0;
//       while (augmentOnShortestPath()) { 
// 	//while (augmentOnBlockingFlow<MutableGraph>()) { 
// 	//std::cout << ++num_of_augmentations << " ";
// 	//std::cout<<std::endl;
//       } 
//     }
// //     template<typename MutableGraph> void run() {
// //       //int num_of_augmentations=0;
// //       //while (augmentOnShortestPath()) { 
// // 	while (augmentOnBlockingFlow<MutableGraph>()) { 
// // 	//std::cout << ++num_of_augmentations << " ";
// // 	//std::cout<<std::endl;
// //       } 
// //     } 
//     Number flowValue() { 
//       Number a=0;
//       EdgeIt e;
//       for(G->/*getF*/first(e); G->valid(e); G->next(e)) {
// 	a+=flow->get(e);
//       }
//       return a;
//     }
//   };





  
// //   template <typename Graph, typename Number, typename FlowMap, typename CapacityMap>
// //   class MaxFlow2 {
// //   public:
// //     typedef typename Graph::Node Node;
// //     typedef typename Graph::Edge Edge;
// //     typedef typename Graph::EdgeIt EdgeIt;
// //     typedef typename Graph::OutEdgeIt OutEdgeIt;
// //     typedef typename Graph::InEdgeIt InEdgeIt;
// //   private:
// //     const Graph& G;
// //     std::list<Node>& S;
// //     std::list<Node>& T;
// //     FlowMap& flow;
// //     const CapacityMap& capacity;
// //     typedef ResGraphWrapper<Graph, Number, FlowMap, CapacityMap > AugGraph;
// //     typedef typename AugGraph::OutEdgeIt AugOutEdgeIt;
// //     typedef typename AugGraph::Edge AugEdge;
// //     typename Graph::NodeMap<bool> SMap;
// //     typename Graph::NodeMap<bool> TMap;
// //   public:
// //     MaxFlow2(const Graph& _G, std::list<Node>& _S, std::list<Node>& _T, FlowMap& _flow, const CapacityMap& _capacity) : G(_G), S(_S), T(_T), flow(_flow), capacity(_capacity), SMap(_G), TMap(_G) { 
// //       for(typename std::list<Node>::const_iterator i=S.begin(); 
// // 	  i!=S.end(); ++i) { 
// // 	SMap.set(*i, true); 
// //       }
// //       for (typename std::list<Node>::const_iterator i=T.begin(); 
// // 	   i!=T.end(); ++i) { 
// // 	TMap.set(*i, true); 
// //       }
// //     }
// //     bool augment() {
// //       AugGraph res_graph(G, flow, capacity);
// //       bool _augment=false;
// //       Node reached_t_node;
      
// //       typedef typename AugGraph::NodeMap<bool> ReachedMap;
// //       BfsIterator4< AugGraph, AugOutEdgeIt, ReachedMap > bfs(res_graph);
// //       for(typename std::list<Node>::const_iterator i=S.begin(); 
// // 	  i!=S.end(); ++i) {
// // 	bfs.pushAndSetReached(*i);
// //       }
// //       //bfs.pushAndSetReached(s);
	
// //       typename AugGraph::NodeMap<AugEdge> pred(res_graph); 
// //       //filled up with invalid iterators
      
// //       typename AugGraph::NodeMap<Number> free(res_graph);
	
// //       //searching for augmenting path
// //       while ( !bfs.finished() ) { 
// // 	AugOutEdgeIt e=/*AugOutEdgeIt*/(bfs);
// // 	if (e.valid() && bfs.isBNodeNewlyReached()) {
// // 	  Node v=res_graph.source(e);
// // 	  Node w=res_graph.target(e);
// // 	  pred.set(w, e);
// // 	  if (pred.get(v).valid()) {
// // 	    free.set(w, std::min(free.get(v), e.free()));
// // 	  } else {
// // 	    free.set(w, e.free()); 
// // 	  }
// // 	  if (TMap.get(res_graph.target(e))) { 
// // 	    _augment=true; 
// // 	    reached_t_node=res_graph.target(e);
// // 	    break; 
// // 	  }
// // 	}
	
// // 	++bfs;
// //       } //end of searching augmenting path

// //       if (_augment) {
// // 	Node n=reached_t_node;
// // 	Number augment_value=free.get(reached_t_node);
// // 	while (pred.get(n).valid()) { 
// // 	  AugEdge e=pred.get(n);
// // 	  e.augment(augment_value); 
// // 	  n=res_graph.source(e);
// // 	}
// //       }

// //       return _augment;
// //     }
// //     void run() {
// //       while (augment()) { } 
// //     }
// //     Number flowValue() { 
// //       Number a=0;
// //       for(typename std::list<Node>::const_iterator i=S.begin(); 
// // 	  i!=S.end(); ++i) { 
// // 	for(OutEdgeIt e=G.template first<OutEdgeIt>(*i); e.valid(); ++e) {
// // 	  a+=flow.get(e);
// // 	}
// // 	for(InEdgeIt e=G.template first<InEdgeIt>(*i); e.valid(); ++e) {
// // 	  a-=flow.get(e);
// // 	}
// //       }
// //       return a;
// //     }
// //   };


} // namespace lemon

#endif //LEMON_EDMONDS_KARP_H
