#ifndef BIN_HEAP_HH
#define BIN_HEAP_HH

#include <vector>
#include <utility>
#include <functional>

namespace marci {

  template <typename Key, typename Val, typename KeyIntMap,
	    typename Compare = std::less<Val> >
  class BinHeap {

  public:
    typedef Key	             KeyType;
    // FIXME: stl-ben nem ezt hivjak value_type -nak, hanem a kovetkezot...
    typedef Val              ValueType;
    typedef std::pair<KeyType,ValueType>     PairType;
    typedef KeyIntMap        KeyIntMapType;
    typedef Compare          ValueCompare;

    /**
     * Each Key element have a state associated to it. It may be "in heap",
     * "pre heap" or "post heap". The later two are indifferent from the
     * heap's point of view, but may be useful to the user.
     *
     * The KeyIntMap _should_ be initialized in such way, that it maps
     * PRE_HEAP (-1) to any element to be put in the heap...
     */

    enum state {
      IN_HEAP = 0,
      PRE_HEAP = -1,
      POST_HEAP = -2
    };

  private:
    std::vector<PairType> data;
    Compare comp;
    // FIXME: jo ez igy???
    KeyIntMap &kim;

  public:
    BinHeap(KeyIntMap &_kim) : kim(_kim) {}
    BinHeap(KeyIntMap &_kim, const Compare &_comp) : comp(_comp), kim(_kim) {}


    int size() const { return data.size(); }
    bool empty() const { return size() == 0; }

  private:
    static int parent(int i) { return (i-1)/2; }
    static int second_child(int i) { return 2*i+2; }
    bool less(const PairType &p1, const PairType &p2) {
      return comp(p1.second, p2.second);
    }

    int bubble_up(int hole, PairType p);
    int bubble_down(int hole, PairType p, int length);

    void move(const PairType &p, int i) {
      data[i] = p;
      kim.put(p.first, i);
    }

  public:
    void push(const PairType &p) {
      int n = data.size();
      data.resize(n+1);
      bubble_up(n, p);
    }
    void push(const Key &k, const Val &v) { push(PairType(k,v)); }

    Key top() const {
      // FIXME: test size>0 ?
      return data[0].first;
    }
    Val topValue() const {
      // FIXME: test size>0 ?
      return data[0].second;
    }

    void pop() {
      int n = data.size()-1;
      if ( n>0 ) {
	bubble_down(0, data[n], n);
      }
      if ( n>=0 ) {
	data.pop_back();
      }
    }

    const Val get(const Key &k) const {
      int idx = kim.get(k);
      return data[idx].second;
    }
    void put(const Key &k, const Val &v) {
      int idx = kim.get(k);
      if( idx < 0 ) {
	push(k,v);
      }
      else if( comp(v, data[idx].second) ) {
	bubble_up(idx, PairType(k,v));
      }
      else {
	bubble_down(idx, PairType(k,v), data.size());
      }
    }

    void decrease(const Key &k, const Val &v) {
      int idx = kim.get(k);
      bubble_up(idx, PairType(k,v));
    }
    void increase(const Key &k, const Val &v) {
      int idx = kim.get(k);
      bubble_down(idx, PairType(k,v), data.size());
    }

  }; // class BinHeap

  
  template <typename K, typename V, typename M, typename C>
  int BinHeap<K,V,M,C>::bubble_up(int hole, PairType p) {
    int par = parent(hole);
    while( hole>0 && less(p,data[par]) ) {
      move(data[par],hole);
      hole = par;
      par = parent(hole);
    }
    move(p, hole);
    return hole;
  }

  template <typename K, typename V, typename M, typename C>
  int BinHeap<K,V,M,C>::bubble_down(int hole, PairType p, int length) {
    int child = second_child(hole);
    while(child < length) {
      if( less(data[child-1], data[child]) ) {
	--child;
      }
      if( !less(data[child], p) )
	goto ok;
      move(data[child], hole);
      hole = child;
      child = second_child(hole);
    }
    child--;
    if( child<length && less(data[child], p) ) {
      move(data[child], hole);
      hole=child;
    }
  ok:
    move(p, hole);
    return hole;
  }

} // namespace marci

#endif // BIN_HEAP_HH
