// -*- c++ -*-
#ifndef VECTOR_MAP_H
#define VECTOR_MAP_H

#include <vector>

#include <hugo/map_iterator.h>
#include <hugo/map_bits.h>

///\ingroup graphmaps
///\file
///\brief Vector based graph maps.

namespace hugo {
  
  /// \addtogroup graphmaps
  /// @{
  
  /** The ArrayMap template class is graph map structure what
   *  automatically updates the map when a key is added to or erased from
   *  the map. This map factory uses the allocators to implement 
   *  the container functionality. This map factory
   *  uses the std::vector to implement the container function.
   *
   *  The template parameter is the MapRegistry that the maps
   *  will belong to and the ValueType.
   * 
   * \todo It should use a faster initialization using the maxNodeId() or
   * maxEdgeId() function of the graph instead of iterating through each
   * edge/node.
   */
	
  template <typename MapRegistry, typename Value>
  class VectorMap : public MapRegistry::MapBase {
    template <typename MR, typename T> friend class VectorMap;
  public:
		
    /// The graph type of the maps. 
    typedef typename MapRegistry::Graph Graph;
    /// The key type of the maps.
    typedef typename MapRegistry::KeyType KeyType;
    /// The iterator to iterate on the keys.
    typedef typename MapRegistry::KeyIt KeyIt;

    /// The map type.
    typedef VectorMap Map;
    /// The MapBase of the Map which implements the core regisitry function.
    typedef typename MapRegistry::MapBase MapBase;

  private:
		
    /// The container type of the map.
    typedef std::vector<Value> Container;	

  public:


    /// The value type of the map.
    typedef Value ValueType;
    /// The reference type of the map;
    typedef typename Container::reference ReferenceType;
    /// The pointer type of the map;
    typedef typename Container::pointer PointerType;

    /// The const value type of the map.
    typedef const Value ConstValueType;
    /// The const reference type of the map;
    typedef typename Container::const_reference ConstReferenceType;
    /// The pointer type of the map;
    typedef typename Container::const_pointer ConstPointerType;

    /** Graph and Registry initialized map constructor.
     */
    VectorMap(const Graph& g, MapRegistry& r) 
      : MapBase(g, r), container(KeyInfo<Graph, KeyIt>::maxId(g)+1) {}

    /** Constructor to use default value to initialize the map. 
     */
    VectorMap(const Graph& g, MapRegistry& r, const Value& v) 
      : MapBase(g, r), container(KeyInfo<Graph, KeyIt>::maxId(g)+1, v) {}

    /** Assign operator to copy a map of an other map type.
     */
    template <typename TT>
    VectorMap(const VectorMap<MapRegistry, TT>& c) 
      : MapBase(c), container(c.container.size()) {
      for (KeyIt it(*MapBase::getGraph()); it != INVALID; ++it) {
	int id = KeyInfo<Graph, KeyIt>::id(*MapBase::getGraph(), it);
	container[id] = c.container[id];
      }
    }

    /** Assign operator to copy a map of an other map type.
     */
    template <typename TT>
    VectorMap& operator=(const VectorMap<MapRegistry, TT>& c) {
      if (MapBase::getGraph() != c.getGraph()) {
	MapBase::operator=(c);
	container.resize(c.container.size());
      }
      for (KeyIt it(*MapBase::getGraph()); it != INVALID; ++it) {
	int id = KeyInfo<Graph, KeyIt>::id(*MapBase::getGraph(), it);
	container[id] = c.container[id];
      }
      return *this;
    }
    /**
     * The subscript operator. The map can be subscripted by the
     * actual keys of the graph. 
     */
    ReferenceType operator[](const KeyType& key) {
      int id = KeyInfo<Graph, KeyIt>::id(*MapBase::getGraph(), key);
      return container[id];
    } 
		
    /**
     * The const subscript operator. The map can be subscripted by the
     * actual keys of the graph. 
     */
    ConstReferenceType operator[](const KeyType& key) const {
      int id = KeyInfo<Graph, KeyIt>::id(*MapBase::getGraph(), key);
      return container[id];
    }

    /** Setter function of the map. Equivalent with map[key] = val.
     *  This is a compatibility feature with the not dereferable maps.
     */
    void set(const KeyType& key, const ValueType& val) {
      int id = KeyInfo<Graph, KeyIt>::id(*MapBase::getGraph(), key);
      container[id] = val;
    }
		
    /** Add a new key to the map. It called by the map registry.
     */
    void add(const KeyType& key) {
      int id = KeyInfo<Graph, KeyIt>::id(*MapBase::getGraph(), key);
      if (id >= (int)container.size()) {
	container.resize(id + 1);
      }
    }
		
    /** Erase a key from the map. It called by the map registry.
     */
    void erase(const KeyType& key) {}

    /** Clear the data structure.
     */
    void clear() { 
      container.clear();
    }

    /// The stl compatible pair iterator of the map.
    typedef MapIterator<VectorMap> Iterator;
    /// The stl compatible const pair iterator of the map.
    typedef MapConstIterator<VectorMap> ConstIterator;

    /** Returns the begin iterator of the map.
     */
    Iterator begin() {
      return Iterator(*this, KeyIt(*MapBase::getGraph()));
    }

    /** Returns the end iterator of the map.
     */
    Iterator end() {
      return Iterator(*this, INVALID);
    }

    /** Returns the begin ConstIterator of the map.
     */
    ConstIterator begin() const {
      return ConstIterator(*this, KeyIt(*MapBase::getGraph()));
    }

    /** Returns the end const_iterator of the map.
     */
    ConstIterator end() const {
      return ConstIterator(*this, INVALID);
    }

    /// The KeySet of the Map.
    typedef MapConstKeySet<VectorMap> ConstKeySet;

    /// KeySet getter function.
    ConstKeySet keySet() const {
      return ConstKeySet(*this); 
    }

    /// The ConstValueSet of the Map.
    typedef MapConstValueSet<VectorMap> ConstValueSet;

    /// ConstValueSet getter function.
    ConstValueSet valueSet() const {
      return ConstValueSet(*this);
    }

    /// The ValueSet of the Map.
    typedef MapValueSet<VectorMap> ValueSet;

    /// ValueSet getter function.
    ValueSet valueSet() {
      return ValueSet(*this);
    }


  private:
		
    Container container;

  public:
    // STL  compatibility typedefs.
    typedef Iterator iterator;
    typedef ConstIterator const_iterator;
    typedef typename Iterator::PairValueType value_type;
    typedef typename Iterator::KeyType key_type;
    typedef typename Iterator::ValueType data_type;
    typedef typename Iterator::PairReferenceType reference;
    typedef typename Iterator::PairPointerType pointer;
    typedef typename ConstIterator::PairReferenceType const_reference;
    typedef typename ConstIterator::PairPointerType const_pointer;
    typedef int difference_type;		
  };
  
  /// @}
  
}

#endif
