/* -*- C++ -*-
 * src/test/dijkstra_test.cc - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2005 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include "test_tools.h"
#include <lemon/smart_graph.h>
#include <lemon/dijkstra.h>
#include <lemon/path.h>
#include <lemon/maps.h>
#include <lemon/concept/graph.h>
#include <lemon/concept/maps.h>
using namespace lemon;

const int PET_SIZE =5;


void check_Dijkstra_BinHeap_Compile() 
{
  typedef int VType;
  typedef concept::StaticGraph Graph;

  typedef Graph::Edge Edge;
  typedef Graph::Node Node;
  typedef Graph::EdgeIt EdgeIt;
  typedef Graph::NodeIt NodeIt;
  typedef concept::ReadMap<Edge,VType> LengthMap;
 
  typedef Dijkstra<Graph, LengthMap> DType;
  
  Graph G;
  Node n;
  Edge e;
  VType l;
  bool b;
  DType::DistMap d(G);
  DType::PredMap p(G);
  //  DType::PredNodeMap pn(G);
  LengthMap cap;

  DType dijkstra_test(G,cap);

  dijkstra_test.run(n);

  l  = dijkstra_test.dist(n);
  e  = dijkstra_test.pred(n);
  n  = dijkstra_test.predNode(n);
  d  = dijkstra_test.distMap();
  p  = dijkstra_test.predMap();
  //  pn = dijkstra_test.predNodeMap();
  b  = dijkstra_test.reached(n);

  DirPath<Graph> pp(G);
  dijkstra_test.getPath(pp,n);
}

void check_Dijkstra_Function_Compile() 
{
  typedef int VType;
  typedef concept::StaticGraph Graph;

  typedef Graph::Edge Edge;
  typedef Graph::Node Node;
  typedef Graph::EdgeIt EdgeIt;
  typedef Graph::NodeIt NodeIt;
  typedef concept::ReadMap<Edge,VType> LengthMap;
   
  dijkstra(Graph(),LengthMap(),Node()).run();
  dijkstra(Graph(),LengthMap()).source(Node()).run();
  dijkstra(Graph(),LengthMap())
    .predMap(concept::WriteMap<Node,Edge>())
    .distMap(concept::WriteMap<Node,VType>())
    .run(Node());
  
}


int main()
{
    
  typedef SmartGraph Graph;

  typedef Graph::Edge Edge;
  typedef Graph::Node Node;
  typedef Graph::EdgeIt EdgeIt;
  typedef Graph::NodeIt NodeIt;
  typedef Graph::EdgeMap<int> LengthMap;

  Graph G;
  Node s, t;
  LengthMap cap(G);
  PetStruct<Graph> ps = addPetersen(G,PET_SIZE);
   
  for(int i=0;i<PET_SIZE;i++) {
    cap[ps.outcir[i]]=4;
    cap[ps.incir[i]]=1;
    cap[ps.chords[i]]=10;
  }
  s=ps.outer[0];
  t=ps.inner[1];
  
  Dijkstra<Graph, LengthMap> 
	dijkstra_test(G, cap);
  dijkstra_test.run(s);
  
  check(dijkstra_test.dist(t)==13,"Dijkstra found a wrong path.");


  DirPath<Graph> p(G);
  check(dijkstra_test.getPath(p,t),"getPath() failed to set the path.");
  check(p.length()==4,"getPath() found a wrong path.");
  

  for(EdgeIt e(G); e!=INVALID; ++e) {
    Node u=G.source(e);
    Node v=G.target(e);
    check( !dijkstra_test.reached(u) ||
	   (dijkstra_test.dist(v) - dijkstra_test.dist(u) <= cap[e]),
	   "dist(target)-dist(source)- edge_length= " 
	   << dijkstra_test.dist(v) - dijkstra_test.dist(u) 
	   - cap[e]);
  }

  ///\bug This works only for integer lengths
  for(NodeIt v(G); v!=INVALID; ++v){
    check(dijkstra_test.reached(v),"Each node should be reached.");
    if ( dijkstra_test.pred(v)!=INVALID ) {
      Edge e=dijkstra_test.pred(v);
      Node u=G.source(e);
      check(u==dijkstra_test.predNode(v),"Wrong tree.");
      check(dijkstra_test.dist(v) - dijkstra_test.dist(u) == cap[e],
	    "Wrong distance! Difference: " 
	    << std::abs(dijkstra_test.dist(v) - dijkstra_test.dist(u) 
			    - cap[e]));
    }
  }

  
  {
    NullMap<Node,Edge> myPredMap;
    dijkstra(G,cap).predMap(myPredMap).run(s);
  }
  return 0;
}
