// -*- C++ -*-
/*
preflow_hl2.h
by jacint. 
Runs the highest label variant of the preflow push algorithm with 
running time O(n^2\sqrt(m)). 

Heuristics:

  gap: we iterate through the nodes for finding the nodes above 
       the gap and under level n. So it is quite slow.
  numb: we maintain the number of nodes in level i.
  highest label
  
'A' is a parameter for the gap, we only upgrade the nodes to level n,
  if the gap is under A*n.

The constructor runs the algorithm.

Members:

T maxFlow() : returns the value of a maximum flow

T flowOnEdge(EdgeIt e) : for a fixed maximum flow x it returns x(e) 

FlowMap Flow() : returns the fixed maximum flow x

void minCut(CutMap& M) : sets M to the characteristic vector of a 
     minimum cut. M should be a map of bools initialized to false.

void minMinCut(CutMap& M) : sets M to the characteristic vector of the 
     minimum min cut. M should be a map of bools initialized to false.

void maxMinCut(CutMap& M) : sets M to the characteristic vector of the 
     maximum min cut. M should be a map of bools initialized to false.

*/

#ifndef PREFLOW_HL2_H
#define PREFLOW_HL2_H

#define A .9

#include <vector>
#include <stack>
#include <queue>

namespace hugo {

  template <typename Graph, typename T, 
    typename FlowMap=typename Graph::EdgeMap<T>, 
    typename CapMap=typename Graph::EdgeMap<T> >
  class preflow_hl2 {
    
    typedef typename Graph::NodeIt NodeIt;
    typedef typename Graph::EdgeIt EdgeIt;
    typedef typename Graph::EachNodeIt EachNodeIt;
    typedef typename Graph::OutEdgeIt OutEdgeIt;
    typedef typename Graph::InEdgeIt InEdgeIt;
    
    Graph& G;
    NodeIt s;
    NodeIt t;
    FlowMap flow;
    CapMap& capacity;  
    T value;
    
  public:

    preflow_hl2(Graph& _G, NodeIt _s, NodeIt _t, CapMap& _capacity) :
      G(_G), s(_s), t(_t), flow(_G), capacity(_capacity) { 

      int n=G.nodeNum(); 
      int b=n-2; 
      /*
	b is a bound on the highest level of an active node. 
      */

      typename Graph::NodeMap<int> level(G,n);      
      typename Graph::NodeMap<T> excess(G); 

      std::vector<int> numb(n);    
      /*
	The number of nodes on level i < n. It is
	initialized to n+1, because of the reverse_bfs-part.
      */

      std::vector<std::stack<NodeIt> > stack(2*n-1);    
      //Stack of the active nodes in level i.


      /*Reverse_bfs from t, to find the starting level.*/
      level.set(t,0);
      std::queue<NodeIt> bfs_queue;
      bfs_queue.push(t);

      while (!bfs_queue.empty()) {

	NodeIt v=bfs_queue.front();	
	bfs_queue.pop();
	int l=level.get(v)+1;

	for(InEdgeIt e=G.template first<InEdgeIt>(v); e.valid(); ++e) {
	  NodeIt w=G.tail(e);
	  if ( level.get(w) == n ) {
	    bfs_queue.push(w);
	    ++numb[l];
	    level.set(w, l);
	  }
	}
      }
	
      level.set(s,n);



      /* Starting flow. It is everywhere 0 at the moment. */     
      for(OutEdgeIt e=G.template first<OutEdgeIt>(s); e.valid(); ++e) 
	{
	  T c=capacity.get(e);
	  if ( c == 0 ) continue;
	  NodeIt w=G.head(e);
	  if ( w!=s ) {	  
	    if ( excess.get(w) == 0 && w!=t ) stack[level.get(w)].push(w); 
	    flow.set(e, c); 
	    excess.set(w, excess.get(w)+c);
	  }
	}
      
      /* 
	 End of preprocessing 
      */


      /*
	Push/relabel on the highest level active nodes.
      */
      /*While there exists an active node.*/
      while (b) { 
	if ( stack[b].empty() ) { 
	  --b;
	  continue;
	} 
	
	NodeIt w=stack[b].top();   
	stack[b].pop();           
	int lev=level.get(w);
	T exc=excess.get(w);
	int newlevel=2*n;      //In newlevel we bound the next level of w.
	
	  for(OutEdgeIt e=G.template first<OutEdgeIt>(w); e.valid(); ++e) {
	    
	    if ( flow.get(e) == capacity.get(e) ) continue; 
	    NodeIt v=G.head(e);            
	    //e=wv	    
	    
	    if( lev > level.get(v) ) {      
	      /*Push is allowed now*/
	      
	      if ( excess.get(v)==0 && v != s && v !=t ) 
		stack[level.get(v)].push(v); 
	      /*v becomes active.*/
	      
	      T cap=capacity.get(e);
	      T flo=flow.get(e);
	      T remcap=cap-flo;
	      
	      if ( remcap >= exc ) {       
		/*A nonsaturating push.*/
		
		flow.set(e, flo+exc);
		excess.set(v, excess.get(v)+exc);
		exc=0;
		break; 
		
	      } else { 
		/*A saturating push.*/
		
		flow.set(e, cap );
		excess.set(v, excess.get(v)+remcap);
		exc-=remcap;
	      }
	    } else if ( newlevel > level.get(v) ) newlevel = level.get(v);
	    
	  } //for out edges wv 
	
	
	if ( exc > 0 ) {	
	  for( InEdgeIt e=G.template first<InEdgeIt>(w); e.valid(); ++e) {
	    
	    if( flow.get(e) == 0 ) continue; 
	    NodeIt v=G.tail(e);  
	    //e=vw
	    
	    if( lev > level.get(v) ) {  
	      /*Push is allowed now*/
	      
	      if ( excess.get(v)==0 && v != s && v !=t) 
		stack[level.get(v)].push(v); 
	      /*v becomes active.*/
	      
	      T flo=flow.get(e);
	      
	      if ( flo >= exc ) { 
		/*A nonsaturating push.*/
		
		flow.set(e, flo-exc);
		excess.set(v, excess.get(v)+exc);
		exc=0;
		break; 
	      } else {                                               
		/*A saturating push.*/
		
		excess.set(v, excess.get(v)+flo);
		exc-=flo;
		flow.set(e,0);
	      }  
	    } else if ( newlevel > level.get(v) ) newlevel = level.get(v);
	    
	  } //for in edges vw
	  
	} // if w still has excess after the out edge for cycle
	
	excess.set(w, exc);
	

	

	/*
	  Relabel
	*/
	
	if ( exc > 0 ) {
	  //now 'lev' is the old level of w
	  level.set(w,++newlevel);
	  
	  if ( lev < n ) {
	    --numb[lev];
	    
	    if ( !numb[lev] && lev < A*n ) {  //If the level of w gets empty. 
	      
	      for (EachNodeIt v=G.template first<EachNodeIt>(); v.valid() ; ++v) {
		if (level.get(v) > lev && level.get(v) < n ) level.set(v,n);  
	      }
	      for (int i=lev+1 ; i!=n ; ++i) numb[i]=0; 
	      if ( newlevel < n ) newlevel=n; 
	    } else { 
	      if ( newlevel < n ) ++numb[newlevel]; 
	    }
	  } 
	  
	  stack[newlevel].push(w);
	  b=newlevel;
	  
	}
	
      } // while(b)
      
      
      value = excess.get(t);
      /*Max flow value.*/


    } //void run()





    /*
      Returns the maximum value of a flow.
     */

    T maxFlow() {
      return value;
    }



    /*
      For the maximum flow x found by the algorithm, 
      it returns the flow value on edge e, i.e. x(e). 
    */

    T flowOnEdge(const EdgeIt e) {
      return flow.get(e);
    }



    /*
      Returns the maximum flow x found by the algorithm.
    */

    FlowMap Flow() {
      return flow;
    }




    /*
      Returns the minimum min cut, by a bfs from s in the residual graph.
    */
    
    template<typename CutMap>
    void minCut(CutMap& M) {
    
      std::queue<NodeIt> queue;
      
      M.set(s,true);      
      queue.push(s);

      while (!queue.empty()) {
        NodeIt w=queue.front();
	queue.pop();
	
	for(OutEdgeIt e=G.template first<OutEdgeIt>(w) ; e.valid(); ++e) {
	  NodeIt v=G.head(e);
	  if (!M.get(v) && flow.get(e) < capacity.get(e) ) {
	    queue.push(v);
	    M.set(v, true);
	  }
	} 

	for(InEdgeIt e=G.template first<InEdgeIt>(w) ; e.valid(); ++e) {
	  NodeIt v=G.tail(e);
	  if (!M.get(v) && flow.get(e) > 0 ) {
	    queue.push(v);
	    M.set(v, true);
	  }
	}

      }
    }



    /*
      Returns the maximum min cut, by a reverse bfs 
      from t in the residual graph.
    */
    
    template<typename CutMap>
    void maxMinCut(CutMap& M) {
    
      std::queue<NodeIt> queue;
      
      M.set(t,true);        
      queue.push(t);

      while (!queue.empty()) {
        NodeIt w=queue.front();
	queue.pop();

	for(InEdgeIt e=G.template first<InEdgeIt>(w) ; e.valid(); ++e) {
	  NodeIt v=G.tail(e);
	  if (!M.get(v) && flow.get(e) < capacity.get(e) ) {
	    queue.push(v);
	    M.set(v, true);
	  }
	}

	for(OutEdgeIt e=G.template first<OutEdgeIt>(w) ; e.valid(); ++e) {
	  NodeIt v=G.head(e);
	  if (!M.get(v) && flow.get(e) > 0 ) {
	    queue.push(v);
	    M.set(v, true);
	  }
	}
      }

      for(EachNodeIt v=G.template first<EachNodeIt>() ; v.valid(); ++v) {
	M.set(v, !M.get(v));
      }

    }



    template<typename CutMap>
    void minMinCut(CutMap& M) {
      minCut(M);
    }



  };
}//namespace marci
#endif 




