///\file
///\brief Implementation of the LEMON-CBC mip solver interface.

#include <lemon/mip_cbc.h>

//#include "CoinBuild.hpp"
//#include "Coinmodel->hpp"


// For Branch and bound
#include "OsiSolverInterface.hpp"
//#include "CbcBranchUser.hpp"
//#include "CbcCompareUser.hpp"
#include "CbcCutGenerator.hpp"
#include "CbcHeuristicLocal.hpp"
#include "CbcHeuristicGreedy.hpp"
#include "CbcHeuristicFPump.hpp"
#include "CbcHeuristicRINS.hpp"

// Cuts

#include "CglGomory.hpp"
#include "CglProbing.hpp"
#include "CglKnapsackCover.hpp"
#include "CglOddHole.hpp"
#include "CglClique.hpp"
#include "CglFlowCover.hpp"
#include "CglMixedIntegerRounding.hpp"

// Heuristics

#include "CbcHeuristic.hpp"

namespace lemon {
  
  MipCbc::MipCbc() : LpCbc() {
	model=NULL;
  }

  MipCbc::~MipCbc() {
	delete model;
  }

  void MipCbc::_colType(int i, MipCbc::ColTypes col_type){
    switch (col_type){
      case INT:
		build.setColIsInteger(i,true);
		break;
      case REAL:
		build.setColIsInteger(i,false);
		break;
    default:;
        //FIXME problem
    }
  }
  
  MipCbc::ColTypes MipCbc::_colType(int i) const {
	if (build.getColumnIsInteger(i))
		return INT;//Or binary
	return REAL;
  }
  
  LpCbc::SolveExitStatus MipCbc::_solve() {
	//if (delete_solver) delete solver; // nem a legszebb megoldas...
	if (!solver){
		#ifdef COIN_HAS_CLP
		  solver=new OsiClpSolverInterface();
		#elif COIN_HAS_OSL
		  solver=new OsiOslSolverInterface();
		#endif
	}
	solver->loadFromCoinModel(build);
 	solver->writeLp("problem_cbc");
	if (model==NULL) {
		model= new CbcModel(*solver);
	}
	model->setLogLevel(loglevel);
	// Do initial solve to continuous
	
	model->initialSolve();
	model->solver()->setHintParam(OsiDoReducePrint,true,OsiHintTry);
	
	if (!model->isInitialSolveAbandoned() &&
		model->isInitialSolveProvenOptimal() &&
		!model->isInitialSolveProvenPrimalInfeasible() &&
		!model->isInitialSolveProvenDualInfeasible())
	{

		// Set up some cut generators and defaults
		// Probing first as gets tight bounds on continuous
		CglProbing generator1;
		generator1.setUsingObjective(true);
		generator1.setMaxPass(3);
		generator1.setMaxProbe(100);
		generator1.setMaxLook(50);
		generator1.setRowCuts(3);
		//  generator1.snapshot(*model->solver());
		//generator1.createCliques(*model->solver(),2,1000,true);
		//generator1.setMode(0);

		CglGomory generator2;
		// try larger limit
		generator2.setLimit(300);

		CglKnapsackCover generator3;

		CglOddHole generator4;
		generator4.setMinimumViolation(0.005);
		generator4.setMinimumViolationPer(0.00002);
		// try larger limit
		generator4.setMaximumEntries(200);

		CglClique generator5;
		generator5.setStarCliqueReport(false);
		generator5.setRowCliqueReport(false);

		CglMixedIntegerRounding mixedGen;
		CglFlowCover flowGen;

		// Add in generators
		model->addCutGenerator(&generator1,-1,"Probing");
		model->addCutGenerator(&generator2,-1,"Gomory");
		model->addCutGenerator(&generator3,-1,"Knapsack");
		model->addCutGenerator(&generator4,-1,"OddHole");
		model->addCutGenerator(&generator5,-1,"Clique");
		model->addCutGenerator(&flowGen,-1,"FlowCover");
		model->addCutGenerator(&mixedGen,-1,"MixedIntegerRounding");

		#ifdef COIN_HAS_CLP
		OsiClpSolverInterface * osiclp = dynamic_cast< OsiClpSolverInterface*> (model->solver());
		// go faster stripes
		if (osiclp->getNumRows()<300&&osiclp->getNumCols()<500) {
		osiclp->setupForRepeatedUse(2,0);
		//printf("trying slightly less reliable but faster version (? Gomory cuts okay?)\n");
		//printf("may not be safe if doing cuts in tree which need accuracy (level 2 anyway)\n");
		}
		#endif

		// Allow rounding heuristic

		CbcRounding heuristic1(*model);
		/// 0 off, 1 at root, 2 other than root, 3 always.
		heuristic1.setWhen(3);
		model->addHeuristic(&heuristic1);

		// And local search when new solution found

		CbcHeuristicLocal heuristic2(*model);
		heuristic2.setWhen(3);
		model->addHeuristic(&heuristic2);

		CbcHeuristicGreedyCover heuristic3(*model);
		// Use original upper and perturb more
		heuristic3.setAlgorithm(11);
		heuristic3.setWhen(3);
		model->addHeuristic(&heuristic3);

		CbcHeuristicFPump heuristic4(*model);
		heuristic4.setWhen(3);
		model->addHeuristic(&heuristic4);

		CbcHeuristicRINS heuristic5(*model);
		heuristic5.setWhen(3);
		model->addHeuristic(&heuristic5);

		// Redundant definition of default branching (as Default == User)
	//	CbcBranchUserDecision branch;
	//	model->setBranchingMethod(&branch);

		// Definition of node choice
	//	CbcCompareUser compare;
	//	model->setNodeComparison(compare);

		// Could tune more
//		model->setMinimumDrop(min(1.0,
//					 fabs(model->getMinimizationObjValue())*1.0e-3+1.0e-4));

		if (model->getNumCols()<500)
		model->setMaximumCutPassesAtRoot(-100); // always do 100 if possible
		else if (model->getNumCols()<5000)
		model->setMaximumCutPassesAtRoot(100); // use minimum drop
		else
		model->setMaximumCutPassesAtRoot(20);
		//model->setMaximumCutPasses(5);

		// Switch off strong branching if wanted
		// model->setNumberStrong(0);
		// Do more strong branching if small
		if (model->getNumCols()<5000)
		model->setNumberStrong(10);

		model->solver()->setIntParam(OsiMaxNumIterationHotStart,100);

		// If time is given then stop after that number of minutes
	//	std::cout<<"Stopping after "<<minutes<<" minutes"<<std::endl;
		model->setDblParam(CbcModel::CbcMaximumSeconds,60.0*time_limit);

		// Switch off most output
		if (model->getNumCols()<3000) {
		model->messageHandler()->setLogLevel(1);
		//model->solver()->messageHandler()->setLogLevel(0);
		} else {
		model->messageHandler()->setLogLevel(2);
		model->solver()->messageHandler()->setLogLevel(1);
		}
		// Do complete search

		model->branchAndBound();

		if (model->getMinimizationObjValue()<1.0e50) {
			std::ofstream result("result_cbc.txt");
			// Print solution if any - we can't get names from Osi!

			int numberColumns = model->solver()->getNumCols();
		    
			const double * solution = model->solver()->getColSolution();
		    
			int iColumn;
			result<<std::setiosflags(std::ios::fixed|std::ios::showpoint)<<std::setw(14);
		    
			result<<"--------------------------------------"<<std::endl;
			for (iColumn=0;iColumn<numberColumns;iColumn++) {
			  double value=solution[iColumn];
			  if (fabs(value)>1.0e-7&&model->solver()->isInteger(iColumn)) 
			result<<std::setw(6)<<build.getColName(iColumn)<<"("<<iColumn<<") "<<value<<std::endl;
			}
			result<<"--------------------------------------"<<std::endl;
			for (iColumn=0;iColumn<numberColumns;iColumn++) {
			  double value=solution[iColumn];
			  if (fabs(value)>1.0e-7&&!model->solver()->isInteger(iColumn)) 
			result<<std::setw(6)<<build.getColName(iColumn)<<"("<<iColumn<<") "<<value<<std::endl;
			}		  
			result<<std::resetiosflags(std::ios::fixed|std::ios::showpoint|std::ios::scientific);
			result.close();
		  }
	}

	if (model->isAbandoned() )
		return UNSOLVED;
	if (delete_solver) delete solver; // nem a legszebb megoldas...
	solver=model->solver();
	delete_solver=false;
	return SOLVED;
  }


    LpCbc::SolutionStatus MipCbc::_getMipStatus() const {
	  if (!model) return UNDEFINED;
	 switch (model->status()){
		case 0:
			if (model->isProvenOptimal()) return OPTIMAL;
			if (model->isProvenInfeasible()) return INFEASIBLE;
		case 1: 
			return FEASIBLE;
	 }/*

	if (solver.isProvenOptimal())
		 return OPTIMAL;
	if (solver.isProvenPrimalInfeasible() || 
		solver.isProvenDualInfeasible())
         return INFEASIBLE;
//	if (solver.isIterationLimitReached() ||
//		solver.isPrimalObjectiveLimitReached()||
//		solver.isDualObjectiveLimitReached()||
//		solver.isAbandoned())*/
     return UNDEFINED;  
  } 
	 

  MipCbc::Value MipCbc::_getPrimal(int i) const {
	if (model)
		return model->solver()->getColSolution()[i];
	else 
		return -1;
  }
  
  MipCbc::Value MipCbc::_getPrimalValue() const {
	if (model)
		return model->solver()->getObjValue();
	else 
		return -1;
  }

  // void setHotstartSolution(const double * solution, const int * priorities=NULL) ;
} //END OF NAMESPACE LEMON
