#ifndef EDGE_COLORING_H
#define EDGE_COLORING_H

#include <lemon/assert.h>
#include <lemon/core.h>
#include <lemon/connectivity.h>
#include <vector>
#include <algorithm>

///\ingroup edge coloring
///\file
///\brief Edge coloring algorithms for bipartite and general simple graphs.

/// A map that stores valid edge colorings and provides a couple of other
/// fast querries.
///
/// \note If you modify the underlying graph structure then the coloring
///       is invalidated. It must be reconstructed.
template<class G>
class EdgeColoring{
public:
	typedef int Color;
	typedef G Graph;
private:
	typedef typename Graph::Edge Edge;
	typedef typename Graph::Node Node;
	typedef typename Graph::Arc Arc;
public:
	explicit EdgeColoring(const Graph&g, int color_count = 1):
		_g(g), _color_count(color_count), _edge_color(g), _out_arc_map(g){
		LEMON_ASSERT(color_count > 0, "must use at least one color");

		for(typename Graph::EdgeIt i(_g); i!=lemon::INVALID; ++i)
			_edge_color[i] = -1;
		for(typename Graph::NodeIt i(_g); i!=lemon::INVALID; ++i){
			_out_arc_map[i].resize(_color_count);
			for(Color c = 0; c<_color_count; ++c)
				_out_arc_map[i][c] = lemon::INVALID;
		}
	}

	void clearColors(){
		for(typename Graph::EdgeIt i(_g); i!=lemon::INVALID; ++i)
			_edge_color[i] = -1;
		for(typename Graph::NodeIt i(_g); i!=lemon::INVALID; ++i){
			for(Color c = 0; c<_color_count; ++c)
				_out_arc_map[i][c] = lemon::INVALID;
		}
	}

	/// Sets the number of colors. All edges colored using a color >= color_count will
	/// be uncolored afterwards.
	void setColorCount(int color_count){
		LEMON_ASSERT(color_count > 0, "must use at least one color");
		_color_count = color_count;

		for(typename Graph::EdgeIt i(_g); i!=lemon::INVALID; ++i)
			if(_edge_color[i] >= _color_count)
				_edge_color[i] = -1;
		
		for(typename Graph::NodeIt i(_g); i!=lemon::INVALID; ++i){
			_out_arc_map[i].resize(_color_count);
			for(Color c = 0; c<_color_count; ++c)
				_out_arc_map[i][c] = lemon::INVALID;
			for(typename Graph::OutArcIt j(_g, i); j!=lemon::INVALID; ++j)
				if(_edge_color[j] != -1)
					_out_arc_map[i][_edge_color[j]] = j;
		}
	}

	int getColorCount()const{
		return _color_count;
	}

	typedef Edge Key;
	typedef Color Value;

	Color operator[](Edge e)const{
		LEMON_ASSERT(e != lemon::INVALID, "only valid edges may have a color");
		return _edge_color[e];
	}

	void set_no_check(Edge e, Color color){
		LEMON_ASSERT(e != lemon::INVALID, "only valid edges may be colored");
		LEMON_ASSERT(0 <= color && color < _color_count, "color must be valid");
		
		Node u = _g.u(e), v = _g.v(e);

		LEMON_ASSERT(isFree(u, color), "invalid coloring, u has 2 incident edges with the same color");
		LEMON_ASSERT(isFree(v, color), "invalid coloring, v has 2 incident edges with the same color");
		
		if(_edge_color[e] != -1){
			_out_arc_map[u][_edge_color[e]] = lemon::INVALID;
			_out_arc_map[v][_edge_color[e]] = lemon::INVALID;
		}
		_edge_color[e] = color;
		_out_arc_map[u][color] = _g.direct(e, u);
		_out_arc_map[v][color] = _g.direct(e, v);
	}


	bool set(Edge e, Color color){
		LEMON_ASSERT(e != lemon::INVALID, "only valid edges may have be colored");
		LEMON_ASSERT(0 <= color && color < _color_count, "color must be valid");
		
		Node u = _g.u(e), v = _g.v(e);

		if(isFree(u, color) && isFree(v, color)){
			if(_edge_color[e] != -1){
				_out_arc_map[u][_edge_color[e]] = lemon::INVALID;
				_out_arc_map[v][_edge_color[e]] = lemon::INVALID;
			}
			_edge_color[e] = color;
			_out_arc_map[u][color] = _g.direct(e, u);
			_out_arc_map[v][color] = _g.direct(e, v);
			return true;
		}else
			return false;
	}


	void removeColor(Edge e){
		LEMON_ASSERT(e != lemon::INVALID, "e must be valid");

		if(_edge_color[e] != -1){
			Node u = _g.u(e), v = _g.v(e);
			_out_arc_map[u][_edge_color[e]] = lemon::INVALID;
			_out_arc_map[v][_edge_color[e]] = lemon::INVALID;
			_edge_color[e] = -1;
		}
	}

	bool isFree(Node v, Color c)const{
		LEMON_ASSERT(v != lemon::INVALID, "v must be valid");
		LEMON_ASSERT(0 <= c && c < _color_count, "color must be valid");
		
		return getArc(v, c) == lemon::INVALID;
	}

	Arc getArc(Node v, int c)const{
		LEMON_ASSERT(v != lemon::INVALID, "v must be valid");
		LEMON_ASSERT(0 <= c && c < _color_count, "color must be valid");
		
		return _out_arc_map[v][c];
	}

	Color getFreeColor(Node v)const{
		LEMON_ASSERT(v != lemon::INVALID, "v must be valid");
		
		for(Color c=0; c<_color_count; ++c)
			if(isFree(v, c))
				return c;
		return -1;
	}

	Color getFreeColor(Edge e)const{
		LEMON_ASSERT(e != lemon::INVALID, "e must be valid");
		
		Node u = _g.u(e), v = _g.v(e);
		for(Color c=0; c<_color_count; ++c)
			if(isFree(v, c) && isFree(u, c))
				return c;
		return -1;
	}

private:
	const Graph&_g;
	int _color_count;

	typename Graph::template EdgeMap<Color>_edge_color;
	typename Graph::template NodeMap<std::vector<Arc> >_out_arc_map;

	EdgeColoring(const EdgeColoring&);
	EdgeColoring&operator=(const EdgeColoring&);
};

template<class Graph, class EdgeColoring = EdgeColoring<Graph> >
class BipartiteEdgeColoring{
private:
	typedef typename EdgeColoring::Color Color;
	typedef typename Graph::Node Node;
	typedef typename Graph::Arc Arc;
	typedef typename Graph::Edge Edge;
public:

	BipartiteEdgeColoring(const Graph&g, EdgeColoring&coloring):
		_g(g), _coloring(coloring){};

	void init(){
		LEMON_ASSERT(lemon::bipartite(_g), "graph must be bipartite");
		
		int max_degree = 0;
		for(typename Graph::NodeIt i(_g); i!=lemon::INVALID; ++i){
			int degree = lemon::countOutArcs(_g, i);
			if(degree > max_degree)
				max_degree = degree;
		}
		
		_coloring.clearColors();
		_coloring.setColorCount(max_degree);
	}

	void start(){
		for(typename Graph::EdgeIt e(_g); e!=lemon::INVALID; ++e){
			Color c = _coloring.getFreeColor(e);
			if(c != -1)
				_coloring.set(e, c);
			else{
				Arc now = _g.direct(e, false);
				
				// These colors always exist because we have max degree many
				// colors.
				Color s_col = _coloring.getFreeColor(_g.source(now));
				Color t_col = _coloring.getFreeColor(_g.target(now));
	
				LEMON_ASSERT(s_col != -1 && t_col != -1, "internal error or graph was modified between init and start");
				
				// This loop only terminates because all cycles have a even
				// length in a bipartite graph.
				for(;;){
					Arc next = _coloring.getArc(_g.target(now), s_col);
					if(next == lemon::INVALID){
						_coloring.set_no_check(now, s_col);
						break;
					}else{
						_coloring.removeColor(next);
						_coloring.set_no_check(now, s_col);
						now = next;
						using std::swap;
						swap(s_col, t_col);
					}
				}
			}
		}
	}

	void run(){
		init();
		start();
	}
private:
	const Graph&_g;
	EdgeColoring&_coloring;

	BipartiteEdgeColoring(const BipartiteEdgeColoring&);
	BipartiteEdgeColoring&operator=(const BipartiteEdgeColoring&);
};

template<class Graph, class EdgeColoring = EdgeColoring<Graph> >
class VizingEdgeColoring{
private:
	typedef typename EdgeColoring::Color Color;
	typedef typename Graph::Node Node;
	typedef typename Graph::Arc Arc;
	typedef typename Graph::Edge Edge;
public:

	VizingEdgeColoring(const Graph&g, EdgeColoring&coloring):
		_g(g), _coloring(coloring){};

	void init(){
		int max_degree = 0;
		for(typename Graph::NodeIt i(_g); i!=lemon::INVALID; ++i){
			int degree = lemon::countOutArcs(_g, i);
			if(degree > max_degree)
				max_degree = degree;
		}
		
		_coloring.clearColors();
		_coloring.setColorCount(max_degree+1);
	}

private:
	// Tests if the path s --c1--> --c2--> --c1--> --c2--> ... ends in t or not.
	bool does_path_end_at(Node s, Node t, Color c1, Color c2)const{
		LEMON_ASSERT(c1 != c2, "colors must be different");
		Color c[] = {c1, c2};
		int next_col = 0;
		Arc next = _coloring.getArc(s, c[next_col]);
		while(next != lemon::INVALID && s != t){
			s = _g.target(next);
			next_col = 1-next_col;
			next = _coloring.getArc(s, c[next_col]);
		}
		return s == t;
	}

	// Recolors the path 
	//   s --c1--> --c2--> --c1--> --c2--> ... 
	// as
	//   s --c2--> --c1--> --c2--> --c1--> ...
	void recolor_path(Node s, Color c1, Color c2){
		LEMON_ASSERT(c1 != c2, "colors must be different");
		Color c[] = {c1, c2};
		int next_col = 0;
		Arc next = _coloring.getArc(s, c[next_col]);
		LEMON_ASSERT(next != lemon::INVALID, "next is valid");
		for(;;){
			s = _g.target(next);
			next_col = 1-next_col;
			Arc next_next = _coloring.getArc(s, c[next_col]);
			if(next_next == lemon::INVALID){
				_coloring.set_no_check(next, c[next_col]);
				break;
			}else{
				_coloring.removeColor(next_next);
				_coloring.set_no_check(next, c[next_col]);
				next = next_next;
			}
		}
	}

public:
	void start(){
		// The algorithm used is described here:
		// http://www.csc.liv.ac.uk/michele/teaching/comp309/2003/vizing.pdf
		for(typename Graph::EdgeIt e(_g); e!=lemon::INVALID; ++e){
			Color col = _coloring.getFreeColor(e);
			if(col != -1){
				// 1
				_coloring.set_no_check(e, col);
			}else{
				std::vector<Arc>a;
				std::vector<Color>c;

				a.push_back(_g.direct(e, true));
				Node w = _g.source(a[0]);
				c.push_back(_coloring.getFreeColor(_g.target(a[0])));
				
				for(;;){
					LEMON_ASSERT(a.size() == c.size(), "internal error");
					
					int h = c.size();
					a.push_back(_coloring.getArc(w, c.back()));
					Color col = _coloring.getFreeColor(a[h]); 
					if(col != -1){
						// 2.2
						_coloring.set_no_check(a[h], col);
						for(int i=h-1; i>=0; --i)
							_coloring.set_no_check(a[i], c[i]);
						break;
					}else{
						Node v = _g.target(a.back());
						Color missed = _coloring.getFreeColor(v);
						int s = std::find(c.begin(), c.end(), missed) - c.begin();
						if(s == h){
							// 2.1
							c.push_back(_coloring.getFreeColor(_g.target(a[h])));
							continue;
						}else{
							// 2.3
							Color w_missed = _coloring.getFreeColor(w);
							if(!does_path_end_at(_g.target(a[s]), w, w_missed, c[s])){
								// 2.3.1
								recolor_path(_g.target(a[s]), w_missed, c[s]);
								_coloring.set_no_check(a[s], w_missed);
								for(int i=s-1; i>=0; --i)
									_coloring.set_no_check(a[i], c[i]);
								break;
							}else{
								// 2.3.2
								recolor_path(_g.target(a[h]), w_missed, c[s]);
								_coloring.set_no_check(a[h], w_missed);
								for(int i=h-1; i>=0; --i)
									_coloring.set_no_check(a[i], c[i]);
								break;
							}
						}
					}
				}
			}
		}
	}

	void run(){
		init();
		start();
	}
private:
	const Graph&_g;
	EdgeColoring&_coloring;

	VizingEdgeColoring(const VizingEdgeColoring&);
	VizingEdgeColoring&operator=(const VizingEdgeColoring&);
};

#endif
