#include "edge_coloring.h"
#include <lemon/smart_graph.h>
#include <lemon/full_graph.h>
#include "test_tools.h"
#include <vector>

void check_edge_coloring(){
	typedef lemon::SmartGraph Graph;
	Graph g;
	
	Graph::Node 
		a = g.addNode(),
		b = g.addNode(),
		c = g.addNode();

	Graph::Edge
		x = g.addEdge(a, b),
		y = g.addEdge(b, c);

	EdgeColoring<Graph>coloring(g, 3);

	check(coloring.isFree(a, 0), "isFree failed");
	check(coloring.getArc(a, 0) == lemon::INVALID, "getArc failed");
	
	
	check(coloring[x] == -1, "getColor failed");
	check(coloring[y] == -1, "getColor failed");

	check(coloring.set(x, 0) == true, "set failed");
	check(coloring.set(y, 0) == false, "set failed");
	check(coloring[y] == -1, "getColor failed");

	check(Graph::Edge(coloring.getArc(a, 0)) == x, "getArc failed");
	check(Graph::Edge(coloring.getArc(b, 0)) == x, "getArc failed");
	check(coloring.getArc(c, 0) == lemon::INVALID, "getArc failed");

	check(coloring.set(y, 2) == true, "set failed");

	check(coloring[x] == 0, "getColor failed");
	check(coloring[y] == 2, "getColor failed");

	check(coloring.getFreeColor(a) != 0, "getFreeColor failed");
	check(coloring.getFreeColor(b) == 1, "getFreeColor failed");
	check(coloring.getFreeColor(c) != 2, "getFreeColor failed");

	check(coloring.getFreeColor(x) == 1, "getFreeColor failed");
	check(coloring.getFreeColor(y) == 1, "getFreeColor failed");
	
	check(coloring.getColorCount() == 3, "getColorCount failed");
	
	coloring.set(y, 2);
	check(coloring.getArc(b, 1) == lemon::INVALID, "getArc failed");
	check(coloring.getArc(c, 1) == lemon::INVALID, "getArc failed");
	check(Graph::Edge(coloring.getArc(b, 2)) == y, "getArc failed");
	check(Graph::Edge(coloring.getArc(c, 2)) == y, "getArc failed");
	
	coloring.set(y, 1);
	check(Graph::Edge(coloring.getArc(b, 1)) == y, "getArc failed");
	check(Graph::Edge(coloring.getArc(c, 1)) == y, "getArc failed");
	check(coloring.getArc(b, 2) == lemon::INVALID, "getArc failed");
	check(coloring.getArc(c, 2) == lemon::INVALID, "getArc failed");
	
	
	coloring.setColorCount(2);

	check(coloring.getArc(a, 0) != lemon::INVALID, "getArc failed");
	check(coloring.getArc(b, 0) != lemon::INVALID, "getArc failed");
	check(coloring.getArc(b, 1) != lemon::INVALID, "getArc failed");
	check(coloring.getArc(c, 1) != lemon::INVALID, "getArc failed");
	
	check(g.target(coloring.getArc(b, 0)) == a, "getArc failed");
	check(g.target(coloring.getArc(b, 1)) == c, "getArc failed");

	check(coloring.getColorCount() == 2, "getColorCount failed");

	check(coloring.getFreeColor(a) != 0, "getFreeColor failed");
	check(coloring.getFreeColor(b) == -1, "getFreeColor failed");
	check(coloring.getFreeColor(c) != 1, "getFreeColor failed");

	check(coloring.getFreeColor(x) == -1, "getFreeColor failed");
	check(coloring.getFreeColor(y) == -1, "getFreeColor failed");

	coloring.setColorCount(1);

	check(coloring[x] == 0, "getColor failed");
	check(coloring[y] == -1, "getColor failed");

	coloring.clearColors();
	check(coloring[x] == -1, "getColor failed");
	check(coloring[y] == -1, "getColor failed");
	check(coloring.getArc(a, 0) == lemon::INVALID, "getArc failed");
	check(coloring.getArc(b, 0) == lemon::INVALID, "getArc failed");
	check(coloring.getArc(c, 0) == lemon::INVALID, "getArc failed");
	
}

void check_bipartite_edge_coloring(){
	typedef lemon::SmartGraph Graph;
	Graph g;

	const int node_count = 20;
	
	std::vector<Graph::Node>left(node_count), right(node_count);
	for(int i=0; i<node_count; ++i){
		left[i] = g.addNode();
		right[i] = g.addNode();
	}

	for(int i=0; i<node_count; ++i)
		for(int j=0; j<node_count; ++j)
			g.addEdge(left[i], right[j]);

	EdgeColoring<Graph>coloring(g);
	BipartiteEdgeColoring<Graph>algo(g, coloring);

	algo.run();

	for(Graph::EdgeIt i(g); i!=lemon::INVALID; ++i)
		check(coloring[i] != -1, "bipartite coloring not complete");
}

void check_vizing(){
	typedef lemon::FullGraph Graph;
	Graph g(20);

	EdgeColoring<Graph>coloring(g);
	VizingEdgeColoring<Graph>algo(g, coloring);

	algo.run();

	for(Graph::EdgeIt i(g); i!=lemon::INVALID; ++i)
		check(coloring[i] != -1, "coloring not complete");
}

int main(){
	check_edge_coloring();
	check_bipartite_edge_coloring();
	check_vizing();
	return 0;
}


