// -*- c++ -*-
#include <iostream>
#include <fstream>

#include <lemon/smart_graph.h>
#include <lemon/list_graph.h>
#include <lemon/dimacs.h>
#include <lemon/time_measure.h>
//#include <graph_wrapper.h>
#include <lemon/preflow.h>
#include <augmenting_flow.h>
//#include <preflow_res.h>
#include <lp_solver_wrapper.h>

// Use a DIMACS max flow file as stdin.
// max_flow_demo < dimacs_max_flow_file

using namespace lemon;

namespace lemon {

  template<typename Edge, typename EdgeIndexMap> 
  class PrimalMap {
  protected:
    LPSolverWrapper* lp;
    EdgeIndexMap* edge_index_map;
  public:
    PrimalMap(LPSolverWrapper& _lp, EdgeIndexMap& _edge_index_map) : 
      lp(&_lp), edge_index_map(&_edge_index_map) { }
    double operator[](Edge e) const { 
      return lp->getPrimal((*edge_index_map)[e]);
    }
  };

  // excess: rho-delta
  template <typename Graph, typename Num,
	    typename Excess=typename Graph::template NodeMap<Num>, 
	    typename LCapMap=typename Graph::template EdgeMap<Num>,
	    typename CapMap=typename Graph::template EdgeMap<Num>,
            typename FlowMap=typename Graph::template EdgeMap<Num>,
	    typename CostMap=typename Graph::template EdgeMap<Num> >
  class MinCostGenFlow {
  protected:
    const Graph& g;
    const Excess& excess;
    const LCapMap& lcapacity;
    const CapMap& capacity;
    FlowMap& flow;
    const CostMap& cost;
  public:
    MinCostGenFlow(const Graph& _g, const Excess& _excess, 
		   const LCapMap& _lcapacity, const CapMap& _capacity, 
		   FlowMap& _flow, 
		   const CostMap& _cost) :
      g(_g), excess(_excess), lcapacity(_lcapacity),
      capacity(_capacity), flow(_flow), cost(_cost) { }
    void run() {
      LPSolverWrapper lp;
      lp.setMinimize();
      typedef LPSolverWrapper::ColIt ColIt;
      typedef LPSolverWrapper::RowIt RowIt;
      typedef typename Graph::template EdgeMap<ColIt> EdgeIndexMap;
      EdgeIndexMap edge_index_map(g);
      PrimalMap<typename Graph::Edge, EdgeIndexMap> lp_flow(lp, edge_index_map);
      for (typename Graph::EdgeIt e(g); e!=INVALID; ++e) {
	ColIt col_it=lp.addCol();
	edge_index_map.set(e, col_it);
	if (lcapacity[e]==capacity[e])
	  lp.setColBounds(col_it, LPX_FX, lcapacity[e], capacity[e]);
	else 
	  lp.setColBounds(col_it, LPX_DB, lcapacity[e], capacity[e]);
	lp.setObjCoef(col_it, cost[e]);
      }
      for (typename Graph::NodeIt n(g); n!=INVALID; ++n) {
	typename Graph::template EdgeMap<Num> coeffs(g, 0);
	for (typename Graph::InEdgeIt e(g, n); e!=INVALID; ++e)
	coeffs.set(e, coeffs[e]+1);
	for (typename Graph::OutEdgeIt e(g, n); e!=INVALID; ++e) 
	coeffs.set(e, coeffs[e]-1);
	RowIt row_it=lp.addRow();
	typename std::vector< std::pair<ColIt, double> > row;
	//std::cout << "node:" <<g.id(n)<<std::endl;
	for (typename Graph::EdgeIt e(g); e!=INVALID; ++e) {
	  if (coeffs[e]!=0) {
	    //std::cout << " edge:" <<g.id(e)<<" "<<coeffs[e];
	    row.push_back(std::make_pair(edge_index_map[e], coeffs[e]));
	  }
	}
	//std::cout << std::endl;
	lp.setRowCoeffs(row_it, row.begin(), row.end());
	lp.setRowBounds(row_it, LPX_FX, 0.0, 0.0);
      }
      lp.solveSimplex();
      //std::cout << lp.colNum() << std::endl;
      //std::cout << lp.rowNum() << std::endl;
      //std::cout << "flow value: "<< lp.getObjVal() << std::endl;
      for (typename Graph::EdgeIt e(g); e!=INVALID; ++e) 
      flow.set(e, lp_flow[e]);
    }
  };

}

int main(int, char **) {

  typedef ListGraph MutableGraph;
  typedef SmartGraph Graph;
  typedef Graph::Node Node;
  typedef Graph::Edge Edge;
  typedef Graph::EdgeIt EdgeIt;

  Graph g;

  Node s, t;
  Graph::EdgeMap<int> cap(g);
  //readDimacsMaxFlow(std::cin, g, s, t, cap);
  readDimacs(std::cin, g, cap, s, t);
  Timer ts;
  Graph::EdgeMap<int> flow(g); //0 flow
  Preflow<Graph, int, Graph::EdgeMap<int>, Graph::EdgeMap<int> > 
    max_flow_test(g, s, t, cap, flow);
  AugmentingFlow<Graph, int, Graph::EdgeMap<int>, Graph::EdgeMap<int> > 
    augmenting_flow_test(g, s, t, cap, flow);
  
  Graph::NodeMap<bool> cut(g);

  {
    std::cout << "preflow ..." << std::endl;
    ts.reset();
    max_flow_test.run();
    std::cout << "elapsed time: " << ts << std::endl;
    std::cout << "flow value: "<< max_flow_test.flowValue() << std::endl;
    max_flow_test.minCut(cut);

    for (EdgeIt e(g); e!=INVALID; ++e) {
      if (cut[g.source(e)] && !cut[g.target(e)] && !flow[e]==cap[e]) 
	std::cout << "Slackness does not hold!" << std::endl;
      if (!cut[g.source(e)] && cut[g.target(e)] && flow[e]>0) 
	std::cout << "Slackness does not hold!" << std::endl;
    }
  }

//   {
//     std::cout << "preflow ..." << std::endl;
//     FOR_EACH_LOC(Graph::EdgeIt, e, g) flow.set(e, 0);
//     ts.reset();
//     max_flow_test.preflow(Preflow<Graph, int, Graph::EdgeMap<int>, Graph::EdgeMap<int> >::GEN_FLOW);
//     std::cout << "elapsed time: " << ts << std::endl;
//     std::cout << "flow value: "<< max_flow_test.flowValue() << std::endl;

//     FOR_EACH_LOC(Graph::EdgeIt, e, g) {
//       if (cut[g.source(e)] && !cut[g.target(e)] && !flow[e]==cap[e]) 
// 	std::cout << "Slackness does not hold!" << std::endl;
//       if (!cut[g.source(e)] && cut[g.target(e)] && flow[e]>0) 
// 	std::cout << "Slackness does not hold!" << std::endl;
//     }
//   }

//   {
//     std::cout << "wrapped preflow ..." << std::endl;
//     FOR_EACH_LOC(Graph::EdgeIt, e, g) flow.set(e, 0);
//     ts.reset();
//     pre_flow_res.run();
//     std::cout << "elapsed time: " << ts << std::endl;
//     std::cout << "flow value: "<< pre_flow_test.flowValue() << std::endl;
//   }

  {
    std::cout << "physical blocking flow augmentation ..." << std::endl;
    for (EdgeIt e(g); e!=INVALID; ++e) flow.set(e, 0);
    ts.reset();
    int i=0;
    while (augmenting_flow_test.augmentOnBlockingFlow<MutableGraph>()) { ++i; }
    std::cout << "elapsed time: " << ts << std::endl;
    std::cout << "number of augmentation phases: " << i << std::endl; 
    std::cout << "flow value: "<< augmenting_flow_test.flowValue() << std::endl;

    for (EdgeIt e(g); e!=INVALID; ++e) {
      if (cut[g.source(e)] && !cut[g.target(e)] && !flow[e]==cap[e]) 
	std::cout << "Slackness does not hold!" << std::endl;
      if (!cut[g.source(e)] && cut[g.target(e)] && flow[e]>0) 
	std::cout << "Slackness does not hold!" << std::endl;
    }
  }

//   {
//     std::cout << "faster physical blocking flow augmentation ..." << std::endl;
//     FOR_EACH_LOC(Graph::EdgeIt, e, g) flow.set(e, 0);
//     ts.reset();
//     int i=0;
//     while (max_flow_test.augmentOnBlockingFlow1<MutableGraph>()) { ++i; }
//     std::cout << "elapsed time: " << ts << std::endl;
//     std::cout << "number of augmentation phases: " << i << std::endl; 
//     std::cout << "flow value: "<< max_flow_test.flowValue() << std::endl;
//   }

  {
    std::cout << "on-the-fly blocking flow augmentation ..." << std::endl;
    for (EdgeIt e(g); e!=INVALID; ++e) flow.set(e, 0);
    ts.reset();
    int i=0;
    while (augmenting_flow_test.augmentOnBlockingFlow2()) { ++i; }
    std::cout << "elapsed time: " << ts << std::endl;
    std::cout << "number of augmentation phases: " << i << std::endl; 
    std::cout << "flow value: "<< augmenting_flow_test.flowValue() << std::endl;

    for (EdgeIt e(g); e!=INVALID; ++e) {
      if (cut[g.source(e)] && !cut[g.target(e)] && !flow[e]==cap[e]) 
	std::cout << "Slackness does not hold!" << std::endl;
      if (!cut[g.source(e)] && cut[g.target(e)] && flow[e]>0) 
	std::cout << "Slackness does not hold!" << std::endl;
    }
  }

//   {
//     std::cout << "on-the-fly shortest path augmentation ..." << std::endl;
//     FOR_EACH_LOC(Graph::EdgeIt, e, g) flow.set(e, 0);
//     ts.reset();
//     int i=0;
//     while (augmenting_flow_test.augmentOnShortestPath()) { ++i; }
//     std::cout << "elapsed time: " << ts << std::endl;
//     std::cout << "number of augmentation phases: " << i << std::endl; 
//     std::cout << "flow value: "<< augmenting_flow_test.flowValue() << std::endl;

//     FOR_EACH_LOC(Graph::EdgeIt, e, g) {
//       if (cut[g.source(e)] && !cut[g.target(e)] && !flow[e]==cap[e]) 
// 	std::cout << "Slackness does not hold!" << std::endl;
//       if (!cut[g.source(e)] && cut[g.target(e)] && flow[e]>0) 
// 	std::cout << "Slackness does not hold!" << std::endl;
//     }
//   }

//   {
//     std::cout << "on-the-fly shortest path augmentation ..." << std::endl;
//     FOR_EACH_LOC(Graph::EdgeIt, e, g) flow.set(e, 0);
//     ts.reset();
//     int i=0;
//     while (augmenting_flow_test.augmentOnShortestPath2()) { ++i; }
//     std::cout << "elapsed time: " << ts << std::endl;
//     std::cout << "number of augmentation phases: " << i << std::endl; 
//     std::cout << "flow value: "<< augmenting_flow_test.flowValue() << std::endl;

//     FOR_EACH_LOC(Graph::EdgeIt, e, g) {
//       if (cut[g.source(e)] && !cut[g.target(e)] && !flow[e]==cap[e]) 
// 	std::cout << "Slackness does not hold!" << std::endl;
//       if (!cut[g.source(e)] && cut[g.target(e)] && flow[e]>0) 
// 	std::cout << "Slackness does not hold!" << std::endl;
//     }
//   }

  ts.reset();

  Edge e=g.addEdge(t, s);
  Graph::EdgeMap<int> cost(g, 0);
  cost.set(e, -1);
  cap.set(e, 10000);
  typedef ConstMap<Node, int> Excess;
  Excess excess(0);
  typedef ConstMap<Edge, int> LCap;
  LCap lcap(0);

  MinCostGenFlow<Graph, int, Excess, LCap> 
    min_cost(g, excess, lcap, cap, flow, cost); 
  min_cost.run();

  std::cout << "elapsed time: " << ts << std::endl;
  std::cout << "flow value: "<< flow[e] << std::endl;
}
