/* -*- C++ -*-
 * src/lemon/graph_to_eps.h - Part of LEMON, a generic C++ optimization library
 *
 * Copyright (C) 2004 Egervary Jeno Kombinatorikus Optimalizalasi Kutatocsoport
 * (Egervary Combinatorial Optimization Research Group, EGRES).
 *
 * Permission to use, modify and distribute this software is granted
 * provided that this copyright notice appears in all copies. For
 * precise terms see the accompanying LICENSE file.
 *
 * This software is provided "AS IS" with no warranty of any kind,
 * express or implied, and with no claim as to its suitability for any
 * purpose.
 *
 */

#include <iostream>
#include <fstream>
#include <algorithm>
#include<math.h>

#include<lemon/xy.h>
#include<lemon/maps.h>
#include<lemon/list_graph.h>


///\ingroup misc
///\file
///\brief Simple graph drawer

namespace lemon {

///Data structure representing RGB colors.

///Data structure representing RGB colors.
///\ingroup misc
class Color
{
  double _r,_g,_b;
public:
  ///Default constructor
  Color() {}
  ///Constructor
  Color(double r,double g,double b) :_r(r),_g(g),_b(b) {};
  ///Returns the red component
  double getR() {return _r;}
  ///Returns the green component
  double getG() {return _g;}
  ///Returns the blue component
  double getB() {return _b;}
  ///Set the color components
  void set(double r,double g,double b) { _r=r;_g=g;_b=b; };
};
  
///Default traits class of \ref GraphToEps

///Default traits class of \ref GraphToEps
///
///\c G is the type of the underlying graph.
template<class G>
struct DefaultGraphToEpsTraits
{
  typedef G Graph;
  typedef typename Graph::Node Node;
  typedef typename Graph::NodeIt NodeIt;
  typedef typename Graph::Edge Edge;
  typedef typename Graph::EdgeIt EdgeIt;
  typedef typename Graph::InEdgeIt InEdgeIt;
  typedef typename Graph::OutEdgeIt OutEdgeIt;
  

  const Graph &g;

  std::ostream& os;
  
  ConstMap<typename Graph::Node,xy<double> > _coords;
  ConstMap<typename Graph::Node,double > _nodeSizes;

  ConstMap<typename Graph::Node,Color > _nodeColors;
  ConstMap<typename Graph::Edge,Color > _edgeColors;

  ConstMap<typename Graph::Edge,double > _edgeWidths;
  
  double _edgeWidthScale;
  
  double _nodeScale;
  double _xBorder, _yBorder;
  double _scale;
  double _nodeBorderQuotient;
  
  bool _drawArrows;
  double _arrowLength, _arrowWidth;
  
  bool _showNodes, _showEdges;

  bool _enableParallel;
  double _parEdgeDist;

  bool _showNodeText;
  ConstMap<typename Graph::Node,bool > _nodeTexts;  
  double _nodeTextSize;

  bool _pleaseRemoveOsStream;
  ///Constructor

  ///Constructor
  ///\param _g is a reference to the graph to be printed
  ///\param _os is a reference to the output stream.
  ///\param _os is a reference to the output stream.
  ///\param _pros If it is \c true, then the \c ostream referenced by \c _os
  ///will be explicitly deallocated by the destructor.
  ///By default it is <tt>std::cout</tt>
  DefaultGraphToEpsTraits(const G &_g,std::ostream& _os=std::cout,
			  bool _pros=false) :
    g(_g), os(_os),
    _coords(xy<double>(1,1)), _nodeSizes(1.0),
    _nodeColors(Color(1,1,1)), _edgeColors(Color(0,0,0)),
    _edgeWidths(1), _edgeWidthScale(0.3),
    _nodeScale(1.0), _xBorder(10), _yBorder(10), _scale(1.0),
    _nodeBorderQuotient(.1),
    _drawArrows(false), _arrowLength(1), _arrowWidth(0.3),
    _showNodes(true), _showEdges(true),
    _enableParallel(false), _parEdgeDist(1),
    _showNodeText(false), _nodeTexts(false), _nodeTextSize(1),
    _pleaseRemoveOsStream(_pros) {}
};

///Helper class to implement the named parameters of \ref graphToEps()

///Helper class to implement the named parameters of \ref graphToEps()
///\todo Is 'helper class' a good name for this?
///
template<class T> class GraphToEps : public T 
{
  typedef typename T::Graph Graph;
  typedef typename Graph::Node Node;
  typedef typename Graph::NodeIt NodeIt;
  typedef typename Graph::Edge Edge;
  typedef typename Graph::EdgeIt EdgeIt;
  typedef typename Graph::InEdgeIt InEdgeIt;
  typedef typename Graph::OutEdgeIt OutEdgeIt;

  bool dontPrint;

  class edgeLess {
    const Graph &g;
  public:
    edgeLess(const Graph &_g) : g(_g) {}
    bool operator()(Edge a,Edge b) const 
    {
      Node ai=min(g.source(a),g.target(a));
      Node aa=max(g.source(a),g.target(a));
      Node bi=min(g.source(b),g.target(b));
      Node ba=max(g.source(b),g.target(b));
      return ai<bi ||
	(ai==bi && (aa < ba || 
		    (aa==ba && ai==g.source(a) && bi==g.target(b))));
    }
  };
  bool isParallel(Edge e,Edge f) const
  {
    return (g.source(e)==g.source(f)&&g.target(e)==g.target(f))||
      (g.source(e)==g.target(f)&&g.target(e)==g.source(f));
  }
  static xy<double> rot(xy<double> v) 
  {
    return xy<double>(v.y,-v.x);
  }
  
public:
  GraphToEps(const T &t) : T(t), dontPrint(false) {};
  
  template<class X> struct CoordsTraits : public T {
    const X &_coords;
    CoordsTraits(const T &t,const X &x) : T(t), _coords(x) {}
  };
  ///Sets the map of the node coordinates

  ///Sets the map of the node coordinates.
  ///\param x must be a node map with xy<double> or xy<int> values. 
  template<class X> GraphToEps<CoordsTraits<X> > coords(const X &x) {
    dontPrint=true;
    return GraphToEps<CoordsTraits<X> >(CoordsTraits<X>(*this,x));
  }
  template<class X> struct NodeSizesTraits : public T {
    const X &_nodeSizes;
    NodeSizesTraits(const T &t,const X &x) : T(t), _nodeSizes(x) {}
  };
  ///Sets the map of the node sizes

  ///Sets the map of the node sizes
  ///\param x must be a node map with \c double (or convertible) values. 
  template<class X> GraphToEps<NodeSizesTraits<X> > nodeSizes(const X &x)
  {
    dontPrint=true;
    return GraphToEps<NodeSizesTraits<X> >(NodeSizesTraits<X>(*this,x));
  }
  template<class X> struct NodeTextsTraits : public T {
    const X &_nodeTexts;
    NodeTextsTraits(const T &t,const X &x) : T(t), _nodeTexts(x) {}
  };
  ///Sets the text printed on the nodes

  ///Sets the text printed on the nodes
  ///\param x must be a node map with type that can be pushed to a standard
  ///ostream. 
  template<class X> GraphToEps<NodeTextsTraits<X> > nodeTexts(const X &x)
  {
    dontPrint=true;
    _showNodeText=true;
    return GraphToEps<NodeTextsTraits<X> >(NodeTextsTraits<X>(*this,x));
  }
   template<class X> struct EdgeWidthsTraits : public T {
    const X &_edgeWidths;
    EdgeWidthsTraits(const T &t,const X &x) : T(t), _edgeWidths(x) {}
  };
  ///Sets the map of the edge widths

  ///Sets the map of the edge widths
  ///\param x must be a edge map with \c double (or convertible) values. 
  template<class X> GraphToEps<EdgeWidthsTraits<X> > edgeWidths(const X &x)
  {
    dontPrint=true;
    return GraphToEps<EdgeWidthsTraits<X> >(EdgeWidthsTraits<X>(*this,x));
  }

  template<class X> struct NodeColorsTraits : public T {
    const X &_nodeColors;
    NodeColorsTraits(const T &t,const X &x) : T(t), _nodeColors(x) {}
  };
  ///Sets the map of the node colors

  ///Sets the map of the node colors
  ///\param x must be a node map with \ref Color values. 
  template<class X> GraphToEps<NodeColorsTraits<X> >
  nodeColors(const X &x)
  {
    dontPrint=true;
    return GraphToEps<NodeColorsTraits<X> >(NodeColorsTraits<X>(*this,x));
  }
  template<class X> struct EdgeColorsTraits : public T {
    const X &_edgeColors;
    EdgeColorsTraits(const T &t,const X &x) : T(t), _edgeColors(x) {}
  };
  ///Sets the map of the edge colors

  ///Sets the map of the edge colors
  ///\param x must be a edge map with \ref Color values. 
  template<class X> GraphToEps<EdgeColorsTraits<X> >
  edgeColors(const X &x)
  {
    dontPrint=true;
    return GraphToEps<EdgeColorsTraits<X> >(EdgeColorsTraits<X>(*this,x));
  }
  ///Sets a global scale factor for node sizes

  ///Sets a global scale factor for node sizes
  ///
  GraphToEps<T> &nodeScale(double d) {_nodeScale=d;return *this;}
  ///Sets a global scale factor for edge widths

  ///Sets a global scale factor for edge widths
  ///
  GraphToEps<T> &edgeWidthScale(double d) {_edgeWidthScale=d;return *this;}
  ///Sets a global scale factor for the whole picture

  ///Sets a global scale factor for the whole picture
  ///
  GraphToEps<T> &scale(double d) {_scale=d;return *this;}
  ///Sets the width of the border around the picture

  ///Sets the width of the border around the picture
  ///
  GraphToEps<T> &border(double b) {_xBorder=_yBorder=b;return *this;}
  ///Sets the width of the border around the picture

  ///Sets the width of the border around the picture
  ///
  GraphToEps<T> &border(double x, double y) {
    _xBorder=x;_yBorder=y;return *this;
  }
  ///Sets whether to draw arrows

  ///Sets whether to draw arrows
  ///
  GraphToEps<T> &drawArrows(bool b=true) {_drawArrows=b;return *this;}
  ///Sets the length of the arrowheads

  ///Sets the length of the arrowheads
  ///
  GraphToEps<T> &arrowLength(double d) {_arrowLength*=d;return *this;}
  ///Sets the width of the arrowheads

  ///Sets the width of the arrowheads
  ///
  GraphToEps<T> &arrowWidth(double d) {_arrowWidth*=d;return *this;}
  
  ///Enables parallel edges

  ///Enables parallel edges
  ///\todo Partially implemented
  GraphToEps<T> &enableParallel(bool b=true) {_enableParallel=b;return *this;}
  
  ///Sets the distance 
  
  ///Sets the distance 
  ///
  GraphToEps<T> &parEdgeDist(double d) {_parEdgeDist*=d;return *this;}
  
  ///Hides the edges
  
  ///Hides the edges
  ///
  GraphToEps<T> &hideEdges(bool b=true) {_showEdges=!b;return *this;}
  ///Hides the nodes
  
  ///Hides the nodes
  ///
  GraphToEps<T> &hideNodes(bool b=true) {_showNodes=!b;return *this;}
  
  ///Sets the size of the node texts
  
  ///Sets the size of the node texts
  ///
  GraphToEps<T> &nodeTextSize(double d) {_nodeTextSize=d;return *this;}

  
  ~GraphToEps() 
  {
    if(dontPrint) return;
    
    os << "%!PS-Adobe-2.0 EPSF-2.0\n";
    //\todo: Chech whether the graph is empty.
    BoundingBox<double> bb;
    for(NodeIt n(g);n!=INVALID;++n) {
      double ns=_nodeSizes[n]*_nodeScale;
      xy<double> p(ns,ns);
      bb+=p+_coords[n];
      bb+=-p+_coords[n];
      }
    os << "%%BoundingBox: "
	 << bb.left()*  _scale-_xBorder << ' '
	 << bb.bottom()*_scale-_yBorder << ' '
	 << bb.right()* _scale+_xBorder << ' '
	 << bb.top()*   _scale+_yBorder << '\n';
    //x1 y1 x2 y2 x3 y3 cr cg cb w
    os << "/lb { setlinewidth setrgbcolor newpath moveto\n"
       << "      4 2 roll 1 index 1 index curveto stroke } bind def\n";
    os << "/l { setlinewidth setrgbcolor newpath moveto lineto stroke } bind def\n";
    os << "/c { newpath dup 3 index add 2 index moveto 0 360 arc closepath } bind def\n";
    // x y r cr cg cb
    os << "/n { setrgbcolor 2 index 2 index 2 index c fill\n"
       << "     0 0 0 setrgbcolor dup "
       << _nodeBorderQuotient << " mul setlinewidth "
       << 1+_nodeBorderQuotient/2 << " div c stroke\n"
       << "   } bind def\n";
    os << "/arrl " << _arrowLength << " def\n";
    os << "/arrw " << _arrowWidth << " def\n";
    // l dx_norm dy_norm
    os << "/lrl { 2 index mul exch 2 index mul exch rlineto pop} bind def\n";
    //len w dx_norm dy_norm x1 y1 cr cg cb
    os << "/arr { setrgbcolor /y1 exch def /x1 exch def /dy exch def /dx exch def\n"
       << "       /w exch def /len exch def\n"
      //	 << "       0.1 setlinewidth x1 y1 moveto dx len mul dy len mul rlineto stroke"
       << "       newpath x1 dy w 2 div mul add y1 dx w 2 div mul sub moveto\n"
       << "       len w sub arrl sub dx dy lrl\n"
       << "       arrw dy dx neg lrl\n"
       << "       dx arrl w add mul dy w 2 div arrw add mul sub\n"
       << "       dy arrl w add mul dx w 2 div arrw add mul add rlineto\n"
       << "       dx arrl w add mul neg dy w 2 div arrw add mul sub\n"
       << "       dy arrl w add mul neg dx w 2 div arrw add mul add rlineto\n"
       << "       arrw dy dx neg lrl\n"
       << "       len w sub arrl sub neg dx dy lrl\n"
       << "       closepath fill } bind def\n";
    os << "/cshow { 2 index 2 index moveto\n"
       << "         dup stringwidth pop neg 2 div fosi .35 mul neg rmoveto show pop pop} def\n";

    os << "\ngsave\n";
    if(_scale!=1.0) os << _scale << " dup scale\n";
    
    os << "%Edges:\ngsave\n";
    
    if(_showEdges)
      if(_enableParallel) {
	vector<Edge> el;
	for(EdgeIt e(g);e!=INVALID;++e) el.push_back(e);
	sort(el.begin(),el.end(),edgeLess(g));
	
	typename vector<Edge>::iterator j;
	for(typename vector<Edge>::iterator i=el.begin();i!=el.end();i=j) {
	  for(j=i+1;j!=el.end()&&isParallel(*i,*j);++j) ;
	  
	  if(_drawArrows) {
	    // 	  xy<double> d(_coords[g.target(e)]-_coords[g.source(e)]);
	    // 	  double l=sqrt(d.normSquare());
	    // 	  d/=l;
	    // 	  xy<double> x1(d*_nodeScale*_nodeSizes[g.source(e)]+
	    // 			_coords[g.source(e)]);
	    // 	  os << l-(_nodeSizes[g.source(e)]+
	    // 		     _nodeSizes[g.target(e)])*_nodeScale << ' '
	    // 	       << _edgeWidths[e]*_edgeWidthScale << ' '
	    // 	       << d.x << ' ' << d.y << ' '
	    // 	       << x1.x << ' ' << x1.y << ' '
	    // 	       << _edgeColors[e].getR() << ' '
	    // 	       << _edgeColors[e].getG() << ' '
	    // 	       << _edgeColors[e].getB() << " arr\n";
	  }
	  else {
	    double sw=0;
	    for(typename vector<Edge>::iterator e=i;e!=j;++e)
	      sw+=_edgeWidths[*e]*_edgeWidthScale+_parEdgeDist;
	    sw-=_parEdgeDist;
	    sw/=-2.0;
	    xy<double> d(_coords[g.target(*i)]-_coords[g.source(*i)]);
	    double l=sqrt(d.normSquare());
	    d/=l;
	    for(typename vector<Edge>::iterator e=i;e!=j;++e) {
	      sw+=_edgeWidths[*e]*_edgeWidthScale/2.0;
	      xy<double> m(_coords[g.target(*e)]+_coords[g.source(*e)]);
	      m=m/2.0+rot(d)*sw/.75;
	      os << _coords[g.source(*e)].x << ' '
		 << _coords[g.source(*e)].y << ' '
		 << m.x << ' ' << m.y << ' '
		 << _coords[g.target(*e)].x << ' '
		 << _coords[g.target(*e)].y << ' '
		 << _edgeColors[*e].getR() << ' '
		 << _edgeColors[*e].getG() << ' '
		 << _edgeColors[*e].getB() << ' '
		 << _edgeWidths[*e]*_edgeWidthScale << " lb\n";
	      sw+=_edgeWidths[*e]*_edgeWidthScale/2.0+_parEdgeDist;
	    }
	  }
	}
      }
      else for(NodeIt n(g);n!=INVALID;++n)
	for(OutEdgeIt e(g,n);e!=INVALID;++e)
	  if(_drawArrows) {
	    xy<double> d(_coords[g.target(e)]-_coords[g.source(e)]);
	    double l=sqrt(d.normSquare());
	    d/=l;
	    xy<double> x1(d*_nodeScale*_nodeSizes[g.source(e)]+
			  _coords[g.source(e)]);
	    os << l-(_nodeSizes[g.source(e)]+
		     _nodeSizes[g.target(e)])*_nodeScale << ' '
	       << _edgeWidths[e]*_edgeWidthScale << ' '
	       << d.x << ' ' << d.y << ' '
	       << x1.x << ' ' << x1.y << ' '
	       << _edgeColors[e].getR() << ' '
	       << _edgeColors[e].getG() << ' '
	       << _edgeColors[e].getB() << " arr\n";
	  }
	  else os << _coords[g.source(e)].x << ' '
		  << _coords[g.source(e)].y << ' '
		  << _coords[g.target(e)].x << ' '
		  << _coords[g.target(e)].y << ' '
		  << _edgeColors[e].getR() << ' '
		  << _edgeColors[e].getG() << ' '
		  << _edgeColors[e].getB() << ' '
		  << _edgeWidths[e]*_edgeWidthScale << " l\n";
    os << "grestore\n%Nodes:\ngsave\n";
    if(_showNodes)
      for(NodeIt n(g);n!=INVALID;++n)
	os << _coords[n].x << ' ' << _coords[n].y << ' '
	   << _nodeSizes[n]*_nodeScale << ' '
	   << _nodeColors[n].getR() << ' '
	   << _nodeColors[n].getG() << ' '
	   << _nodeColors[n].getB() << " n\n"; 
    if(_showNodeText) {
      os << "grestore\n%Node texts:\ngsave\n";
      os << "/fosi " << _nodeTextSize << " def\n";
      os << "(Helvetica) findfont fosi scalefont setfont\n";
      os << "0 0 0 setrgbcolor\n";
      for(NodeIt n(g);n!=INVALID;++n)
	os << _coords[n].x << ' ' << _coords[n].y
	   << " (" << _nodeTexts[n] << ") cshow\n";
    }
    os << "grestore\ngrestore\n";

    //CleanUp:
    if(_pleaseRemoveOsStream) {delete &os;}
  } 
};


///Generates an EPS file from a graph

///\ingroup misc
///Generates an EPS file from a graph.
///\param g is a reference to the graph to be printed
///\param os is a reference to the output stream.
///By default it is <tt>std::cout</tt>
///
///This function also has a lot of \ref named-templ-param "named parameters",
///they are declared as the members of class \ref GraphToEps. The following
///example shows how to use these parameters.
///\code
/// graphToEps(g).scale(10).coords(coords)
///              .nodeScale(2).nodeSizes(sizes)
///              .edgeWidthScale(.4);
///\endcode
///\sa GraphToEps
///\sa graphToEps(G &g, char *file_name)
template<class G>
GraphToEps<DefaultGraphToEpsTraits<G> > 
graphToEps(G &g, std::ostream& os=std::cout)
{
  return 
    GraphToEps<DefaultGraphToEpsTraits<G> >(DefaultGraphToEpsTraits<G>(g,os));
}
 
///Generates an EPS file from a graph

//\ingroup misc
///This function does the same as
///\ref graphToEps(G &g,std::ostream& os)
///but it writes its output into the file \c file_name
///instead of a stream.
///\sa graphToEps(G &g, std::ostream& os)
template<class G>
GraphToEps<DefaultGraphToEpsTraits<G> > 
graphToEps(G &g,char *file_name)
{
  return GraphToEps<DefaultGraphToEpsTraits<G> >
    (DefaultGraphToEpsTraits<G>(g,*new ofstream(file_name),true));
}

//Generates an EPS file from a graph.
//\param g is a reference to the graph to be printed
//\param file_name is the output file_name.
//
//This function also has a lot of \ref named-templ-param "named parameters",
//they are declared as the members of class \ref GraphToEps. The following
//example shows how to use these parameters.
//\code
// graphToEps(g).scale(10).coords(coords)
//              .nodeScale(2).nodeSizes(sizes)
//              .edgeWidthScale(.4);
//\endcode
//\sa GraphToEps
//\todo Avoid duplicated documentation
//\bug Exception handling is missing? (Or we can just ignore it?)

}

using namespace lemon;

class ColorSet : public MapBase<int,Color>
{
public:
  Color operator[](int i) const
  {
    switch(i%8){
    case 0: return Color(0,0,0);
    case 1: return Color(1,0,0);
    case 2: return Color(0,1,0);
    case 3: return Color(0,0,1);
    case 4: return Color(1,1,0);
    case 5: return Color(1,0,1);
    case 6: return Color(0,1,1);
    case 7: return Color(1,1,1);
    }
    return Color(0,0,0);
  }
} colorSet;

class IdMap :public MapBase<ListGraph::Node,int>
{
  const ListGraph &g;
public:
  IdMap(const ListGraph &_g) :g(_g) {}
  Value operator[](Key n) const { return g.id(n); }
};



int main()
{
  ListGraph g;
  typedef ListGraph::Node Node;
  typedef ListGraph::NodeIt NodeIt;
  typedef ListGraph::Edge Edge;
  typedef xy<int> Xy;
  
  Node n1=g.addNode();
  Node n2=g.addNode();
  Node n3=g.addNode();
  Node n4=g.addNode();
  Node n5=g.addNode();

  ListGraph::NodeMap<Xy> coords(g);
  ListGraph::NodeMap<double> sizes(g);
  ListGraph::NodeMap<int> colors(g);
  ListGraph::EdgeMap<int> ecolors(g);
  ListGraph::EdgeMap<int> widths(g);
  
  coords[n1]=Xy(50,50);  sizes[n1]=1; colors[n1]=1;
  coords[n2]=Xy(50,70);  sizes[n2]=2; colors[n2]=2;
  coords[n3]=Xy(70,70);  sizes[n3]=1; colors[n3]=3;
  coords[n4]=Xy(70,50);  sizes[n4]=2; colors[n4]=4;
  coords[n5]=Xy(85,60);  sizes[n5]=3; colors[n5]=5;
  
  Edge e;

  e=g.addEdge(n1,n2); ecolors[e]=0; widths[e]=1;
  e=g.addEdge(n2,n3); ecolors[e]=0; widths[e]=1;
  e=g.addEdge(n3,n5); ecolors[e]=0; widths[e]=3;
  e=g.addEdge(n5,n4); ecolors[e]=0; widths[e]=1;
  e=g.addEdge(n4,n1); ecolors[e]=0; widths[e]=1;
  e=g.addEdge(n2,n4); ecolors[e]=1; widths[e]=2;
  e=g.addEdge(n3,n4); ecolors[e]=2; widths[e]=1;
  
  IdMap id(g);

  graphToEps(g,"proba.eps").scale(10).coords(coords).
    nodeScale(2).nodeSizes(sizes).
    nodeColors(composeMap(colorSet,colors)).
    edgeColors(composeMap(colorSet,ecolors)).
    edgeWidthScale(.4).edgeWidths(widths).
    nodeTexts(id).nodeTextSize(3);

  graphToEps(g,"proba_arr.eps").scale(10).coords(coords).
    nodeScale(2).nodeSizes(sizes).
    nodeColors(composeMap(colorSet,colors)).
    edgeColors(composeMap(colorSet,ecolors)).
    edgeWidthScale(.4).edgeWidths(widths).
    nodeTexts(id).nodeTextSize(3).
    drawArrows().arrowWidth(1).arrowLength(1);

  e=g.addEdge(n1,n4); ecolors[e]=2; widths[e]=1;
  e=g.addEdge(n4,n1); ecolors[e]=1; widths[e]=2;

  e=g.addEdge(n1,n2); ecolors[e]=1; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=2; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=3; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=4; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=5; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=6; widths[e]=1;
  e=g.addEdge(n1,n2); ecolors[e]=7; widths[e]=1;

  graphToEps(g,"proba_par.eps").scale(10).coords(coords).
    nodeScale(2).nodeSizes(sizes).
    nodeColors(composeMap(colorSet,colors)).
    edgeColors(composeMap(colorSet,ecolors)).
    edgeWidthScale(.4).edgeWidths(widths).
    nodeTexts(id).nodeTextSize(3).
    enableParallel().parEdgeDist(1.5);
}
