#include <iostream>
#include <fstream>
#include <cstring>

#include <lemon/smart_graph.h>
#include <lemon/dimacs.h>
#include <lemon/graph_writer.h>

using namespace std;
using namespace lemon;

const char* versionString =
"dim_to_lgf - part of lemon library\n";

const char* helpString =
"Dimacs to LGF converter\n"
"Usage: dim_to_lgf [OPTIONS]\n"
"\n"
"Examples:\n"
"  dim_to_lgf --type shortestpath --input graph.dim --output graph.lgf\n"
"\n"
"Options:\n"
"  -i FILE, --input FILE    use FILE as input instead of standard input\n"
"  -o FILE, --output FILE   use FILE as output instead of standard output\n"
"  -t TYPE, --type TYPE     set up the type of the graph\n"
"                             Possible types:\n"
"                               mincostflow\n"
"                               maxflow (default)\n"
"                               shortestpath\n"
"                               capacitated\n"
"                               plain\n"
"  -v, --version            shows the version of the converter\n"
"  -h, --help               shows the help of the converter\n";


int main(int argc, const char *argv[]) {
  typedef SmartGraph Graph;

  typedef Graph::Edge Edge;
  typedef Graph::Node Node;
  typedef Graph::EdgeIt EdgeIt;
  typedef Graph::NodeIt NodeIt;
  typedef Graph::EdgeMap<string> StringMap;

  std::string inputName;
  std::string outputName;
  std::string typeName;

  bool help = false;
  bool version = false;

  for (int arg = 1; arg < argc; ++arg) {
    if (strcmp(argv[arg], "--type") == 0 || 
	strcmp(argv[arg], "-t") == 0) {
      if (!typeName.empty()) {
	cerr << "Multiple type description" << endl;
	return -1;
      }
      if (arg + 1 == argc) {
	cerr << "Parameter without value" << endl;
	return -1;
      }
      typeName = argv[++arg];
    }
    else if (strcmp(argv[arg], "--input") == 0 || 
	     strcmp(argv[arg], "-i") == 0) {
      if (!inputName.empty()) {
	cerr << "Multiple input description" << endl;
	return -1;
      }
      if (arg + 1 == argc) {
	cerr << "Parameter without value" << endl;
	return -1;
      }
      inputName = argv[++arg];
    }
    else if (strcmp(argv[arg], "--output") == 0 || 
	     strcmp(argv[arg], "-o") == 0) {
      if (!outputName.empty()) {
	cerr << "Multiple input description" << endl;
	return -1;
      }
      if (arg + 1 == argc) {
	cerr << "Parameter without value" << endl;
	return -1;
      }
      outputName = argv[++arg];
    } else if (strcmp(argv[arg], "--help") == 0 ||
	       strcmp(argv[arg], "-h") == 0) {
      help = true;
    } else if (strcmp(argv[arg], "--version") == 0 ||
	       strcmp(argv[arg], "-v") == 0) {
      version = true;
    } else {
      cerr << "Invalid option: " << argv[arg] << endl;
      return -1;
    }
  }

  if (version) {
    cout << versionString;
  }
  if (help) {
    cout << helpString;
  }
  if (help || version) {
    return 0;
  }

  ifstream input;
  if (!inputName.empty()) {
    input.open(inputName.c_str());
    if (!input) {
      cerr << "File open error" << endl;
      return -1;
    }
  }
  istream& is = (inputName.empty() ? cin : input);

  ofstream output;
  if (!outputName.empty()) {
    output.open(outputName.c_str());
    if (!output) {
      cerr << "File open error" << endl;
      return -1;
    }
  }
  ostream& os = (outputName.empty() ? cout : output);

  if (typeName.empty()) {
    typeName = "maxflow";
  }

  if (typeName == "mincostflow") {
    Graph graph;
    Node s, t;
    StringMap cost(graph), capacity(graph);
    readDimacs(is, graph, capacity, s, t, cost);
    writeGraph(os, graph, capacity, s, t, cost);
  } else if (typeName == "maxflow") {
    Graph graph;
    Node s, t;
    StringMap capacity(graph);
    readDimacs(is, graph, capacity, s, t);
    writeGraph(os, graph, capacity, s, t);
  } else if (typeName == "shortestpath") {
    Graph graph;
    Node s;
    StringMap capacity(graph);
    readDimacs(is, graph, capacity, s);
    writeGraph(os, graph, capacity, s);
  } else if (typeName == "capacitated") {
    Graph graph;
    StringMap capacity(graph);
    readDimacs(is, graph, capacity);
    writeGraph(os, graph, capacity);
  } else if (typeName == "plain") {
    Graph graph;
    readDimacs(is, graph);
    writeGraph(os, graph);
  } else {
    cerr << "Invalid type error" << endl;
    return -1;
  }
  return 0;
}
